/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: DelegatingEFactory.java,v 1.7 2010/03/14 00:56:49 mtaal Exp $
 */

package org.eclipse.emf.texo.xml;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.texo.model.ModelFactory;
import org.eclipse.emf.texo.utils.ModelUtils;

/**
 * Wraps a {@link ModelFactory} to forward the calls to convertToString and createFromString to it.
 * 
 * Other EFactory calls are forwarded to the internal EFactory delegate.
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class DelegatingEFactory implements EFactory {

  private ModelFactory modelFactory;

  private EFactory delegate;

  /**
   * Call is delegated to {@link ModelFactory#convertToString(EDataType, Object)}.
   * 
   * @see org.eclipse.emf.ecore.impl.EFactoryImpl#convertToString(org.eclipse.emf.ecore.EDataType,
   *      java.lang.Object)
   * @see #getXmlHandler()
   */
  public String convertToString(final EDataType dataType, final Object objectValue) {
    if (objectValue instanceof EEnumLiteral) {
      final EEnumLiteral eEnumLiteral = (EEnumLiteral) objectValue;
      return eEnumLiteral.getLiteral();
    }
    return modelFactory.convertToString(dataType, objectValue);
  }

  /**
   * Call is delegated to {@link ModelFactory#createFromString(EDataType, String)}.
   * 
   * Overrides handling of enums to return the EMF Enumerator instance instead of the generated java
   * type.
   * 
   * @see org.eclipse.emf.ecore.impl.EFactoryImpl#createFromString(org.eclipse.emf.ecore.EDataType,
   *      java.lang.String)
   * @see #getXmlHandler()
   */
  public Object createFromString(final EDataType dataType, final String stringValue) {
    if (dataType instanceof EEnum) {
      if (stringValue == null || stringValue.trim().length() == 0) {
        return null;
      }
      return ((EEnum) dataType).getEEnumLiteralByLiteral(stringValue);
    }

    final EDataType baseType = ModelUtils.getEnumBaseDataTypeIfObject(dataType);
    if (baseType != null && baseType instanceof EEnum) {
      if (stringValue == null || stringValue.trim().length() == 0) {
        return null;
      }
      return ((EEnum) baseType).getEEnumLiteralByLiteral(stringValue);
    }

    return modelFactory.createFromString(dataType, stringValue);
  }

  /**
   * @return the modelFactory
   */
  public ModelFactory getModelFactoryy() {
    return modelFactory;
  }

  /**
   * @param modelFactory
   *          the modelFactory to set
   */
  public void setModelFactory(final ModelFactory modelFactory) {
    this.modelFactory = modelFactory;
  }

  /**
   * @return the delegate
   */
  public EFactory getDelegate() {
    return delegate;
  }

  /**
   * @param delegate
   *          the delegate to set
   */
  public void setDelegate(final EFactory delegate) {
    this.delegate = delegate;
  }

  /**
   * @param class1
   * @return
   * @see org.eclipse.emf.ecore.EFactory#create(org.eclipse.emf.ecore.EClass)
   */
  public EObject create(final EClass class1) {
    return delegate.create(class1);
  }

  /**
   * @return
   * @see org.eclipse.emf.common.notify.Notifier#eAdapters()
   */
  public EList<Adapter> eAdapters() {
    return delegate.eAdapters();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eAllContents()
   */
  public TreeIterator<EObject> eAllContents() {
    return delegate.eAllContents();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eClass()
   */
  public EClass eClass() {
    return delegate.eClass();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eContainer()
   */
  public EObject eContainer() {
    return delegate.eContainer();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eContainingFeature()
   */
  public EStructuralFeature eContainingFeature() {
    return delegate.eContainingFeature();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eContainmentFeature()
   */
  public EReference eContainmentFeature() {
    return delegate.eContainmentFeature();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eContents()
   */
  public EList<EObject> eContents() {
    return delegate.eContents();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eCrossReferences()
   */
  public EList<EObject> eCrossReferences() {
    return delegate.eCrossReferences();
  }

  /**
   * @return
   * @see org.eclipse.emf.common.notify.Notifier#eDeliver()
   */
  public boolean eDeliver() {
    return delegate.eDeliver();
  }

  /**
   * @param feature
   * @param resolve
   * @return
   * @see org.eclipse.emf.ecore.EObject#eGet(org.eclipse.emf.ecore.EStructuralFeature, boolean)
   */
  public Object eGet(final EStructuralFeature feature, final boolean resolve) {
    return delegate.eGet(feature, resolve);
  }

  /**
   * @param feature
   * @return
   * @see org.eclipse.emf.ecore.EObject#eGet(org.eclipse.emf.ecore.EStructuralFeature)
   */
  public Object eGet(final EStructuralFeature feature) {
    return delegate.eGet(feature);
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eIsProxy()
   */
  public boolean eIsProxy() {
    return delegate.eIsProxy();
  }

  /**
   * @param feature
   * @return
   * @see org.eclipse.emf.ecore.EObject#eIsSet(org.eclipse.emf.ecore.EStructuralFeature)
   */
  public boolean eIsSet(final EStructuralFeature feature) {
    return delegate.eIsSet(feature);
  }

  /**
   * @param notification
   * @see org.eclipse.emf.common.notify.Notifier#eNotify(org.eclipse.emf.common.notify.Notification)
   */
  public void eNotify(final Notification notification) {
    delegate.eNotify(notification);
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EObject#eResource()
   */
  public Resource eResource() {
    return delegate.eResource();
  }

  /**
   * @param feature
   * @param newValue
   * @see org.eclipse.emf.ecore.EObject#eSet(org.eclipse.emf.ecore.EStructuralFeature,
   *      java.lang.Object)
   */
  public void eSet(final EStructuralFeature feature, final Object newValue) {
    delegate.eSet(feature, newValue);
  }

  /**
   * @param deliver
   * @see org.eclipse.emf.common.notify.Notifier#eSetDeliver(boolean)
   */
  public void eSetDeliver(final boolean deliver) {
    delegate.eSetDeliver(deliver);
  }

  /**
   * @param feature
   * @see org.eclipse.emf.ecore.EObject#eUnset(org.eclipse.emf.ecore.EStructuralFeature)
   */
  public void eUnset(final EStructuralFeature feature) {
    delegate.eUnset(feature);
  }

  /**
   * @param source
   * @return
   * @see org.eclipse.emf.ecore.EModelElement#getEAnnotation(java.lang.String)
   */
  public EAnnotation getEAnnotation(final String source) {
    return delegate.getEAnnotation(source);
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EModelElement#getEAnnotations()
   */
  public EList<EAnnotation> getEAnnotations() {
    return delegate.getEAnnotations();
  }

  /**
   * @return
   * @see org.eclipse.emf.ecore.EFactory#getEPackage()
   */
  public EPackage getEPackage() {
    return delegate.getEPackage();
  }

  /**
   * @param value
   * @see org.eclipse.emf.ecore.EFactory#setEPackage(org.eclipse.emf.ecore.EPackage)
   */
  public void setEPackage(final EPackage value) {
    delegate.setEPackage(value);
  }

  @SuppressWarnings("unused")
  public Object eInvoke(final EOperation operation, final EList<?> arguments)
      throws InvocationTargetException {
    // supported from 2.6
    return null; // delegate.eInvoke(operation, arguments);
  }
}