/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelEMFConverter.java,v 1.10 2010/03/15 10:18:53 mtaal Exp $
 */

package org.eclipse.emf.texo.xml;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.impl.DynamicEObjectImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;
import org.eclipse.emf.ecore.xml.type.internal.XMLCalendar;
import org.eclipse.emf.texo.model.ModelFeatureMapEntry;
import org.eclipse.emf.texo.model.ModelObject;
import org.eclipse.emf.texo.model.ModelResolver;
import org.eclipse.emf.texo.utils.Check;
import org.eclipse.emf.texo.utils.ModelUtils;

/**
 * Converts a set of model objects to a set of DynamicEObjectImpl instances.
 * 
 * Internally a map from {@link ModelObject} to {@link EObject} is maintained so that an object is
 * at most converted once.
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 * @see ModelObject
 * @see DynamicEObjectImpl
 */
public class ModelEMFConverter {

  private Map<Object, InternalEObject> objectMapping = new HashMap<Object, InternalEObject>();

  private List<Object> toConvert = new ArrayList<Object>();

  /**
   * Converts a set of model managed objects and all the objects they reference to a collection of
   * EObjects.
   * 
   * @param objects
   *          the model objects to convert, also the references/children are converted
   * @return the created EObjects
   */
  public List<EObject> convert(final List<Object> objects) {
    // the process creates the new target objects and then converts the content
    // this multi-step process prevents stack overflow with large object graphs
    final List<EObject> result = new ArrayList<EObject>();
    Check.isNotNullArgument(objects, "objects"); //$NON-NLS-1$
    for (final Object object : objects) {
      result.add(createTarget(object));
    }

    while (!toConvert.isEmpty()) {
      final ArrayList<Object> beingConverted = new ArrayList<Object>(toConvert);
      toConvert.clear();
      for (Object object : beingConverted) {
        convertContent(object);
      }
    }

    return result;
  }

  /**
   * Converts a single model managed object.
   * 
   * @param modelObject
   *          the object to convert
   * @return the created EObject
   */
  protected EObject createTarget(final Object target) {
    InternalEObject eObject = objectMapping.get(target);
    if (eObject != null) {
      return eObject;
    }

    // not found, create it and add a new entry to the mapping
    final ModelObject<?> modelObject = ModelResolver.getInstance().getModelObject(target);
    final EClass eClass = modelObject.eClass();
    eObject = (InternalEObject) EcoreUtil.create(eClass);
    objectMapping.put(target, eObject);
    toConvert.add(target);
    return eObject;
  }

  protected void convertContent(Object target) {
    // if a proxy then do no feature conversions as this may load
    // the object
    InternalEObject eObject = objectMapping.get(target);
    final ModelObject<?> modelObject = ModelResolver.getInstance().getModelObject(target);
    final String proxyId = getProxyId(modelObject);
    if (proxyId != null) {
      eObject.eSetProxyURI(URI.createURI(proxyId));
      return;
    }

    for (final EStructuralFeature eStructuralFeature : eObject.eClass().getEAllStructuralFeatures()) {
      if (!eStructuralFeature.isChangeable() || eStructuralFeature.isVolatile()) {
        continue;
      }

      if (FeatureMapUtil.isFeatureMap(eStructuralFeature)) {
        convertFeatureMap(modelObject, eObject, eStructuralFeature);
      } else if (eStructuralFeature.isMany()) {
        if (eStructuralFeature instanceof EAttribute) {
          final EAttribute eAttribute = (EAttribute) eStructuralFeature;
          convertManyEAttribute(modelObject, eObject, eAttribute);
        } else {
          final EReference eReference = (EReference) eStructuralFeature;
          convertManyEReference(modelObject, eObject, eReference);
        }
      } else {
        if (eStructuralFeature instanceof EAttribute) {
          final EAttribute eAttribute = (EAttribute) eStructuralFeature;
          convertSingleEAttribute(modelObject, eObject, eAttribute);
        } else {
          final EReference eReference = (EReference) eStructuralFeature;
          convertSingleEReference(modelObject, eObject, eReference);
        }
      }
    }
  }

  /**
   * If a non-null value is returned then the content of the modelObject is not converted.
   * 
   * The default implementation returns null.
   * 
   * @param modelObject
   *          the modelObject to get the proxy id for
   * @return the proxy id, should encode the type of the object as well as its id
   */
  protected String getProxyId(final Object modelObject) {
    return null;
  }

  /**
   * Converts the values of an FeatureMap, the values of the collection are converted to and added
   * to the list in the correct feature in the modelObject.
   * 
   * @param eObject
   *          the eObject from which the value is read
   * @param modelObject
   *          the {@link ModelObject} in which the value is to be set
   * @param eFeature
   *          the eFeature which is converted
   */
  protected void convertFeatureMap(final ModelObject<?> modelObject, final EObject eObject,
      final EStructuralFeature eFeature) {
    final Collection<?> mValues = (Collection<?>) modelObject.eGet(eFeature);

    @SuppressWarnings("unchecked")
    final Collection<Object> values = (Collection<Object>) eObject.eGet(eFeature);
    for (final Object mValue : mValues) {
      final ModelFeatureMapEntry<?> mEntry = ModelResolver.getInstance().getModelFeatureMapEntry(
          eFeature, mValue);
      final EStructuralFeature entryFeature = mEntry.getEStructuralFeature();
      final Object entryValue = mEntry.getValue();
      final Object convertedValue;
      if (entryFeature instanceof EAttribute) {
        convertedValue = convertEAttributeValue(entryValue, ((EAttribute) entryFeature)
            .getEAttributeType());
      } else {
        convertedValue = createTarget(entryValue);
      }
      final FeatureMap.Entry eEntry = FeatureMapUtil.createEntry(entryFeature, convertedValue);
      values.add(eEntry);
    }
  }

  /**
   * Converts the value of an EReference with isMany==false, the value is converted to an EObject
   * and set in the correct feature in the eObject.
   * 
   * @param modelObject
   *          the modelObject from which the value is retrieved.
   * @param eObject
   *          the eObject in which the value is set (after it has been converted)
   * @param eReference
   *          the eReference which is converted
   */
  protected void convertSingleEReference(final ModelObject<?> modelObject, final EObject eObject,
      final EReference eReference) {
    // containment/container features are always set from the
    // containment side
    if (eReference.isContainer()) {
      return;
    }
    // 2-sided, has already been set from the other side
    if (eReference.getEOpposite() != null) {
      if (eObject.eIsSet(eReference)) {
        return;
      }
      // for bi-directional, the many side always takes care of
      // converting as this does not change the order
      if (eReference.getEOpposite().isMany()) {
        return;
      }
    }

    final Object value = modelObject.eGet(eReference);
    if (value == null) {
      eObject.eSet(eReference, null);
    } else {
      final InternalEObject eValue = (InternalEObject) createTarget(value);
      // if not yet set, set it
      // if set do nothing to prevent bi-directional behavior
      if (eObject.eGet(eReference) == null) {
        eObject.eSet(eReference, eValue);
      }
    }
  }

  /**
   * Converts the value of an EReference with isMany==true, the values of the collection are
   * converted to EObjects and added to the list in the correct feature in the eObject.
   * 
   * @param modelObject
   *          the modelObject from which the value is retrieved.
   * @param eObject
   *          the eObject in which the value is set (after it has been converted)
   * @param eReference
   *          the eReference which is converted
   */
  protected void convertManyEReference(final ModelObject<?> modelObject, final EObject eObject,
      final EReference eReference) {
    // container feature is always set from the other side, the containment
    // side
    if (eReference.isContainer()) {
      return;
    }
    final Object manyValue = modelObject.eGet(eReference);
    if (Map.class.isAssignableFrom(manyValue.getClass())) {
      Check.isTrue(ModelUtils.isEMap(eReference),
          "Expected emap EReference, but this is not the case for EReference " //$NON-NLS-1$
              + eReference);
      @SuppressWarnings("unchecked")
      final Collection<EObject> eValues = (Collection<EObject>) eObject.eGet(eReference);

      final Map<?, ?> map = (Map<?, ?>) manyValue;

      for (final Object key : map.keySet()) {
        final Object value = map.get(key);
        final EObject mapEntryEObject = EcoreUtil.create(eReference.getEReferenceType());
        final EStructuralFeature valueFeature = mapEntryEObject.eClass().getEStructuralFeature(
            "value"); //$NON-NLS-1$
        final EStructuralFeature keyFeature = mapEntryEObject.eClass().getEStructuralFeature("key"); //$NON-NLS-1$

        // key and value maybe primitive types but can also be
        // references to model objects.
        if (valueFeature instanceof EReference) {
          mapEntryEObject.eSet(valueFeature, createTarget(value));
        } else {
          mapEntryEObject.eSet(valueFeature, value);
        }
        if (keyFeature instanceof EReference) {
          mapEntryEObject.eSet(keyFeature, createTarget(key));
        } else {
          mapEntryEObject.eSet(keyFeature, key);
        }
        eValues.add(mapEntryEObject);
      }
    } else {
      @SuppressWarnings("unchecked")
      final Collection<Object> values = (Collection<Object>) manyValue;
      @SuppressWarnings("unchecked")
      final Collection<EObject> eValues = (Collection<EObject>) eObject.eGet(eReference);
      for (final Object value : values) {
        final InternalEObject eValue = (InternalEObject) createTarget(value);
        if (!eValues.contains(eValue)) {
          eValues.add(eValue);
        }
      }
    }
  }

  /**
   * Converts the value of an EAttribute with isMany==false, the value is converted (
   * {@link #convertEAttributeValue(Object, EDataType)}) and set in the correct feature in the
   * eObject.
   * 
   * @param modelObject
   *          the modelObject from which the value is retrieved.
   * @param eObject
   *          the eObject in which the value is set (after it has been converted)
   * @param eAttribute
   *          the EAttribute which is converted
   * @see #convertEAttributeValue(Object, EDataType)
   */
  protected void convertSingleEAttribute(final ModelObject<?> modelObject, final EObject eObject,
      final EAttribute eAttribute) {
    final Object value = modelObject.eGet(eAttribute);
    ((InternalEObject) eObject).eSet(eAttribute, convertEAttributeValue(value, eAttribute
        .getEAttributeType()));
  }

  /**
   * Converts the value of an EAttribute with isMany==true, the values of the collection are
   * converted and added to the list in the correct feature in the eObject.
   * 
   * @param modelObject
   *          the modelObject from which the value is retrieved.
   * @param eObject
   *          the eObject in which the value is set (after it has been converted)
   * @param eAttribute
   *          the EAttribute which is converted
   * @see #convertEAttributeValue(Object, EDataType)
   */
  protected void convertManyEAttribute(final ModelObject<?> modelObject, final EObject eObject,
      final EAttribute eAttribute) {
    final Collection<?> values = (Collection<?>) modelObject.eGet(eAttribute);
    final EDataType eDataType = eAttribute.getEAttributeType();
    @SuppressWarnings("unchecked")
    final List<Object> eValues = (List<Object>) eObject.eGet(eAttribute);
    for (final Object value : values) {
      eValues.add(convertEAttributeValue(value, eDataType));
    }
  }

  /**
   * Converts a primitive type value, this implementation only converts an Enum to an EEnum value.
   * 
   * @param value
   *          the value to convert
   * @param eDataType
   *          its EDataType
   * @return the converted value
   */
  protected Object convertEAttributeValue(final Object value, final EDataType eDataType) {
    if (value instanceof Enum<?>) {
      final EDataType enumDataType = getDataTypeOrBaseType(eDataType);
      Check.isInstanceOf(enumDataType, EEnum.class);
      final EEnum eeNum = (EEnum) enumDataType;
      for (final EEnumLiteral enumLiteral : eeNum.getELiterals()) {
        // the code generation template uppercases enum
        if (enumLiteral.getName().toUpperCase().equals(((Enum<?>) value).name())) {
          return enumLiteral;
        }
      }
    }

    if (value instanceof Date && eDataType == XMLTypePackage.eINSTANCE.getDate()) {
      final Date date = (Date) value;
      final XMLCalendar xmlCalendar = new XMLCalendar(date, XMLCalendar.DATE);
      final Calendar calendar = Calendar.getInstance();
      calendar.setTime(date);
      xmlCalendar.clear();
      xmlCalendar.setYear(calendar.get(Calendar.YEAR));
      xmlCalendar.setMonth(1 + calendar.get(Calendar.MONTH));
      xmlCalendar.setDay(calendar.get(Calendar.DATE));
      // note xmlcalendar expects minutes, calendar gives millis
      xmlCalendar.setTimezone((calendar.get(Calendar.ZONE_OFFSET) / 60000));
      return xmlCalendar;
    }

    if (value instanceof Date && eDataType == XMLTypePackage.eINSTANCE.getDateTime()) {
      final Date date = (Date) value;
      return new XMLCalendar(date, XMLCalendar.DATETIME);
    }

    return value;
  }

  /**
   * See the javadoc in the {@link ModelUtils#getEnumBaseDataTypeIfObject(EDataType)} for details.
   * 
   * @param eDataType
   * @return the passed EDataType or its base type if the base type is an EEnum
   */
  private EDataType getDataTypeOrBaseType(EDataType eDataType) {
    final EDataType baseType = ModelUtils.getEnumBaseDataTypeIfObject(eDataType);
    if (baseType != null) {
      return baseType;
    }
    return eDataType;
  }

}