/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelXMLSaver.java,v 1.7 2010/03/14 14:58:52 mtaal Exp $
 */

package org.eclipse.emf.texo.xml;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.impl.ElementHandlerImpl;

/**
 * Responsible for writing a set of modelObjects to an outputstream or writer.
 * 
 * The ModelXMLSaver makes use of the standard EMF {@link XMLResource} or {@link XMIResource} (if
 * {@link #isSaveAsXMI} is true).
 * 
 * The following options are set as a default (override by calling {@link #setOptions(Map)} with
 * your own options):
 * 
 * XMLResource.OPTION_ENCODING: "UTF-8"
 * 
 * XMLResource.OPTION_EXTENDED_META_DATA: true
 * 
 * XMLResource.OPTION_SCHEMA_LOCATION: true;
 * 
 * XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE: true
 * 
 * XMLResource.OPTION_KEEP_DEFAULT_CONTENT: true
 * 
 * This option settings ensure that the XML corresponds to the XML schema definition.
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class ModelXMLSaver {

  private Writer writer;
  private XMLResource xmlResource;
  private Map<String, Object> options = new HashMap<String, Object>();
  private List<Object> objects;
  private ModelEMFConverter modelEMFConverter = new ModelEMFConverter();
  private boolean saveAsXMI = false;

  /**
   * Writes the model objects ({@link #getModelObjects()}) to the writer ( {@link #getWriter()})
   * using the XML/XMIResource ( {@link #getXmlResource()}).
   */
  @SuppressWarnings("unchecked")
  public void write() {
    try {
      final XMLResource localXMLResource = getXmlResource();
      final List<EObject> eObjects = getModelEMFConverter().convert(getObjects());

      // now do a special method to find all objects without container
      // which are not
      // in the root, they should be added to the root, otherwise they get
      // lost
      addObjectsToRoot(eObjects);

      localXMLResource.getContents().addAll(eObjects);

      // set default options which ensure that XML schemas are followed
      setDefaultOptions(XMLResource.OPTION_ENCODING, "UTF-8"); //$NON-NLS-1$
      setDefaultOptions(XMLResource.OPTION_EXTENDED_META_DATA, true);
      setDefaultOptions(XMLResource.OPTION_SCHEMA_LOCATION, true);
      setDefaultOptions(XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE, true);
      setDefaultOptions(XMLResource.OPTION_KEEP_DEFAULT_CONTENT, true);
      setDefaultOptions(XMLResource.OPTION_ELEMENT_HANDLER, new ElementHandlerImpl(false));

      localXMLResource.save(writer, getOptions());
    } catch (final IOException e) {
      throw new IllegalStateException(e);
    }
  }

  private void addObjectsToRoot(final List<EObject> rootObjects) {
    final HashMap<EObject, EObject> visited = new HashMap<EObject, EObject>();
    for (final EObject eObject : new ArrayList<EObject>(rootObjects)) {
      visit(eObject, visited, rootObjects);
    }
  }

  private void visit(final EObject eObject, final HashMap<EObject, EObject> visited,
      final List<EObject> rootObjects) {
    if (visited.containsKey(eObject)) {
      return;
    }
    visited.put(eObject, eObject);
    if (eObject.eIsProxy()) {
      return;
    }
    if (eObject.eContainer() == null && !rootObjects.contains(eObject)) {
      rootObjects.add(eObject);
    }
    for (final EReference eReference : eObject.eClass().getEAllReferences()) {
      if (eReference.isMany()) {
        @SuppressWarnings("unchecked")
        final List<EObject> list = (List<EObject>) eObject.eGet(eReference);
        for (final EObject refEObject : list) {
          visit(refEObject, visited, rootObjects);
        }
      } else {
        final EObject refEObject = (EObject) eObject.eGet(eReference);
        if (refEObject != null) {
          visit(refEObject, visited, rootObjects);
        }
      }
    }
  }

  // find a DocumentRoot in the EPackage of the eClass or one of its
  // super EClasses
  @SuppressWarnings("unchecked")
  private EObject getDocumentRoot(final EObject eObject, final EClass eClass) {
    final EPackage ePackage = eClass.getEPackage();
    final EClass docRootEClass = ExtendedMetaData.INSTANCE.getDocumentRoot(ePackage);
    if (docRootEClass != null) {
      for (final EReference eRef : docRootEClass.getEAllReferences()) {
        // note the upperbound check on 0 is required and not isMany
        // because the upperbound can be -2 while isMany=false
        if (eRef.getEReferenceType().isInstance(eObject)) {
          final EObject docRoot = EcoreUtil.create(docRootEClass);
          if (eRef.isMany()) {
            ((List<Object>) docRoot.eGet(eRef)).add(eObject);
          } else {
            docRoot.eSet(eRef, eObject);
          }
          return docRoot;
        }
      }
    }
    for (final EClass eSuperClass : eClass.getESuperTypes()) {
      return getDocumentRoot(eObject, eSuperClass);
    }
    return null;
  }

  protected void setDefaultOptions(final String option, final Object value) {
    if (getOptions().get(option) != null) {
      return;
    }
    getOptions().put(option, value);
  }

  public Writer getWriter() {
    return writer;
  }

  public void setWriter(final Writer writer) {
    this.writer = writer;
  }

  /**
   * Returns the {@link XMIResource} or the {@link XMLResource} which is being used. When no xml
   * resource has been set explicitly then one is created. The one created is either a
   * {@link ModelXMIResourceImpl} or a {@link ModelXMLResourceImpl}. This depends on the setting of
   * the saveAsXMI ({@link #isSaveAsXMI()}) option.
   * 
   * @return the resource which is used to convert the model objects to a writer.
   */
  public XMLResource getXmlResource() {
    if (xmlResource == null) {
      if (saveAsXMI) {
        xmlResource = new ModelXMIResourceImpl();
      } else {
        xmlResource = new ModelXMLResourceImpl();
      }
    }
    return xmlResource;
  }

  public void setXmlResource(final XMLResource xmlResource) {
    this.xmlResource = xmlResource;
  }

  public Map<String, Object> getOptions() {
    return options;
  }

  public void setOptions(final Map<String, Object> options) {
    this.options = options;
  }

  public List<Object> getObjects() {
    return objects;
  }

  public void setObjects(final List<Object> objects) {
    this.objects = objects;
  }

  public ModelEMFConverter getModelEMFConverter() {
    return modelEMFConverter;
  }

  public void setModelEMFConverter(final ModelEMFConverter modelEMFConverter) {
    this.modelEMFConverter = modelEMFConverter;
  }

  public boolean isSaveAsXMI() {
    return saveAsXMI;
  }

  public void setSaveAsXMI(final boolean saveAsXMI) {
    this.saveAsXMI = saveAsXMI;
  }

}