/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelController.java,v 1.3 2010/03/07 19:16:06 mtaal Exp $
 */

package org.eclipse.emf.texo.generator;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.texo.annotations.annotationsmodel.ENamedElementAnnotation;

/**
 * Is the main controller for annotating the models with the available annotators. Controls which
 * EPackage instances are annotated, calls annotators in a specific order and manages the single
 * instance of the annotation manager.
 * 
 * An instance of this class is available in the templates as the modelController slot ( (
 * {@link WorkFlowStarter#MODEL_CONTROLLER_CONTEXT_SLOT}).
 * 
 * @see ModelAnnotator
 * @see Annotator
 * @see AnnotationManager
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class ModelController {

  private List<EPackage> ePackages = new ArrayList<EPackage>();

  private List<ModelAnnotator> modelAnnotators = new ArrayList<ModelAnnotator>();

  private AnnotationManager annotationManager = new AnnotationManager();

  /**
   * First calls the annotate method of each model annotator ({@link ModelAnnotator#annotate()}).
   * Then calls the {@link ModelAnnotator#postAnnotate()} method for each model annotator. The model
   * annotators are the ones set in this instance using the
   * {@link #addModelAnnotator(ModelAnnotator)} method.
   */
  public void annotate() {
    // first initialize
    for (ModelAnnotator modelAnnotator : modelAnnotators) {
      modelAnnotator.setEPackages(ePackages);
      modelAnnotator.setAnnotationManager(annotationManager);
      for (Annotator<? extends ENamedElementAnnotation> annotator : modelAnnotator.getAnnotators()) {
        annotationManager.addAnnotator(annotator);
      }
    }

    // then annotate
    for (ModelAnnotator modelAnnotator : modelAnnotators) {
      modelAnnotator.annotate();
    }

    for (ModelAnnotator modelAnnotator : modelAnnotators) {
      modelAnnotator.postAnnotate();
    }
  }

  public List<EPackage> getEPackages() {
    return ePackages;
  }

  public void setEPackages(final List<EPackage> ePackages) {
    this.ePackages = ePackages;
  }

  /**
   * Checks if there is annotation provided by the {@link EPackage} denoted by the prefix.
   * 
   * @param eNamedElement
   *          the model element to check for an annotation.
   * @param nsPrefix
   *          the identifier of the annotation {@link EPackage}
   * @return true if there is an annotation, false otherwise
   * @see #getAnnotation(ENamedElement, String)
   */
  public boolean isAnnotated(final ENamedElement eNamedElement, final String nsPrefix) {
    return null != getAnnotation(eNamedElement, nsPrefix);
  }

  /**
   * Convenience method for a template. Searches for the {@link ModelAnnotation} instances for the
   * passed {@link ENamedElement} and returns the first one belonging to an {@link EPackage} with
   * the specified name space prefix. The first annotation found is returned.
   * 
   * Returns null if no annotation can be found.
   * 
   * @param eNamedElement
   *          the named element to search annotations.
   * @param nsURI
   *          the name space uri used to select the epackage and filter the correct
   *          {@link EModelElementAnnotation}
   * @return the found annotation or null if none is found.
   */
  public ENamedElementAnnotation getAnnotation(final ENamedElement eNamedElement, final String nsURI) {
    final List<ENamedElementAnnotation> annotations = getAnnotationManager().getAnnotations(
        eNamedElement);
    for (final ENamedElementAnnotation annotation : annotations) {
      if (annotation.getENamedElement() == eNamedElement
          && annotation.eClass().getEPackage().getNsURI().equals(nsURI)) {
        return annotation;
      }
    }
    return null;
  }

  /**
   * Method is used to add java annotations to a generated class.
   * 
   * @param eNamedElement
   *          the element for which to create java annotations.
   * @param identifier
   *          identifies how the java annotations is used, examples of values: type, field, method
   * @return java annotations to add
   */
  public String getJavaAnnotations(final ENamedElement eNamedElement, final String identifier) {
    final List<ENamedElementAnnotation> annotations = getAnnotationManager().getAnnotations(
        eNamedElement);
    final StringBuilder sb = new StringBuilder();
    for (final ENamedElementAnnotation annotation : annotations) {
      if (annotation.getENamedElement() == eNamedElement) {
        final String javaAnnotation = annotation.getJavaAnnotation(identifier);
        if (javaAnnotation != null) {
          if (sb.length() > 0) {
            sb.append("\n"); //$NON-NLS-1$
          }
          sb.append(javaAnnotation);
        }
      }
    }

    return sb.toString();
  }

  public void addModelAnnotator(ModelAnnotator modelAnnotator) {
    modelAnnotator.setAnnotationManager(getAnnotationManager());
    modelAnnotators.add(modelAnnotator);
  }

  public void setModelAnnotators(List<ModelAnnotator> modelAnnotators) {
    this.modelAnnotators = modelAnnotators;
  }

  public void setInitialAnnotations(List<ENamedElementAnnotation> initialAnnotations) {
    getAnnotationManager().setInitialAnnotations(initialAnnotations);
  }

  public AnnotationManager getAnnotationManager() {
    return annotationManager;
  }

  public void setAnnotationManager(AnnotationManager annotationManager) {
    this.annotationManager = annotationManager;
  }
}
