/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *   OpenArchitectureWare - some lines copied from org.eclipse.xtend.expression.ResourceManagerDefaultImpl
 *
 * </copyright>
 *
 * $Id: TexoResourceManager.java,v 1.1 2010/03/07 20:25:35 mtaal Exp $
 */

package org.eclipse.emf.texo.generator;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.Reader;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.emf.texo.utils.Check;
import org.eclipse.internal.xtend.expression.parser.SyntaxConstants;
import org.eclipse.xtend.expression.Resource;
import org.eclipse.xtend.expression.ResourceManagerDefaultImpl;
import org.eclipse.xtend.expression.ResourceParser;

/**
 * Special XPand resource manager that reads templates from a project if present in the project.
 * 
 * Note a few lines have been copied from org.eclipse.xtend.expression.ResourceManagerDefaultImpl
 * 
 * @author mtaal
 */
public class TexoResourceManager extends ResourceManagerDefaultImpl {

  public static final QualifiedName TEMPLATE_FOLDER_PROPERTY = new QualifiedName(
      "org.eclipse.emf.texo.eclipse", "TEMPLATE_FOLDER"); //$NON-NLS-1$ //$NON-NLS-2$

  private final Map<String, Resource> resources = new HashMap<String, Resource>();

  private File templateFolder;

  public void setProjectName(String projectName) {
    try {
      final IProject project = EclipseGeneratorUtils.getProject(projectName);
      final String projectFilePath = EclipseGeneratorUtils.getProjectFilePath(projectName);
      final File projectDirectory = new File(projectFilePath);
      final String templateFolderPath = project.getPersistentProperty(TEMPLATE_FOLDER_PROPERTY);
      if (templateFolderPath != null) {
        templateFolder = new File(projectDirectory, templateFolderPath);
        if (!templateFolder.exists()) {
          templateFolder = null;
        }
      } else {
        templateFolder = null;
      }
    } catch (CoreException e) {
      throw new IllegalStateException(e);
    }
  }

  @Override
  public Resource loadResource(String fullyQualifiedName, String extension) {
    final String resourceName = fullyQualifiedName.replace(SyntaxConstants.NS_DELIM, "/") + "." //$NON-NLS-1$//$NON-NLS-2$
        + extension;
    if (resources.containsKey(resourceName)) {
      return resources.get(resourceName);
    }
    File resourceFile = null;
    final boolean useDefaultResourceManager;
    if (templateFolder != null) {
      resourceFile = new File(templateFolder, resourceName);
      useDefaultResourceManager = !resourceFile.exists();
    } else {
      useDefaultResourceManager = true;
    }

    if (useDefaultResourceManager) {
      final Resource resource = super.loadResource(fullyQualifiedName, extension);
      resources.put(resourceName, resource);
      return resource;
    }
    Check.isNotNull(resourceFile, "Resourcefile may not be null at this point"); //$NON-NLS-1$
    try {
      final InputStream in = new FileInputStream(resourceFile);

      final Reader reader = createReader(in);
      final ResourceParser parser = registeredParsers.get(extension);
      if (parser == null) {
        throw new RuntimeException("No Parser registered for extension '" + extension //$NON-NLS-1$
            + "'! Known extensions are '" + registeredParsers.keySet() + "'"); //$NON-NLS-1$//$NON-NLS-2$
      }
      final Resource res = parser.parse(reader, resourceName);
      res.setFullyQualifiedName(fullyQualifiedName);
      resources.put(resourceName, res);
      return res;
    } catch (Exception e) {
      throw new IllegalStateException("Exception loading resource " + resourceName, e); //$NON-NLS-1$
    }

  }
}
