/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelEAttributeAnnotator.java,v 1.11 2010/03/14 00:57:33 mtaal Exp $
 */

package org.eclipse.emf.texo.modelgenerator.annotator;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.emf.texo.generator.Annotator;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EAttributeModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EDataTypeModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.ModelcodegeneratorPackage;
import org.eclipse.emf.texo.utils.Check;

/**
 * Responsible for setting the values in a {@link EAttributeModelGenAnnotation}.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.11 $
 */

public class ModelEAttributeAnnotator extends ModelEStructuralFeatureAnnotator implements
    Annotator<EAttributeModelGenAnnotation> {

  /*
   * (non-Javadoc)
   * 
   * @seeorg.eclipse.emf.texo.generator.Annotator#postAnnotating(org.eclipse.emf.texo.annotations.
   * annotationsmodel.ENamedElementAnnotation)
   */
  public void postAnnotating(EAttributeModelGenAnnotation annotation) {
    super.postAnnotating(annotation);
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.texo.generator.Annotator#getAnnotationEClass()
   */
  @Override
  public EClass getAnnotationEClass() {
    return ModelcodegeneratorPackage.eINSTANCE.getEAttributeModelGenAnnotation();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.generator.Annotator#annotate(org.eclipse.emf.texo.annotations.annotationsmodel
   * .ENamedElementAnnotation)
   */
  public void setAnnotationFeatures(EAttributeModelGenAnnotation annotation) {

    Check.isNotNull(annotation.getEAttribute(), "eAttribute must be set");//$NON-NLS-1$

    super.annotate(annotation);

    final EAttribute eAttribute = annotation.getEAttribute();
    if (!annotation.isSetBoolean()) {
      annotation.setBoolean(isBooleanType(eAttribute));
    }
    if (!annotation.isSetPrimitive()) {
      annotation.setPrimitive(isPrimitiveType(eAttribute));
    }
    if (!annotation.isSetReference()) {
      annotation.setReference(false);
    }
  }

  private boolean isBooleanType(EStructuralFeature eFeature) {
    // note == comparison is possible
    return isPrimitiveType(eFeature)
        && (getInstanceClass(eFeature) == java.lang.Boolean.TYPE || getInstanceClass(eFeature) == java.lang.Boolean.class);
  }

  private boolean isPrimitiveType(EStructuralFeature eFeature) {
    if (eFeature.isMany() || getInstanceClass(eFeature) == null) {
      return false;
    }
    return getInstanceClass(eFeature).isPrimitive();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.modelgenerator.annotator.ModelEStructuralFeatureAnnotator#getGetter(org
   * .eclipse.emf.ecore.EStructuralFeature)
   */
  @Override
  protected String getGetter(EStructuralFeature eFeature) {
    if (isBooleanType(eFeature)) {
      return "is" + GenUtils.upCaseFirst(getName(eFeature));//$NON-NLS-1$
    }
    return super.getGetter(eFeature);
  }

  private boolean isManyInstanceClass(EStructuralFeature eFeature) {
    final Class<?> localInstanceClass = getInstanceClass(eFeature);
    if (localInstanceClass == null) {
      return false;
    }
    return localInstanceClass.isArray() || Collection.class.isAssignableFrom(localInstanceClass);
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.modelgenerator.annotator.ModelEStructuralFeatureAnnotator#getType(org.
   * eclipse.emf.ecore.EStructuralFeature)
   */
  @Override
  protected String getType(EStructuralFeature eFeature) {
    if (eFeature.isMany()) {
      if (eFeature.isOrdered()) {
        return List.class.getName() + "<" + getItemType(eFeature) //$NON-NLS-1$
            + ">"; //$NON-NLS-1$
      }
      return Set.class.getName() + "<" + getItemType(eFeature) + //$NON-NLS-1$ 
          ">"; //$NON-NLS-1$
    } else if (isManyInstanceClass(eFeature)) {
      // support two constructors: array, set and list
      final Class<?> localInstanceClass = getInstanceClass(eFeature);
      if (localInstanceClass == null) {
        return getInstanceClassName(eFeature);
      }
      if (localInstanceClass.isArray()) {
        return localInstanceClass.getComponentType().getName() + "[]"; //$NON-NLS-1$
      }
      // a collection...
      final EDataType eDataType = ((EAttribute) eFeature).getEAttributeType();
      final EDataTypeModelGenAnnotation annotation = getEDataTypeModelGenAnnotation(eDataType);
      return annotation.getInstanceClassName() + "<" + annotation.getItemType() + ">"; //$NON-NLS-1$ //$NON-NLS-2$
    }
    return getInstanceClassName(eFeature);
  }

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.texo.annotator.modelgenerator.
   * ModelEStructuralFeatureAnnotator#getObjectType()
   */
  @Override
  protected String getObjectType(EStructuralFeature eStructuralFeature) {
    if (getType(eStructuralFeature).contains(GenConstants.DOT)) {
      return getType(eStructuralFeature);
    }
    final EDataType eDataType = ((EAttribute) eStructuralFeature).getEAttributeType();
    final EDataTypeModelGenAnnotation annotation = getEDataTypeModelGenAnnotation(eDataType);
    return annotation.getObjectClassName();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.modelgenerator.annotator.ModelEStructuralFeatureAnnotator#getItemType(
   * org.eclipse.emf.ecore.EStructuralFeature)
   */
  @Override
  protected String getItemType(EStructuralFeature eFeature) {
    if (FeatureMapUtil.isFeatureMap(eFeature)) {
      return getFeatureMapQualifiedClassName(eFeature);
    }
    final EAttribute eAttribute = (EAttribute) eFeature;
    final EDataType eDataType = eAttribute.getEAttributeType();
    final EDataTypeModelGenAnnotation annotation = getEDataTypeModelGenAnnotation(eDataType);
    return annotation.getObjectClassName();
  }

  /** @return true if efeature isMany or {@link #isManyInstanceClass()} is true */
  @Override
  protected boolean isMany(EStructuralFeature eFeature) {
    return eFeature.isMany() || isManyInstanceClass(eFeature);
  }

  /** Returns a default value for the feature */
  @Override
  protected String getDefaultValue(EStructuralFeature eFeature) {
    if (eFeature.isMany()) {
      if (eFeature.isOrdered()) {
        return "new " + ArrayList.class.getName() + //$NON-NLS-1$ 
            "<" //$NON-NLS-1$
            + getItemType(eFeature) + ">()"; //$NON-NLS-1$
      }
      return "new " + HashSet.class.getName() + //$NON-NLS-1$ 
          "<" //$NON-NLS-1$
          + getItemType(eFeature) + ">()"; //$NON-NLS-1$
    }
    try {
      return GenUtils.getStaticDefaultValue(getAnnotationManager(), ((EAttribute) eFeature)
          .getEAttributeType(), eFeature.getDefaultValueLiteral());
    } catch (final Throwable t) {
      t.printStackTrace(System.err);
      throw new Error(t);
    }
  }

  /**
   * @return the instance class of the EDataType. Note can return NULL, because the instance class
   *         of the EDataType maybe null or the instance class is not in the classpath.
   */
  private Class<?> getInstanceClass(EStructuralFeature eFeature) {
    // invariant, see remark in getInstanceClassName()
    Check.isTrue(!FeatureMapUtil.isFeatureMap(eFeature) && !eFeature.isMany(),
        "The EStructuralFeature " //$NON-NLS-1$
            + eFeature + " is a featuremap, this method should not be called"); //$NON-NLS-1$
    final EDataType eDataType = ((EAttribute) eFeature).getEAttributeType();
    final EDataTypeModelGenAnnotation annotation = getEDataTypeModelGenAnnotation(eDataType);
    return GenUtils.getClassForName(eDataType, annotation.getInstanceClassName());
  }
}