/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands), IBM Corporation and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *   IBM - default value handling
 *
 * </copyright>
 *
 * $Id: ModelEDataTypeAnnotator.java,v 1.14 2010/03/14 14:58:56 mtaal Exp $
 */

package org.eclipse.emf.texo.modelgenerator.annotator;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.texo.generator.Annotator;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EDataTypeModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EPackageModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.ModelcodegeneratorPackage;
import org.eclipse.emf.texo.utils.Check;
import org.eclipse.emf.texo.utils.ModelUtils;

/**
 * Responsible for setting the values in a {@link EDataTypeModelGenAnnotation}.
 * 
 * Note: the code related to default value handling originates to a large extent from the
 * GenDataTypeImpl class which is part of EMF core.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.14 $
 */

public class ModelEDataTypeAnnotator extends ModelEClassifierAnnotator implements
    Annotator<EDataTypeModelGenAnnotation> {

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.texo.generator.Annotator#getAnnotationEClass()
   */
  @Override
  public EClass getAnnotationEClass() {
    return ModelcodegeneratorPackage.eINSTANCE.getEDataTypeModelGenAnnotation();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.generator.Annotator#annotate(org.eclipse.emf.texo.annotations.annotationsmodel
   * .ENamedElementAnnotation)
   */
  public void setAnnotationFeatures(EDataTypeModelGenAnnotation annotation) {
    super.annotate(annotation);
    Check.isNotNull(annotation.getEDataType(), "eDataType must be set");//$NON-NLS-1$

    final EDataType eDataType = annotation.getEDataType();
    if (!annotation.isSetEnum()) {
      annotation.setEnum(eDataType instanceof EEnum);
    }

    final EEnum eEnumBaseType = ModelUtils.getEnumBaseDataTypeIfObject(eDataType);
    final boolean isTypeWithEnumBaseType = eEnumBaseType != null;

    final String instanceClassName;
    if (annotation.getInstanceClassName() != null) {
      instanceClassName = annotation.getInstanceClassName();
    } else if (isTypeWithEnumBaseType) {
      instanceClassName = getInstanceClassNameForEnum(eEnumBaseType);
    } else {
      instanceClassName = getInstanceClassName(eDataType);
    }

    final Class<?> instanceClass = getInstanceClass(eDataType, instanceClassName);

    if (!annotation.isSetDateType()) {
      if (isTypeWithEnumBaseType) {
        annotation.setDateType(false);
      } else if (GenUtils.isDateOrDateTime(eDataType)) {
        annotation.setDateType(true);
      } else {
        annotation.setDateType(false);
      }
    }

    // find the item type, if not found directly try to find it in the basetype
    EDataType itemDataType = ExtendedMetaData.INSTANCE.getItemType(annotation.getEDataType());
    EDataType baseType = annotation.getEDataType();
    while (itemDataType == null
        && (baseType = ExtendedMetaData.INSTANCE.getBaseType(baseType)) != null) {
      itemDataType = ExtendedMetaData.INSTANCE.getItemType(baseType);
    }
    if (itemDataType != null) {
      final EDataTypeModelGenAnnotation itemDataTypeAnnotation = getEDataTypeModelGenAnnotation(itemDataType);
      annotation.setItemType(itemDataTypeAnnotation.getInstanceClassName());
    } else {
      annotation.setItemType(Object.class.getName());
    }

    if (annotation.getObjectClassName() == null) {
      if (isTypeWithEnumBaseType) {
        annotation.setObjectClassName(instanceClassName);
      } else if (instanceClass != null && instanceClass.isArray()) {
        annotation.setObjectClassName(instanceClass.getComponentType() + "[]"); //$NON-NLS-1$
      } else if (instanceClass != null && instanceClass.getTypeParameters().length > 0
          && annotation.getItemType() != null) {
        // currently only one type parameter is supported
        annotation.setObjectClassName(instanceClass.getName() + "<" + annotation.getItemType() //$NON-NLS-1$
            + ">"); //$NON-NLS-1$
      } else {
        annotation.setObjectClassName(GenUtils.getObjectClassName(eDataType, instanceClassName));
      }
    }

    if (annotation.getQualifiedClassName() == null) {
      annotation.setQualifiedClassName(instanceClassName);
    }
    if (annotation.getInstanceClassName() == null) {
      annotation.setInstanceClassName(instanceClassName);
    }
    if (annotation.getSimpleClassName() == null) {
      annotation.setSimpleClassName(getSimpleClassName(eDataType));
    }
    if (!annotation.isSetGenerateCode()) {
      annotation.setGenerateCode(true);
    }
    if (!annotation.isSetAutomaticStringConversion()) {
      if (isTypeWithEnumBaseType) {
        annotation.setAutomaticStringConversion(false);
      } else {
        annotation.setAutomaticStringConversion(hasInstanceClassStringConstructor(instanceClass));
      }
    }
    final EPackageModelGenAnnotation ePackageAnnotation = getEPackageModelGenAnnotation(eDataType
        .getEPackage());
    annotation.setOwnerEPackageAnnotation(ePackageAnnotation);
    if (isTypeWithEnumBaseType) {
      final EDataTypeModelGenAnnotation baseTypeAnnotation = getEDataTypeModelGenAnnotation(eEnumBaseType);
      annotation.setBaseTypeAnnotation(baseTypeAnnotation);
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @seeorg.eclipse.emf.texo.generator.Annotator#postAnnotating(org.eclipse.emf.texo.annotations.
   * annotationsmodel.ENamedElementAnnotation)
   */
  public void postAnnotating(EDataTypeModelGenAnnotation annotation) {
  }

  protected Class<?> getInstanceClass(EDataType eDataType, String instanceClassName) {
    if (GenUtils.isObjectTypeWithEnumBaseType(eDataType)) {

    }
    return GenUtils.getClassForName(eDataType, instanceClassName);
  }

  protected String getInstanceClassName(EDataType eDataType) {
    return GenUtils.getInstanceClassName(eDataType);
  }

  /**
   * @return true if the instance class is not null (see {@link #getInstanceClass()} and it has a
   *         constructor with a single String constructor.
   */
  private boolean hasInstanceClassStringConstructor(Class<?> instanceClass) {
    if (instanceClass == null) {
      return false;
    }
    try {
      instanceClass.getConstructor(String.class);
      return true;
    } catch (final NoSuchMethodException e) {
      return false;
    }
  }

  protected String getInstanceClassNameForEnum(EEnum eEnum) {
    final EPackageModelGenAnnotation annotation = getEPackageModelGenAnnotation(eEnum.getEPackage());
    final String qualifiedClassName = annotation.getPackagePath().toLowerCase() + GenConstants.DOT
        + getSimpleClassName(eEnum);
    return qualifiedClassName;
  }
}