/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelEPackageAnnotator.java,v 1.7 2010/02/14 22:14:00 mtaal Exp $
 */

package org.eclipse.emf.texo.modelgenerator.annotator;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.emf.ecore.xml.type.XMLTypePackage;
import org.eclipse.emf.texo.generator.Annotator;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EClassModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EDataTypeModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EEnumModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.EPackageModelGenAnnotation;
import org.eclipse.emf.texo.modelgenerator.modelannotations.ModelcodegeneratorPackage;
import org.eclipse.emf.texo.utils.Check;

/**
 * Responsible for setting the values in a {@link EPackageModelGenAnnotation}.
 * 
 * @author <a href="mailto:mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.7 $
 */

public class ModelEPackageAnnotator extends ModelENamedElementAnnotator implements
    Annotator<EPackageModelGenAnnotation> {

  /*
   * (non-Javadoc)
   * 
   * @see org.eclipse.emf.texo.generator.Annotator#getAnnotationEClass()
   */
  @Override
  public EClass getAnnotationEClass() {
    return ModelcodegeneratorPackage.eINSTANCE.getEPackageModelGenAnnotation();
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * org.eclipse.emf.texo.generator.Annotator#annotate(org.eclipse.emf.texo.annotations.annotationsmodel
   * .ENamedElementAnnotation)
   */
  public void setAnnotationFeatures(EPackageModelGenAnnotation annotation) {

    super.annotate(annotation);

    Check.isNotNull(annotation.getEPackage(), "ePackage must be set");//$NON-NLS-1$

    final EPackage ePackage = annotation.getEPackage();

    final String packagePath;
    if (annotation.getPackagePath() == null) {
      packagePath = GenUtils.createJavaPackagePath(ePackage.getNsURI());
      annotation.setPackagePath(packagePath);
    } else {
      packagePath = annotation.getPackagePath();
    }
    if (annotation.getQualifiedClassName() == null) {
      annotation.setQualifiedClassName(packagePath + GenConstants.DOT
          + getSimpleClassName(ePackage) + annotation.getModelPackageClassNamePostFix());
    }

    if (annotation.getSimpleClassName() == null) {
      annotation.setSimpleClassName(getSimpleClassName(ePackage)
          + annotation.getModelPackageClassNamePostFix());
    }
    if (annotation.getSimpleModelFactoryClassName() == null) {
      annotation.setSimpleModelFactoryClassName(getSimpleClassName(ePackage)
          + annotation.getModelFactoryClassNamePostFix());
    }
    if (annotation.getEcoreFileName() == null) {
      annotation.setEcoreFileName(ePackage.getName() + ".ecore"); //$NON-NLS-1$
    }
    final List<EPackageModelGenAnnotation> dependsOn = getDependsOn(ePackage);
    if (annotation.getDependsOn().isEmpty()) {
      annotation.getDependsOn().addAll(dependsOn);
    }
    if (annotation.getEcoreFileContent() == null) {
      annotation.setEcoreFileContent(getEcoreFileContent(ePackage, dependsOn));
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @seeorg.eclipse.emf.texo.generator.Annotator#postAnnotating(org.eclipse.emf.texo.annotations.
   * annotationsmodel.ENamedElementAnnotation)
   */
  public void postAnnotating(EPackageModelGenAnnotation annotation) {
    final EPackage ePackage = annotation.getEPackage();

    if (annotation.getEClassifierAnnotations().isEmpty()) {
      for (EClassifier eClassifier : ePackage.getEClassifiers()) {
        if (eClassifier instanceof EClass) {
          final EClassModelGenAnnotation eClassifierAnnotation = (EClassModelGenAnnotation) getAnnotationManager()
              .getAnnotation(eClassifier,
                  ModelcodegeneratorPackage.eINSTANCE.getEClassModelGenAnnotation());
          annotation.getEClassifierAnnotations().add(eClassifierAnnotation);
        } else if (eClassifier instanceof EEnum) {
          final EEnumModelGenAnnotation eClassifierAnnotation = (EEnumModelGenAnnotation) getAnnotationManager()
              .getAnnotation(eClassifier,
                  ModelcodegeneratorPackage.eINSTANCE.getEEnumModelGenAnnotation());
          annotation.getEClassifierAnnotations().add(eClassifierAnnotation);
        } else if (eClassifier instanceof EDataType) {
          final EDataTypeModelGenAnnotation eClassifierAnnotation = (EDataTypeModelGenAnnotation) getAnnotationManager()
              .getAnnotation(eClassifier,
                  ModelcodegeneratorPackage.eINSTANCE.getEDataTypeModelGenAnnotation());
          annotation.getEClassifierAnnotations().add(eClassifierAnnotation);
        } else {
          throw new IllegalStateException("Not supported here " + eClassifier); //$NON-NLS-1$
        }
      }
    }
  }

  private String getSimpleClassName(EPackage ePackage) {
    return GenUtils.upCaseFirst(getName(ePackage));
  }

  /**
   * Compute the depends on and returns it. The depends on packages are all the packages which are
   * refered from this package.
   * 
   * @return the list of {@link ModelEPackageAnnotator} on which this one depends.
   */
  private List<EPackageModelGenAnnotation> getDependsOn(EPackage ePackage) {
    final Set<EPackage> epacks = new HashSet<EPackage>();
    // get the epackage of the supertypes of each eclass
    for (final EClassifier eClassifier : ePackage.getEClassifiers()) {
      if (eClassifier instanceof EClass) {
        final EClass eClass = (EClass) eClassifier;
        for (final EClass superEClass : eClass.getESuperTypes()) {
          if (superEClass.getEPackage() != ePackage && !epacks.contains(superEClass.getEPackage())) {
            Check.isNotNull(superEClass.getEPackage(), "The epackage of the eclass " //$NON-NLS-1$
                + superEClass + " is not set, it is the superEClass of " //$NON-NLS-1$
                + eClass + ". This indicates that the super EClass is " //$NON-NLS-1$
                + "not loaded correctly, for example the file with " //$NON-NLS-1$
                + "the EPackage can not be found"); //$NON-NLS-1$
            epacks.add(superEClass.getEPackage());
          }
        }

        // now handle the efeatures
        for (final EReference eref : eClass.getEAllReferences()) {
          final EPackage refEPackage = eref.getEReferenceType().getEPackage();
          if (refEPackage != ePackage && !epacks.contains(refEPackage)) {
            epacks.add(refEPackage);
          }
        }
        for (final EAttribute eattr : eClass.getEAllAttributes()) {
          final EPackage refEPackage = eattr.getEType().getEPackage();
          if (refEPackage != ePackage && !epacks.contains(refEPackage)) {
            epacks.add(refEPackage);
          }
        }
      }
    }

    // now the set of dependent epackages is found, get the CodeGenEPackage
    final List<EPackageModelGenAnnotation> dependsOn = new ArrayList<EPackageModelGenAnnotation>();
    for (final EPackage depEPackage : epacks) {
      if (depEPackage == EcorePackage.eINSTANCE || depEPackage == XMLTypePackage.eINSTANCE) {
        continue;
      }
      dependsOn.add(getEPackageModelGenAnnotation(depEPackage));
    }
    return dependsOn;
  }

  public String getEcoreFileContent(EPackage ePackage, List<EPackageModelGenAnnotation> dependsOn) {

    // final URI uri = URI.createURI(ePackage.getNsURI());
    final HashMap<Resource, URI> previousURI = new HashMap<Resource, URI>();
    try {
      // store the epackage in a class readable file
      // in every package java impl package so that they can be
      // loaded. to ensure that references between packages use nsuri's
      // first the
      // uri of all other resources (containing the other epackages) has
      // to be
      // set to the nsuri of the other packages
      boolean dependsOnEcorePackage = false;
      boolean dependsOnXMLTypePackage = false;
      for (final EPackageModelGenAnnotation dependsOnAnnotation : dependsOn) {
        if (dependsOnAnnotation.getEPackage() == ePackage) { // skip this one
          continue;
        }
        final EPackage localEPackage = dependsOnAnnotation.getEPackage();

        dependsOnEcorePackage |= localEPackage == EcorePackage.eINSTANCE;
        dependsOnXMLTypePackage |= localEPackage == XMLTypePackage.eINSTANCE;
        final Resource res = setEPackageResource(localEPackage);
        previousURI.put(res, res.getURI());
        res.setURI(URI.createURI(localEPackage.getNsURI()));
      }

      if (!dependsOnEcorePackage) {
        final Resource res = setEPackageResource(EcorePackage.eINSTANCE);
        previousURI.put(res, res.getURI());
      }
      if (!dependsOnXMLTypePackage) {
        final Resource res = setEPackageResource(XMLTypePackage.eINSTANCE);
        previousURI.put(res, res.getURI());
      }

      final EcoreResourceFactoryImpl ecoreResourceFactory = new EcoreResourceFactoryImpl();
      // note the uri of the resource is the same as the one used to read
      // it again in
      // ModelUtils.readEPackagesFromFile
      final XMIResource outputRes = (XMIResource) ecoreResourceFactory.createResource(URI
          .createURI(GenConstants.EMPTY));
      outputRes.getContents().add(ePackage);
      final StringWriter sw = new StringWriter();
      outputRes.save(sw, Collections.EMPTY_MAP);
      return sw.toString();
      // set the uris back
    } catch (final IOException e) {
      throw new IllegalStateException(e);
    } finally {
      for (final Resource res : previousURI.keySet()) {
        res.setURI(previousURI.get(res));
      }
    }
  }

  private Resource setEPackageResource(EPackage ePackage) {
    Resource res = ePackage.eResource();
    if (res == null) {
      res = new XMIResourceImpl();
      res.getContents().add(ePackage);
    }
    res.setURI(URI.createURI(ePackage.getNsURI()));
    return res;
  }

  @Override
  public String getName(ENamedElement namedElement) {
    final String name = super.getName(namedElement);
    if (name.indexOf(GenConstants.DOT) != -1) {
      return name.substring(1 + name.lastIndexOf(GenConstants.DOT));
    }
    return name;
  }
}