/**
 * <copyright>
 *
 * Copyright (c) 2009, 2010 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: ModelXMLLoader.java,v 1.6 2010/03/14 14:58:51 mtaal Exp $
 */

package org.eclipse.emf.texo.xml;

import java.io.IOException;
import java.io.Reader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.xml.sax.InputSource;

/**
 * Responsible for reading a set of model managed objects from an inputstream or reader.
 * 
 * The ModelXMLLoader makes use of the standard EMF {@link XMLResource} or {@link XMIResource} (if
 * {@link #isLoadAsXMI()} is true).
 * 
 * The following options are set as a default (override by calling {@link #setOptions(Map)} with
 * your own options):
 * 
 * XMLResource.OPTION_ENCODING: "UTF-8"
 * 
 * XMLResource.OPTION_EXTENDED_META_DATA: true
 * 
 * XMLResource.OPTION_SCHEMA_LOCATION: true;
 * 
 * XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE: true
 * 
 * XMLResource.OPTION_USE_LEXICAL_HANDLER: true
 * 
 * This option setting ensure that the XML corresponds to the XML schema definition.
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 */
public class ModelXMLLoader {

  private Reader reader;
  private XMLResource xmlResource;
  private Map<String, Object> options = new HashMap<String, Object>();
  private List<Object> modelObjects;
  private EMFModelConverter emfModelConverter = new EMFModelConverter();
  private boolean loadAsXMI = false;

  /**
   * Execute the read from the inputstream and return the list of objects.
   * 
   * @return the list of read objects, in case of XML/XMI the list contains the objects in the root
   *         of the XML/XMI.
   */
  public List<Object> read() {
    try {
      final XMLResource localXMLResource = getXmlResource();

      setDefaultOptions(XMLResource.OPTION_ENCODING, "UTF-8"); //$NON-NLS-1$
      setDefaultOptions(XMLResource.OPTION_EXTENDED_META_DATA, true);
      setDefaultOptions(XMLResource.OPTION_SUPPRESS_DOCUMENT_ROOT, true);
      setDefaultOptions(XMLResource.OPTION_SCHEMA_LOCATION, true);
      setDefaultOptions(XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE, true);
      setDefaultOptions(XMLResource.OPTION_USE_LEXICAL_HANDLER, true);

      localXMLResource.load(new InputSource(getReader()), options);
      return getEmfModelConverter().convert(localXMLResource.getContents());
    } catch (final IOException e) {
      throw new IllegalStateException(e);
    }
  }

  protected void setDefaultOptions(final String option, final Object value) {
    if (options.get(option) != null) {
      return;
    }
    options.put(option, value);
  }

  /**
   * Returns the {@link XMIResource} or the {@link XMLResource} which is being used. When no xml
   * resource has been set explicitly then one is created. The one created is either a
   * {@link ModelXMIResourceImpl} or a {@link ModelXMLResourceImpl}. This depends on the setting of
   * the saveAsXMI ({@link #isLoadAsXMI()}) option.
   * 
   * @return the resource which is used to convert the model objects to a writer.
   */

  public XMLResource getXmlResource() {
    if (xmlResource == null) {
      if (loadAsXMI) {
        xmlResource = new ModelXMIResourceImpl();
      } else {
        xmlResource = new ModelXMLResourceImpl();
      }
    }
    return xmlResource;
  }

  public void setXmlResource(final XMLResource xmlResource) {
    this.xmlResource = xmlResource;
  }

  public Map<String, Object> getOptions() {
    return options;
  }

  public void setOptions(final Map<String, Object> options) {
    this.options = options;
  }

  public List<Object> getModelObjects() {
    return modelObjects;
  }

  public void setModelObjects(final List<Object> modelObjects) {
    this.modelObjects = modelObjects;
  }

  public Reader getReader() {
    return reader;
  }

  public void setReader(final Reader reader) {
    this.reader = reader;
  }

  public EMFModelConverter getEmfModelConverter() {
    return emfModelConverter;
  }

  public void setEmfModelConverter(final EMFModelConverter emfModelConverter) {
    this.emfModelConverter = emfModelConverter;
  }

  public boolean isLoadAsXMI() {
    return loadAsXMI;
  }

  public void setLoadAsXMI(final boolean loadAsXMI) {
    this.loadAsXMI = loadAsXMI;
  }

}