/**
 * <copyright>
 *
 * Copyright (c) 2011 Springsite BV (The Netherlands) and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Martin Taal - Initial API and implementation
 * </copyright>
 *
 * $Id: BaseDao.java,v 1.3 2011/09/24 04:56:36 mtaal Exp $
 */
package org.eclipse.emf.texo.server.store;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Root;
import javax.persistence.metamodel.EntityType;
import javax.persistence.metamodel.Metamodel;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.texo.component.TexoComponent;
import org.eclipse.emf.texo.model.ModelConstants;
import org.eclipse.emf.texo.model.ModelObject;
import org.eclipse.emf.texo.model.ModelResolver;
import org.eclipse.emf.texo.utils.ModelUtils;

/**
 * The BaseDao implements the base data access operations for a certain type generated by Texo.
 * 
 * The BaseDao can be assigned an EntityManager, if this is not done then it will automatically retrieve an
 * EntityManager from the {@link EntityManagerProvider} using the {@link EntityManagerProvider#getEntityManager()}
 * method.
 * 
 * @see DaoRegistry
 * 
 * @author <a href="mtaal@elver.org">Martin Taal</a>
 * @version $Revision: 1.3 $
 */
public abstract class BaseDao<T extends Object> implements TexoComponent {

  @PersistenceContext
  private EntityManager entityManager;

  private String entityName;

  /**
   * Return the class handled by this Dao, the T in the class definition.
   */
  public abstract Class<T> getEntityClass();

  /**
   * The {@link EClass} managed by this Dao.
   */
  public abstract EClass getEClass();

  /**
   * Returns the entity manager used by this Dao. If this Dao does not have an EntityManager set then one is retrieved
   * through {@link EntityManagerProvider#getEntityManager()}.
   */
  public EntityManager getEntityManager() {
    if (entityManager == null) {
      entityManager = EntityManagerProvider.getInstance().getEntityManager();
    }
    return entityManager;
  }

  public void setEntityManager(EntityManager entityManager) {
    this.entityManager = entityManager;
  }

  /**
   * Get a specific T using the id.
   */
  public T get(Object id) {
    if (id == null) {
      return null;
    }
    return getEntityManager().find(getEntityClass(), id);
  }

  /**
   * Returns all the T objects.
   */
  public List<T> getAll() {
    final CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    final CriteriaQuery<T> criteria = builder.createQuery(getEntityClass());
    final Root<T> root = criteria.from(getEntityClass());
    criteria.select(root);
    return getEntityManager().createQuery(criteria).getResultList();
  }

  /**
   * Count all T objects
   */
  public int countAll() {
    final CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    final CriteriaQuery<Long> criteria = builder.createQuery(Long.class);
    final Root<T> root = criteria.from(getEntityClass());
    criteria.select(builder.count(root));
    return getEntityManager().createQuery(criteria).getSingleResult().intValue();
  }

  /**
   * Return a list of T based on a certain property and its value.
   */
  public List<T> findBy(String property, Object value) {
    final CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    final CriteriaQuery<T> criteria = builder.createQuery(getEntityClass());
    final Root<T> root = criteria.from(getEntityClass());
    criteria.select(root);
    criteria.where(builder.equal(root.get(property), value));
    return getEntityManager().createQuery(criteria).getResultList();
  }

  /**
   * Return a list of T based on a certain {@link EStructuralFeature} and its value.
   */
  public List<T> findBy(EStructuralFeature eFeature, Object value) {
    return findBy(getJavaPropertyName(eFeature), value);
  }

  /**
   * Return a list of T based on a property-value map, the prop-values are combined in the where clause using and.
   */
  public List<T> findByProperties(Map<String, Object> filterBy) {
    final CriteriaBuilder builder = getEntityManager().getCriteriaBuilder();
    final CriteriaQuery<T> criteria = builder.createQuery(getEntityClass());
    final Root<T> root = criteria.from(getEntityClass());
    criteria.select(root);
    Expression<Boolean> rootExpression = null;
    for (String property : filterBy.keySet()) {
      Object value = filterBy.get(property);
      if (rootExpression == null) {
        rootExpression = builder.equal(root.get(property), value);
      } else {
        rootExpression = builder.and(rootExpression, builder.equal(root.get(property), value));
      }
    }
    criteria.where(rootExpression);
    return getEntityManager().createQuery(criteria).getResultList();
  }

  /**
   * Return a list of T based on a feature-value map, the feature-values are combined in the where clause using and.
   */
  public List<T> findByFeatures(Map<EStructuralFeature, Object> filterBy) {
    final Map<String, Object> filterByProp = new HashMap<String, Object>();
    for (EStructuralFeature eFeature : filterBy.keySet()) {
      final String prop = getJavaPropertyName(eFeature);
      filterByProp.put(prop, filterBy.get(eFeature));
    }
    return findByProperties(filterByProp);
  }

  /**
   * Refreshes t from the backend db
   */
  public void refresh(T t) {
    getEntityManager().refresh(t);
  }

  /**
   * Removes t, first removes the dependent objects referenced by t.
   */
  public void remove(T t) {
    deleteCascade(new ArrayList<Object>(), t);
  }

  /**
   * Calls {@link #remove(Object)} for each t in ts
   */
  public void remove(List<T> ts) {
    for (T t : ts) {
      remove(t);
    }
  }

  /**
   * Calls {@link EntityManager#merge(Object)}.
   */
  public T update(T t) {
    return getEntityManager().merge(t);
  }

  /**
   * Calls {@link EntityManager#persist(Object)}.
   */
  public void insert(T t) {
    getEntityManager().persist(t);
  }

  /**
   * Returns true if t is referenced by other objects. If the second parameter (includeContainerReferences is true then
   * also container associations (to the target from its children) are considered.
   * 
   * @see ModelResolver#getReferingEReferences(EClass, boolean)
   */
  public boolean isReferenced(T t, boolean includeContainerReferences) {
    final EClass localEClass = getEClass();
    final List<EReference> eRefs = ModelResolver.getInstance().getReferingEReferences(localEClass,
        includeContainerReferences);
    for (EReference eReference : eRefs) {
      if (eReference.isContainment() && !includeContainerReferences) {
        continue;
      }

      final EClass referencingEClass = eReference.getEContainingClass();
      final Class<?> referencingClass = ModelResolver.getInstance().getImplementationClass(referencingEClass);
      final BaseDao<?> referencingDao = DaoRegistry.getInstance().getDaoForEntity(referencingClass);
      final String refEntityName = referencingDao.getEntityName();
      final String propertyName = referencingDao.getJavaPropertyName(eReference);

      final String qryStr;
      if (eReference.isMany()) {
        qryStr = "select count(e) from " + refEntityName + " e where :target member of e." + propertyName; //$NON-NLS-1$ //$NON-NLS-2$
      } else {
        qryStr = "select count(e) from " + refEntityName + " e where e." + propertyName + "=:target"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      }
      final Query qry = getEntityManager().createQuery(qryStr);

      qry.setParameter("target", t); //$NON-NLS-1$
      final Number result = (Number) qry.getSingleResult();
      if (result.intValue() > 0) {
        return true;
      }
    }
    return false;
  }

  /**
   * Returns the objects referencing t, if includeContainerReferences is true then also container references (so from
   * children to the target) are considered for this retrieval. The maxResult parameter can be used to limit the number
   * of objects returned, a value of 0 (or less) will return all referees.
   */
  public List<Object> getReferingObjects(T t, int maxResult, boolean includeContainerReferences) {
    final EClass localEClass = getEClass();
    final List<EReference> eRefs = ModelResolver.getInstance().getReferingEReferences(localEClass,
        includeContainerReferences);
    final List<Object> result = new ArrayList<Object>();
    for (EReference eReference : eRefs) {
      if (eReference.isContainer() && !includeContainerReferences) {
        continue;
      }
      final EClass referencingEClass = eReference.getEContainingClass();
      final Class<?> referencingClass = ModelResolver.getInstance().getImplementationClass(referencingEClass);
      final BaseDao<?> referencingDao = DaoRegistry.getInstance().getDaoForEntity(referencingClass);
      referencingDao.setEntityManager(getEntityManager());
      final String refEntityName = referencingDao.getEntityName();
      final String propertyName = referencingDao.getJavaPropertyName(eReference);
      final String qryStr;
      if (eReference.isMany()) {
        qryStr = "select e from " + refEntityName + " e where :target member of e." + propertyName; //$NON-NLS-1$ //$NON-NLS-2$
      } else {
        qryStr = "select e from " + refEntityName + " e where e." + propertyName + "=:target"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
      }
      final Query qry = getEntityManager().createQuery(qryStr);
      qry.setParameter("target", t); //$NON-NLS-1$
      if (maxResult > 0) {
        qry.setMaxResults(maxResult);
      }
      final List<?> resultList = qry.getResultList();
      // get rid of the duplicates
      result.removeAll(resultList);
      result.addAll(resultList);
      if (maxResult > 0 && result.size() >= maxResult) {
        if (result.size() > maxResult) {
          return result.subList(0, maxResult);
        }
        return result;
      }
    }
    return result;
  }

  /**
   * Returns the entity name of the type managed by this Dao.
   * 
   * @see Metamodel#entity(Class)
   * @see EntityManager#getMetamodel()
   */
  public String getEntityName() {
    if (entityName == null) {
      final Metamodel metaModel = getEntityManager().getMetamodel();
      final EntityType<?> entityType = metaModel.entity(getEntityClass());
      entityName = entityType.getName();
    }
    return entityName;
  }

  protected String getJavaPropertyName(EStructuralFeature eFeature) {
    // TODO: one day generate this in a big if statement in the generated
    // subclass
    final String propertyName = ModelUtils.getEAnnotation(eFeature, ModelConstants.JAVAMEMBER_ANNOTATION_KEY);
    if (propertyName == null) {
      return eFeature.getName();
    }
    return propertyName;
  }

  protected void deleteCascade(List<Object> deleteSet, Object object) {
    if (deleteSet.contains(object)) {
      return;
    }
    deleteSet.add(object);

    final ModelObject<?> modelObject = ModelResolver.getInstance().getModelObject(object);
    for (EReference eReference : modelObject.eClass().getEAllReferences()) {
      if (eReference.isContainment() && !eReference.isMany()) {
        final Object value = modelObject.eGet(eReference);
        if (value != null) {
          // first nullify the current value
          modelObject.eSet(eReference, null);

          // and remove it now
          final BaseDao<?> dao = DaoRegistry.getInstance().getDaoForEntity(value.getClass());
          dao.setEntityManager(getEntityManager());
          dao.deleteCascade(deleteSet, value);
        }
      }
    }

    // finally remove the object itself
    getEntityManager().remove(object);
  }
}
