/*******************************************************************************
 * Copyright (c) 2007 Wind River Systems, Inc. and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * 
 * Contributors: 
 * Michael Scharf (Wind River) - initial API and implementation
 *******************************************************************************/
package org.eclipse.tm.internal.terminal.textcanvas;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tm.terminal.model.ITerminalTextDataReadOnly;
import org.eclipse.tm.terminal.model.LineSegment;
import org.eclipse.tm.terminal.model.Style;

/**
 *
 */
public class TextLineRenderer implements ILinelRenderer {
	TextCanvas fCanvas;
	private final ITextCanvasModel fModel;
	StyleMap fStyleMap=new StyleMap();
	Color fBackgroundColor;
	public TextLineRenderer(TextCanvas c, ITextCanvasModel model) {
		fCanvas=c;
		fModel=model;
		fBackgroundColor=c.getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
	}
	/* (non-Javadoc)
	 * @see com.imagicus.thumbs.view.ICellRenderer#getCellWidth()
	 */
	public int getCellWidth() {
		return fStyleMap.getFontWidth();
	}
	/* (non-Javadoc)
	 * @see com.imagicus.thumbs.view.ICellRenderer#getCellHeight()
	 */
	public int getCellHeight() {
		return fStyleMap.getFontHeight();
	}
	public void drawLine(ITextCanvasModel model, GC gc, int line, int x, int y, int colFirst, int colLast) {
		if(line<0 || line>=getTerminalText().getHeight() || colFirst>=getTerminalText().getWidth() || colFirst-colLast==0) {
			fillBackground(gc, x, y, getCellWidth()*(colFirst-colLast), getCellHeight());
		} else {
			colLast=Math.min(colLast, getTerminalText().getWidth());
			LineSegment[] segments=getTerminalText().getLineSegments(line, colFirst, colLast-colFirst);
			for (int i = 0; i < segments.length; i++) {
				LineSegment segment=segments[i];
				Style style=segment.getStyle();
				setupGC(gc, style);
				String text=segment.getText();
				drawText(gc, x, y, colFirst, segment.getColumn(), text);
				drawCursor(model, gc, line, x, y, colFirst);
			}
			if(fModel.hasLineSelection(line)) {
				gc.setForeground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_SELECTION_TEXT));
				gc.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_LIST_SELECTION));
				Point start=model.getSelectionStart();
				Point end=model.getSelectionEnd();
				char[] chars=model.getTerminalText().getChars(line);
				if(chars==null)
					return;
				int offset=0;
				if(start.y==line)
					offset=start.x;
				offset=Math.max(offset, colFirst);
				int len;
				if(end.y==line)
					len=end.x-offset+1;
				else
					len=chars.length-offset+1;
				len=Math.min(len,chars.length-offset);
				if(len>0) {
					String text=new String(chars,offset,len);
					drawText(gc, x, y, colFirst, offset, text);
				}
			}
		}
	}
	
	private void fillBackground(GC gc, int x, int y, int width, int height) {
		Color bg=gc.getBackground();
		gc.setBackground(getBackgroundColor());
		gc.fillRectangle (x,y,width,height);
		gc.setBackground(bg);
		
	}

	private Color getBackgroundColor() {
		return fBackgroundColor;
	}
	private void drawCursor(ITextCanvasModel model, GC gc, int row, int x, int y, int colFirst) {
		if(!model.isCursorOn())
			return;
		int cursorLine=model.getCursorLine();
			
		if(row==cursorLine) {
			int cursorColumn=model.getCursorColumn();
			if(cursorColumn<getTerminalText().getWidth()) {
				Style style=getTerminalText().getStyle(row, cursorColumn);
				if(style==null) {
					// TODO make the cursor color customizable
					style=Style.getStyle("BLACK", "WHITE");  //$NON-NLS-1$//$NON-NLS-2$
				}
				style=style.setReverse(!style.isReverse());
				setupGC(gc,style);
				String text=String.valueOf(getTerminalText().getChar(row, cursorColumn));
				drawText(gc, x, y, colFirst, cursorColumn, text);
			}
		}
	}
	private void drawText(GC gc, int x, int y, int colFirst, int col, String text) {
		int offset=(col-colFirst)*getCellWidth();
		text=text.replace('\000', ' ');
		gc.drawString(text,x+offset,y,false);
	}
	private void setupGC(GC gc, Style style) {
		Color c=fStyleMap.getForegrondColor(style);
		if(c!=gc.getForeground()) {
			gc.setForeground(c);
		}
		c=fStyleMap.getBackgroundColor(style);
		if(c!=gc.getBackground()) {
			gc.setBackground(c);
		}
		Font f=fStyleMap.getFont(style);
		if(f!=gc.getFont()) {
			gc.setFont(f);
		}
	}
	ITerminalTextDataReadOnly getTerminalText() {
		return fModel.getTerminalText();
	}
}
