/**********************************************************************
Copyright (c) 2002 IBM Corporation and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html
Contributors:
Adrian Colyer, Andy Clement, Tracy Gardner - initial version
...
**********************************************************************/
package org.eclipse.ajdt.internal.core;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;

/**
 * @author Andy Clement
 * This implementation of the resource change interfaces is particular interested in
 * knowing when new resources have been added to a project.  For any set of added
 * resources, it will popup a wizard to query which .lst files (build configurations)
 * should be updated to include the new resources.  Notice the wizard is started
 * asynchronously since whilst processing the 'resource changed' event, the resource
 * tree is locked.
 */
public class AspectJResourceChangeListener
	implements IResourceChangeListener, IResourceDeltaVisitor {

	
    /**
     * Many different kinds of resource might be added to the project, this list
     * should contain only .java, .aj and .lst files - these are the three kinds
     * of resource that might be added to a build configuration
     */
	List additionCandidates = null;


	/**
	 * The entry point when a resource change occurs.  
	 * 
	 * @see IResourceChangeListener#resourceChanged(IResourceChangeEvent)
	 */
	public void resourceChanged(IResourceChangeEvent rcEvent) {

		additionCandidates = new ArrayList();

		IResource res = rcEvent.getResource();
		
/* TEMPORARILY DISABLED AS THE DIALOG THAT POPS UP IS UNFRIENDLY, IF A RESOURCE
 * IS CHANGED IN THE PROJECT, THE .LST FILES WILL NEED MANUALLY UPDATING
 * THROUGH THE BUILD CONFIG EDITOR PANEL.
		switch (rcEvent.getType()) {
			case IResourceChangeEvent.POST_CHANGE :

				try {
					rcEvent.getDelta().accept(this);
				} catch (CoreException ce) { //ASCFIXME - Use correct error reporting.
					AspectJPlugin.getDefault().getErrorHandler().handleError("Unexpected exception whilst processing resource change",ce);
				}

				break;
		}
		
				
		// If multiple resource changes occur, they can be across multiple projects (ASCFIXME: Can they
		// reallllly????), so pop up one wizard per project.  
		// Basically this is implemented by picking the resource first
		// on the list, determining its project and pulling out all the other candidate resources for
		// that project - a wizard is then popped up for this project.  Then the remainder of the
		// candidate list is examined, to see if there is another project that requires a wizard.
		while (additionCandidates.size() > 0) {
			
			// Check all the candidates are in the same project, just assert it for now...  
			IProject project = ((IResource) additionCandidates.get(0)).getProject();
			
			
			// Check if there are any config files to update!
			AspectJPlugin.getDefault().setCurrentProject(project);
			List buildConfigFiles =
			AspectJPlugin.getDefault().getListOfConfigFilesForCurrentProject();
			if (buildConfigFiles.size()==0) break; 
			
			List candidatesFromOneProject = new ArrayList();
			candidatesFromOneProject.add((IResource)additionCandidates.get(0));
			
			for (int i = additionCandidates.size()-1;i>0;i--) {
				IResource ir = (IResource)additionCandidates.get(i);
				IProject projectIsNow = ir.getProject();
				if (projectIsNow.equals(project)) {
					candidatesFromOneProject.add(ir);
					additionCandidates.remove(i);
				}
			}
			
			additionCandidates.remove(0);
			
			// Need to finalize this for use in the run() method below
			final List finalizedCandidates = (ArrayList) ((ArrayList) candidatesFromOneProject).clone();
			
			// Run the wizard asyncrhonously so the resource tree can unlock
			org.eclipse.swt.widgets.Display.getDefault().asyncExec(new Runnable() {
				public void run() {

					try {
						// Pop up the wizard to determine which lst files should be updated.
						ResourceAddedWizard wizard = new ResourceAddedWizard();
						wizard.init(AspectJPlugin.getDefault().getWorkbench(), null);
						wizard.setNewResources(finalizedCandidates);
						WizardDialog wdialog =
							new WizardDialog(
								AspectJPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow().getShell(),
								wizard);
						wdialog.open();
					} catch (Throwable t) {
						t.printStackTrace();
					}
				}
			});

		}
		*/

	}
	
	
	/**
	 * Check if the resource supplied is of the type that could be inserted in a .lst file.  For this
	 * to be true, it must be a .aj, a .java or a .lst file.
	 */
	private boolean isCompilationUnitOrBuildFile(IResource res) {
		String path = res.getFullPath().toString();
		if (path.endsWith(".java") || path.endsWith(".aj") || path.endsWith(".lst"))
			return true;
		else
			return false;
	}


	/**
	 * For each change to a resource, determine if its something interesting. :
	 * The first case of interest is the addition of a resource, since that is when a popup wizard
	 * is required to determine which .lst files to alter.
	 */
	public boolean visit(IResourceDelta rdelta) {
		IResource res = rdelta.getResource();
		if (isCompilationUnitOrBuildFile(res)) {
			switch (rdelta.getKind()) {
				case IResourceDelta.ADDED :
					additionCandidates.add(res);
					break;
				case IResourceDelta.REMOVED :
					break;
				case IResourceDelta.CHANGED :
					break;
			}
		}
		return true; // visit the children
	}

}