/**********************************************************************
Copyright (c) 2000, 2002 IBM Corp. and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html

Contributors:
    IBM Corporation - Initial implementation
    Andy Clement, 1st Version, 7th October 2002
**********************************************************************/
package org.eclipse.ajdt.internal.ui.editor;

import org.eclipse.ajdt.internal.core.resources.AspectJImages;
import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.source.IVerticalRulerInfo;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.texteditor.AbstractRulerActionDelegate;
import org.eclipse.ui.texteditor.IDocumentProvider;
import org.eclipse.ui.texteditor.ITextEditor;

public class AdviceActionDelegate extends AbstractRulerActionDelegate {


	IEditorPart editor;
	IVerticalRulerInfo rulerInfo;

	public AdviceActionDelegate() {
		editor    = null;
		rulerInfo = null;
	}



	/**
	 * @see IEditorActionDelegate#setActiveEditor(bIAction, IEditorPart)
	 */
	public void setActiveEditor(IAction callerAction,IEditorPart targetEditor) {
		// We only care about compilation unit and class file editors
		if (targetEditor != null) {
			String id = targetEditor.getSite().getId();

			if (!id.equals(JavaUI.ID_CU_EDITOR) && !id.equals(JavaUI.ID_CF_EDITOR)
				&& !id.equals("CompilationUnitEditor")) // The AspectJ editor
				targetEditor = null;
		}
		editor = targetEditor; // Remember the editor
		super.setActiveEditor(callerAction, targetEditor);
	}



	/**
	 * @see AbstractRulerActionDelegate#createAction()
	 */
	protected IAction createAction(ITextEditor editor,IVerticalRulerInfo rulerInfo) {
		this.rulerInfo = rulerInfo;
		return null;
	}



    /**
     * Called to see if this action delegate wants to influence the menu before it
     * is displayed - in the case of AJDT we have to check if there is an advice
     * marker in affect on the line in which the user has right clicked.  If there
     * is then we add an 'Advised By' line to the context submenu that
     * will appear.  By going through the submenu and selecting advice, we force
     * the editor to jump to a particular file and location - selecting the
     * advice that is in effect.
     */
	public void menuAboutToShow(IMenuManager manager) {

		try {

            // Retrieve some information that allows us to query
            // the advice markers currently in place.
			IEditorSite ies = editor.getEditorSite();
			
			// Work out which file is currently being edited
			IFileEditorInput ifep =	(IFileEditorInput) this.editor.getEditorInput();
			IFile ifile = ifep.getFile();
			
			IDocumentProvider idp = ((ITextEditor) editor).getDocumentProvider();
			IDocument id = idp.getDocument(ifep);
			
			// Which line was right clicked in the ruler?
			int linenumber = rulerInfo.getLineOfLastMouseButtonActivity();
			
			// Which markers are in effect on the whole file
			IMarker markers[] = ifile.findMarkers("org.eclipse.ajdt.ui.advicemarker", true, 2);
			IMarker decMarkers[] = ifile.findMarkers("org.eclipse.ajdt.ui.declarationmarker", true, 2);
			
			// The submenu is built the first time we discover some advice is affecting this
			// line.
			MenuManager adviceSubmenu         = null;
		    boolean adviceSubmenuInitialized = false;
			MenuManager declarationSubmenu         = null;
		    boolean declarationSubmenuInitialized = false;
		    
			// Go through the advice markers attached to this file, if any have a line number that
			// matches the line clicked by the user, create the submenu (if it has not already
			// been created) and add a new submenu entry for the advice.  The data that is
			// stored with the submenu entry gives the run() method of the inner class the
			// ability to create a jump marker such that it can jump to the location where
			// the advice is defined.
			if (markers != null && markers.length != 0) {
				
				Integer clickedLine = new Integer(linenumber+1);
				
				for (int j = 0; j < markers.length; j++) {
					IMarker m = markers[j];
					
					if (m.getAttribute(IMarker.LINE_NUMBER).equals(clickedLine)) {
						String textLabel = ((String)m.getAttribute(IMarker.MESSAGE));//.substring(8);
						// substring(8) skips the 'Advice: ' bit on the front.
						
						// Build a new action for our menu.  Set the text label and remember the
						// marker (an advice marker) in effect on this line, so that if the run
						// method of the action is driven, it can correctly jump to the right
						// location in the aspect.
						AJDTMenuAction ama = new AJDTMenuAction(textLabel,m);
						
						// Initialize the submenu if we haven't done it already.
						if (!adviceSubmenuInitialized) {
							adviceSubmenu = new MenuManager(
							  AspectJPlugin.getResourceString("EditorRulerContextMenu.adviceInAffect"));
							manager.add(new Separator());
							manager.add(adviceSubmenu);			
							adviceSubmenuInitialized = true; 
						}
							
					    // Add our new action to the submenu
						adviceSubmenu.add(ama);
					}
				}
			}

			// Go through the ITD markers attached to this file, if any have a line number that
			// matches the line clicked by the user, create the submenu (if it has not already
			// been created) and add a new submenu entry for the advice.  The data that is
			// stored with the submenu entry gives the run() method of the inner class the
			// ability to create a jump marker such that it can jump to the location where
			// the advice is defined.
			if (decMarkers != null && decMarkers.length != 0) {
				
				Integer clickedLine = new Integer(linenumber+1);
				
				for (int j = 0; j < decMarkers.length; j++) {
					IMarker m = decMarkers[j];
					
					if (m.getAttribute(IMarker.LINE_NUMBER).equals(clickedLine)) {
						String textLabel = ((String)m.getAttribute(IMarker.MESSAGE));//.substring(8);
						// substring(8) skips the 'Advice: ' bit on the front.
						
						// Build a new action for our menu.  Set the text label and remember the
						// marker (an advice marker) in effect on this line, so that if the run
						// method of the action is driven, it can correctly jump to the right
						// location in the aspect.
						AJDTMenuAction ama = new AJDTMenuAction(textLabel,m);
						
						// Initialize the submenu if we haven't done it already.
						if (!declarationSubmenuInitialized) {
							declarationSubmenu = new MenuManager(
							  AspectJPlugin.getResourceString("EditorRulerContextMenu.aspectDeclarations"));
							if(!adviceSubmenuInitialized) {
								manager.add(new Separator());
							}
							manager.add(declarationSubmenu);			
							declarationSubmenuInitialized = true; 
						}
							
					    // Add our new action to the submenu
						declarationSubmenu.add(ama);
					}
				}
			}
		} catch (CoreException ce) {
		  AspectJPlugin.getDefault().getErrorHandler().
            handleError("Exception whilst extending ruler context menu with advice items", ce);
		}
	}
	
	
	
	/**
	 * Inner class that represents an entry on the submenu for "Advised by >" 
	 * or "Aspect Declarations >"
	 * - each AJDTMenuAction is a piece of advice or an ITD in affect on the current line.
	 * When each AJDTMenuAction is created, it is given a name (the advice in affect)
	 * and a marker.  This is the advice marker attached to the line.  Both advice markers
	 * and ITD markers are like normal markers but have an extra attribute: sourceLocationOfAdvice
	 * This attribute has the format FFFF:::NNNN:::NNNN:::NNNN
	 * - The FFFF is the file which contains the source of the advice or ITD in affect
	 * - The other three NNNN fields are integers indicating (in order) the
	 *   start line number of the advice in that file, the end line number of the
	 *   advice in that file and the column number for the advice.
	 * 
	 * I had to code it this way because you can't set arbitrary object values for
	 * attributes.  Using the value of this attribute, the run() method for the
	 * action can create a jump marker that points to the real advice definition
	 * and jump to it.
	 */
	class AJDTMenuAction extends Action {
		
		// Advice marker, the sourceLocationOfAdvice attribute is the only
		// attribute used in the run() method.
		IMarker adviceMarker;
		
		/**
		 * Set the name, remember the marker and set the image !
		 */
		AJDTMenuAction(String s,IMarker marker) {
			super(s);
			String runtimeTest = AspectJPlugin.getResourceString("AspectJEditor.runtimetest");
			boolean hasRuntimeTest = false;
			if(s.endsWith(runtimeTest)) {
				hasRuntimeTest = true;
			}
			if(s.indexOf(".before(") != -1) {
				if(hasRuntimeTest) {
					setImageDescriptor(AspectJImages.DYNAMIC_BEFORE_ADVICE.getImageDescriptor());
				} else {
					setImageDescriptor(AspectJImages.BEFORE_ADVICE.getImageDescriptor());
				}
			} else if (s.indexOf(".after(")!=-1 || s.indexOf(".afterReturning(")!=-1 || s.indexOf(".afterThrowing(")!=-1) {
				if(hasRuntimeTest) {
					setImageDescriptor(AspectJImages.DYNAMIC_AFTER_ADVICE.getImageDescriptor());
				} else {
					setImageDescriptor(AspectJImages.AFTER_ADVICE.getImageDescriptor());
				}				
			} else if (s.indexOf(".around(") != -1) {
				if(hasRuntimeTest) {
					setImageDescriptor(AspectJImages.DYNAMIC_AROUND_ADVICE.getImageDescriptor());
				} else {
					setImageDescriptor(AspectJImages.AROUND_ADVICE.getImageDescriptor());
				}
			} else {
				setImageDescriptor(AspectJImages.ITD.getImageDescriptor());
			}
			adviceMarker = marker;
		}


        /**
         * Take the sourceLocationOfAdvice attribute and build a marker that
         * can be passed to openEditor() to force Eclipse to open the right
         * source file and jump to the right location.
         */
		public void run() {

			// Fetch the real advice marker from the marker that is attached to affected sites.
			try {
				
				String jumpLocation =
					(String) adviceMarker.getAttribute("sourceLocationOfAdvice");
					
				// Take it apart.  It is initially:  FFFF:::NNNN:::NNNN:::NNNN

				String filepath = jumpLocation.substring(0, jumpLocation.indexOf(":::"));

				jumpLocation    = jumpLocation.substring(jumpLocation.indexOf(":::") + 3);
				final String linenumber = jumpLocation.substring(0, jumpLocation.indexOf(":::"));

				jumpLocation = jumpLocation.substring(jumpLocation.indexOf(":::") + 3);
				final String endlinenumber = jumpLocation.substring(0, jumpLocation.indexOf(":::"));

				final String columnnumber = jumpLocation.substring(jumpLocation.indexOf(":::") + 3);

                /*
				System.err.println("FilePath=" + filepath);
				System.err.println("linenum=" + linenumber);
				System.err.println("endlinenum=" + endlinenumber);
				System.err.println("columnnumber = " + columnnumber);
				*/
				final IResource ir = AspectJPlugin.getDefault().getAjdtProjectProperties()
						.findResource(filepath);
			    
				IMarker jumpMarker = null;
				
				if (ir != null) {
					try {
						jumpMarker = ir.createMarker(IMarker.TEXT);
						/* GOTCHA:
						 * If setting LINE_NUMBER for a marker, you *have* to call the version of setAttribute that
						 * takes an int and not the version that takes a string (even if your line number is in a 
						 * string) - it won't give you an error but will *not* be interpreted correctly.
						 */
						jumpMarker.setAttribute(
							IMarker.LINE_NUMBER,
							new Integer(linenumber).intValue());

					} catch (CoreException ce) {
					  AspectJPlugin.getDefault().getErrorHandler().
          				  handleError("Unable to build jump marker in AdviceMenuAction run method", ce);
					}
				

					try {
						IDE.openEditor(AspectJPlugin.getDefault().getActiveWorkbenchWindow().getActivePage(),jumpMarker,true);
				    } catch (Exception e) {
					    AspectJPlugin.getDefault().getErrorHandler().
          				    handleError("Exception whilst asking editor to jump to advice location", e);

				    } 
				}
			} catch (IndexOutOfBoundsException ioobe) {
			  AspectJPlugin.getDefault().getErrorHandler().
          		  handleError("Problem parsing sourceLocationOfAdvice attribute value (probably!)", ioobe);
			} catch (CoreException ce) {
				AspectJPlugin.getDefault().getErrorHandler().
          		  handleError("Exception whilst executing AdviceMenuAction run method", ce);
			}
		}
	}

}
