/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     Luzius Meisser - initial implementation
 *******************************************************************************/

package org.eclipse.ajdt.buildconfigurator;

import java.util.ArrayList;

import org.eclipse.ajdt.internal.core.resources.AspectJImages;
import org.eclipse.ajdt.internal.ui.ajde.ProjectProperties;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.internal.ui.JavaPlugin;
import org.eclipse.jdt.internal.ui.JavaPluginImages;
import org.eclipse.jdt.internal.ui.viewsupport.ImageDescriptorRegistry;
import org.eclipse.jdt.internal.ui.viewsupport.JavaElementImageProvider;
import org.eclipse.jdt.internal.ui.viewsupport.TreeHierarchyLayoutProblemsDecorator;
import org.eclipse.jdt.ui.JavaElementImageDescriptor;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.ui.internal.WorkbenchPlugin;
import org.eclipse.ui.internal.decorators.DecoratorManager;

/**
 * @see ILabelDecorator
 */
public class ImageDecorator implements ILabelDecorator {
	
	private ArrayList listeners;
	private ImageDescriptorRegistry fRegistry;
	private boolean preventRecursion = false;
	private TreeHierarchyLayoutProblemsDecorator problemsDecorator;
	private DecoratorManager decman;
	private ImageDescriptor halfFilledPackageID;
	private ImageDescriptor activeConfigFileImage;
	private BuildConfigurator buildConfor;
	
	/**
	 *
	 */
	public ImageDecorator() {
		listeners = new ArrayList(2);
		problemsDecorator = new TreeHierarchyLayoutProblemsDecorator();
		decman = WorkbenchPlugin.getDefault().getDecoratorManager();
		halfFilledPackageID = AspectJImages.BC_HALF_FILLED_PACKAGE.getImageDescriptor();
		activeConfigFileImage = AspectJImages.BC_SELECTED_FILE.getImageDescriptor();
		buildConfor = BuildConfigurator.getBuildConfigurator();
	}

	/**
	 * @see ILabelDecorator#addListener
	 */
	public void addListener(ILabelProviderListener listener)  {
		listeners.add(listener);
	}

	/**
	 * @see ILabelDecorator#dispose
	 */
	public void dispose()  {
	}

	/**
	 * @see ILabelDecorator#isLabelProperty
	 */
	public boolean isLabelProperty(Object element, String property)  {
		return false;
	}

	/**
	 * @see ILabelDecorator#removeListener
	 */
	public void removeListener(ILabelProviderListener listener)  {
		listeners.remove(listener);
	}

	/**
	 * @see ILabelDecorator#decorateImage
	 */
	public Image decorateImage(Image image, Object element)  {
	
		if (preventRecursion)
			return null;
		
		if (element instanceof ICompilationUnit){
			ICompilationUnit comp = (ICompilationUnit)element;
			try {
				element = comp.getCorrespondingResource();
			} catch (JavaModelException e) {
				element = null;
			}
		}
		
		Image img = null;
		if (element instanceof IFile){
			IFile file= (IFile) element;
			if (ProjectProperties.ASPECTJ_SOURCE_FILTER.accept(file.getName())){
				ProjectBuildConfigurator pbc = buildConfor.getProjectBuildConfigurator(file.getProject());
				
				if (pbc == null)
					return null;
				
				if (pbc.getActiveBuildConfiguration().isIncluded(file)){			
					Rectangle rect = image.getBounds();
					int flags = (rect.height == 16)?JavaElementImageProvider.SMALL_ICONS:0;
					img = getImageLabel(getCUImageDescriptor(file, flags));
				} else {
					Rectangle rect = image.getBounds();
					int flags = (rect.height == 16)?JavaElementImageProvider.SMALL_ICONS:0;
					img = getImageLabel(getCUResourceImageDescriptor(file, flags));
				}
			} else {
				if (BuildConfiguration.EXTENSION.equals(file.getFileExtension())){
					ProjectBuildConfigurator pbc = buildConfor.getProjectBuildConfigurator(file.getProject());
					if (pbc != null){
						if (file.equals(pbc.getActiveBuildConfiguration().getFile())){
							img = getImageLabel(activeConfigFileImage);
						}
					}
				
				}
			}
		} else if (element instanceof IPackageFragment){
			IPackageFragment pack = (IPackageFragment)element;
			ProjectBuildConfigurator pbc = buildConfor.getProjectBuildConfigurator(pack.getJavaProject());
			
			if (pbc == null)
				return null;
			
			BuildConfiguration bc = pbc.getActiveBuildConfiguration();
			
			try {
				if (containsIncludedFiles(bc, pack)){
					if (containsExcludedFiles(bc, pack)){
						//half filled package
						img = getImageLabel(halfFilledPackageID);
						
					} else {
						//if all included files are aj files, override icon
						//(full package icon needed)
						if (!pack.containsJavaResources()){
							img = getImageLabel(JavaPluginImages.DESC_OBJS_PACKAGE);
						}
					}
				} else {
					if (containsExcludedFiles(bc, pack))
						img = getImageLabel(JavaPluginImages.DESC_OBJS_EMPTY_PACKAGE);
				}
			} catch (JavaModelException e) {
				// problems with package, better don't do anything
			}
		} else {
			return null;
		}
		if (img != null){
			preventRecursion = true;
			
			//the Java ProblemsDecorator is not registered in the official
			//decorator list of eclipse, so we need it to call ourself.
			//problem: if jdt includes more decorators, we won't know it.
			img = problemsDecorator.decorateImage(img, element);
			
			//apply standard decorators (eg cvs)
			img = decman.decorateImage(img, element);
			preventRecursion = false;
			return img;
		}
		return null;
	}
	
	public static boolean containsIncludedFiles(BuildConfiguration bc, IPackageFragment pack){
		try {
			IJavaElement[] javachildren = pack.getChildren();
			for(int i=0; i<javachildren.length; i++){
				IResource res = javachildren[i].getCorrespondingResource();
				if ((res != null) && (res.getType() == IResource.FILE))
					if (bc.isIncluded(res))
						return true;
			}
				
			Object[] children = pack.getNonJavaResources();
			for(int i=0; i<children.length; i++){
				if (children[i] instanceof IFile){
					IFile f = (IFile)children[i];
					if (("aj".equals(f.getFileExtension()))
						&& (bc.isIncluded(f)))
						return true;
				}
			}
		} catch (JavaModelException e) {
			//assume empty
		}
		return false;
	}
	
	public static boolean containsExcludedFiles(BuildConfiguration bc, IPackageFragment pack){
		try {
			IJavaElement[] javachildren = pack.getChildren();
			for(int i=0; i<javachildren.length; i++){
				IResource res = javachildren[i].getCorrespondingResource();
				if ((res != null) && (res.getType() == IResource.FILE))
					if (!bc.isIncluded(res))
						return true;
			}
				
			Object[] children = pack.getNonJavaResources();
			for(int i=0; i<children.length; i++){
				if (children[i] instanceof IFile){
					IFile f = (IFile)children[i];
					if (("aj".equals(f.getFileExtension()))
						&& (!bc.isIncluded(f)))
						return true;
				}
			}
		} catch (JavaModelException e) {
			//assume empty
		}
		return false;
	}
	
	private Image getImageLabel(ImageDescriptor descriptor){
		if (descriptor == null) 
			return null;	
		return getRegistry().get(descriptor);
	}
	
	private ImageDescriptorRegistry getRegistry() {
		if (fRegistry == null) {
			fRegistry= JavaPlugin.getImageDescriptorRegistry();
		}
		return fRegistry;
	}
	
	public ImageDescriptor getCUResourceImageDescriptor(IFile file, int flags) {
		Point size= useSmallSize(flags) ? JavaElementImageProvider.SMALL_SIZE : JavaElementImageProvider.BIG_SIZE;
		return new JavaElementImageDescriptor(JavaPluginImages.DESC_OBJS_CUNIT_RESOURCE, 0, size);
	}
	
	public ImageDescriptor getCUImageDescriptor(IFile file, int flags) {
		Point size= useSmallSize(flags) ? JavaElementImageProvider.SMALL_SIZE : JavaElementImageProvider.BIG_SIZE;
		return new JavaElementImageDescriptor(JavaPluginImages.DESC_OBJS_CUNIT, 0, size);
	}
	
	private static boolean useSmallSize(int flags) {
		return (flags & JavaElementImageProvider.SMALL_ICONS) != 0;
	}

	/**
	 * @see ILabelDecorator#decorateText
	 */
	public String decorateText(String text, Object element)  {
		return null;
	}
}
