/**********************************************************************
Copyright (c) 2002 IBM Corporation and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html
Contributors:
Adrian Colyer, Andy Clement, Tracy Gardner - initial version
...
**********************************************************************/
package org.eclipse.ajdt.internal.ui.wizards;

//import org.eclipse.jdt.internal.ui.wizards.*;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.ajdt.internal.core.AJDTEventTrace;
import org.eclipse.ajdt.internal.core.AJDTUtils;
import org.eclipse.ajdt.internal.core.resources.AspectJImages;
import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.jdt.internal.ui.JavaPluginImages;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jdt.ui.wizards.NewJavaProjectWizardPage;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;
import org.eclipse.ui.dialogs.WizardNewProjectCreationPage;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;

/**
 * Wizard to create a new AspectJ project
 */
public class AspectJProjectWizard
	extends BasicNewProjectResourceWizard
	implements IExecutableExtension {

	private NewJavaProjectWizardPage fJavaPage;
	private WizardNewProjectCreationPage fMainPage;
	private IConfigurationElement fConfigElement;

	/**
	 * Create a new wizard
	 */
	public AspectJProjectWizard() {
		super();
		setDefaultPageImageDescriptor(AspectJImages.W_NEW_AJ_PRJ.getImageDescriptor());
		setDialogSettings(AspectJPlugin.getDefault().getDialogSettings());
		setWindowTitle(AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.title"));
	}

	/*
	 * @see Wizard#addPages
	 */
	public void addPages() {
		
		//super.addPages();
		fMainPage = new WizardNewProjectCreationPage("NewAspectjProjectCreationWizard");
		
		fMainPage.setTitle(
			AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.MainPage.title"));
		fMainPage.setDescription(
			AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.MainPage.description"));
		fMainPage.setImageDescriptor(AspectJImages.W_NEW_AJ_PRJ.getImageDescriptor());
		
		addPage(fMainPage);
		IWorkspaceRoot root = AspectJPlugin.getWorkspace().getRoot();
		fJavaPage = new NewJavaProjectWizardPage(root, fMainPage);
		addPage(fJavaPage);
	}

	/*
	 * @see Wizard#performFinish
	 */
	public boolean performFinish() {
		IRunnableWithProgress op =
			new WorkspaceModifyDelegatingOperation(fJavaPage.getRunnable());
		try {
			getContainer().run(false, true, op);
		} catch (InvocationTargetException e) {
			String title =
				AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.op_error.title");
			String message =
				AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.op_error.message");
			ExceptionHandler.handle(e, getShell(), title, message);
	
			return false;
		} catch (InterruptedException e) {
			return false;
		}
		BasicNewProjectResourceWizard.updatePerspective(fConfigElement);

		IProject project = fJavaPage.getNewJavaProject().getProject();

        // Bugzilla 46271
        // Force a build of the new AspectJ project using the Java builder
        // so that project state can be created. The creation of project 
        // state means that Java projects can reference this project on their
        // build path and successfully continue to build.
        try {
			project.build(IncrementalProjectBuilder.FULL_BUILD, null);
		} catch (CoreException e) {
			System.err.println("Failed initial Java build of new project "
					+ project.getName() + " : " + e);
			AspectJPlugin.getDefault().getLog().log(e.getStatus());
		}
        
		try {
			// The nature to add is the PluginID+NatureID - it is not the
			// name of the class implementing IProjectNature !!
			// When the nature is attached, the project will be driven through
			// INatureProject.configure() which will replace the normal javabuilder
			// with the aspectj builder.
			AJDTUtils.addAspectJNature(project);
		} catch (Throwable e) {
			System.out.println("> Error creating new project: " + e);
			e.printStackTrace();
		}
		AspectJPlugin.getDefault().setCurrentProject( project );
		selectAndReveal(project);
		AJDTEventTrace.newProjectCreated( project );
		return true;
	}

	/*
	 * Stores the configuration element for the wizard.  The config element will be used
	 * in <code>performFinish</code> to set the result perspective.
	 */
	public void setInitializationData(
		IConfigurationElement cfig,
		String propertyName,
		Object data) {
		fConfigElement = cfig;
	}
}