/*********************************************************************
 Copyright (c) August 2002 IBM Corporation and others.
 All rights reserved.   This program and the accompanying materials
 are made available under the terms of the Common Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/cpl-v10.html 
 Contributors: Sian Whiting - initial version.
 **********************************************************************/

package org.eclipse.ajdt.ui.visualiser.menu;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.eclipse.ajdt.ui.visualiser.AspectVisualiserPlugin;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.part.ViewPart;

/**
 * The Menu part of the Aspect Visualiser.  
 * Displays Aspects, colour buttons and checkboxes.
 * 
 * @author Sian Whiting
 */

public class Menu extends ViewPart {
	
	Button[] buttons;
	Button[] checkboxes;
	Label[] labels;
	Shell[] shells;
	ColorDialog[] colorDialogs;
	Image[] images;
	Color[] colors;
	SelectionAdapter selectionListener;
	MouseListener mouseListener;
	MouseListener labelListener;	
	Composite canvas;
	ScrolledComposite scrollpane;
	
	GridLayout layout = new GridLayout(3, false);
	AspectVisualiserPlugin plugin = AspectVisualiserPlugin.getDefault();
	
	private static Map colourMemory = new HashMap();
	private static Map availableColours = new HashMap();
	
	
	/**
	 * The constructor.
	 */
	public Menu() {
		plugin.setMenu(this);
	}

	/**
	 * Private function used to create square images on colour chooser buttons.
	 */
	private void drawImage (Image image, Color color) {
		GC gc = new GC(image);
		gc.setBackground(color);
		Rectangle bounds = image.getBounds();
		gc.fillRectangle(0, 0, bounds.width, bounds.height);
		gc.drawRectangle(0, 0, bounds.width - 1, bounds.height - 1);
		gc.dispose();
	}


	/**
	 * This is a callback that allows us
	 * to create the composite and initialize it.
	 * It also creates listeners for the colour buttons and the checkboxes.
	 */
	public void createPartControl(Composite parent) {
		scrollpane = new ScrolledComposite(parent, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		canvas = new Composite(scrollpane, SWT.NONE);
		scrollpane.setContent(canvas);
		canvas.setLayout(layout);

		labelListener = new MouseListener() {
			public void mouseUp(MouseEvent e){}
			public void mouseDown(MouseEvent e){}
			public void mouseDoubleClick(MouseEvent e){
				for (int i = 0; i < labels.length; i++){
					if ((Label)e.getSource() == labels[i]){
						plugin.visualiser.openAspectSource(i);
						return;
					}
				}
			}
		};
		
		/* 
		 * Listener for colour buttons - if clicked produces a ColorDialog
		 * then redraws the square image with the chosen colour. 
		 */
		selectionListener = new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (e.getSource() instanceof Button){
					Button button = (Button)e.getSource();
					int location = 0;
					for (int j = 0; j < buttons.length; j++){
						if ((buttons[j]).equals(button)){
							location=j;							
						}
					}	
					RGB rgb = colorDialogs[location].open();
					if (rgb == null){
						return;
					}
					colors[location] = new Color (buttons[location].getDisplay(), rgb);
					Image image = buttons[location].getImage ();
					drawImage (image, colors[location]);
					buttons[location].setImage (image);
					if (!(plugin.visualiser == null)){
						Color[] addColors = new Color[colors.length];
						for (int i = 0; i < colors.length; i++){
							if (checkboxes[i].getSelection()){
								addColors[i] = colors[i];
							}
						}
						plugin.visualiser.draw(addColors);
					}
				}
			}
		};
		
		/* Listener for checkboxes. */
		mouseListener = new MouseListener() {
			public void mouseUp(MouseEvent e){
				if (!(plugin.visualiser == null)){
					Color[] addColors = new Color[colors.length];
					for (int i = 0; i < colors.length; i++){
						if (checkboxes[i].getSelection()){
							addColors[i] = colors[i];
						}
					}
					plugin.visualiser.draw(addColors);
				}
			}
			public void mouseDown(MouseEvent e){}
			public void mouseDoubleClick(MouseEvent e){}	
		};
	}

	/**
	 * The main method - adds aspect names to the menu.
	 */
	public void addAspects(IJavaProject project, Vector aspects){
		clearCanvas();
		buttons = new Button[aspects.size()];
		checkboxes = new Button[aspects.size()];
		labels = new Label[aspects.size()];
		shells = new Shell[aspects.size()];
		images = new Image[aspects.size()];
		colorDialogs = new ColorDialog[aspects.size()];
		colors = new Color[aspects.size()];
		
		for (int i = 0; i < aspects.size(); i++){
			int imageSize = 12;
			//colors[i] = getColor(i);
			colors[i] = getColorFor(project, (String)aspects.elementAt(i));
			buttons[i] = new Button(canvas, SWT.PUSH);
			shells[i] = buttons[i].getShell();
			colorDialogs[i] = new ColorDialog (shells[i]);
			Display display = shells[i].getDisplay ();
			images[i] = new Image(display, imageSize, imageSize);
			buttons[i].setImage(images[i]);
			buttons[i].addSelectionListener(selectionListener);
			Image image = buttons[i].getImage ();
			drawImage (image, colors[i]);
			buttons[i].setImage (image);

			checkboxes[i] = new Button(canvas, SWT.CHECK);
			checkboxes[i].addMouseListener(mouseListener);
			checkboxes[i].setSelection(true);
			labels[i] = new Label(canvas, SWT.NONE);
			labels[i].setText((String)aspects.elementAt(i));
			labels[i].addMouseListener(labelListener);
		}
		canvas.layout();
		canvas.setSize(canvas.computeSize(SWT.DEFAULT, SWT.DEFAULT, true));
		if (!(plugin.visualiser == null)){
			plugin.visualiser.draw(colors);
		}
		
	}

	/**
	 * Private method - returns a different color for inputs 0-9 and grey after that.
	 */
	private Color getColor(int i){
		switch(i){
			case 0:
			return ColorConstants.red;
			case 1:
			return ColorConstants.blue;
			case 2:
			return ColorConstants.yellow;
			case 3:
			return ColorConstants.green;
			case 4:
			return ColorConstants.darkBlue;
			case 5:
			return ColorConstants.cyan;
			case 6:
			return ColorConstants.orange;
			case 7:
			return ColorConstants.lightBlue;
			case 8:
			return ColorConstants.darkGreen;
			case 9:
			return ColorConstants.darkGray;
		}
		return ColorConstants.gray;
	}
	
	
	private Color getColorFor(IJavaProject p, String aspect) {
		Color aspectColour;
		String key = new String(p + ":" + aspect);
		if (colourMemory.containsKey(key)) {
			aspectColour = (Color) colourMemory.get(key);
		} else {
			aspectColour = getNextColourFor(p);
			colourMemory.put(key,aspectColour);		
		}
		
		return aspectColour;
	}
	
	
	private Color getNextColourFor(IJavaProject p) {
		Color c = ColorConstants.gray;
		if (!availableColours.containsKey(p.toString())) {
			List colourList = new ArrayList();
			colourList.add(ColorConstants.red);
			colourList.add(ColorConstants.blue);
			colourList.add(ColorConstants.yellow);
			colourList.add(ColorConstants.green);
			colourList.add(ColorConstants.darkBlue);
			colourList.add(ColorConstants.cyan);
			colourList.add(ColorConstants.orange);
			colourList.add(ColorConstants.lightBlue);
			colourList.add(ColorConstants.darkGreen);
			colourList.add(ColorConstants.darkGray);
			availableColours.put(p.toString(),colourList);
		}

		List colours = (List) availableColours.get(p.toString());
		if ( !colours.isEmpty()) {
			c = (Color) colours.get(0);
			colours.remove(0);
		}			
		return c;	

	}
	
	
	/**
	 * Private method to clear the menu.
	 */
	private void clearCanvas(){
		Control[] children = canvas.getChildren();
		if (children.length > 0){
			for (int i = 0; i < children.length; i++){
				children[i].dispose();
			}	
		}

	}
	
	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
	}

	/**
	 * Dispose of the menu when closed.
	 */
	public void dispose(){
		plugin.removeMenu();
	}

}