/*********************************************************************
 Copyright (c) August 2002 IBM Corporation and others.
 All rights reserved.   This program and the accompanying materials
 are made available under the terms of the Common Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/cpl-v10.html 
 Contributors: Sian Whiting - initial version.
 **********************************************************************/

package org.eclipse.ajdt.ui.visualiser.views;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.eclipse.draw2d.Button;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.draw2d.Figure;
import org.eclipse.draw2d.FlowLayout;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.LightweightSystem;
import org.eclipse.draw2d.LineBorder;
import org.eclipse.draw2d.MouseListener;
import org.eclipse.draw2d.RectangleFigure;
import org.eclipse.draw2d.ScrollPane;
import org.eclipse.draw2d.Shape;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;

/**
 * When given a canvas and various data AspectDrawing displays the data on the
 * canvas using draw2d.  Designed for aspect visualisation.
 * 
 * @author Sian Whiting
 */
public class AspectDrawing {
	
	int spacing = 3;
	int maxBarWidth = 60;
	int minBarWidth = 20;
	int highlightDepth = 3;
	int location = -1;
	int classNum = -1;
	int lineNum = -1;
	float zoom = 1;
	float max_zoom = 4;
	float scale;
	Shape[] rectangles;
	Vector[] highlights;
	Vector[] lineNums;
	Vector lengths;
	Vector names;
	Vector tooltips;
	Vector aspects;
	Map[] map;
	IFigure panel;
	MouseListener mouseListener;
	Button[] buttons;
	Color[] colors;
	Listener listener;
	boolean limit_mode = false;
	public org.eclipse.swt.graphics.Rectangle viewsize;

/**
 * The Constructor. 
 * Instantiates the panel and mouse listener and implements the mouse 
 * listener event handlers.
 */
	public AspectDrawing () {
		panel = new Figure();
		
		/* Instantiate the mouse listener. */
		mouseListener = new org.eclipse.draw2d.MouseListener() {
			public void mousePressed(org.eclipse.draw2d.MouseEvent e){}
			public void mouseReleased(org.eclipse.draw2d.MouseEvent e){}
			public void mouseDoubleClicked(org.eclipse.draw2d.MouseEvent e){
				if (e.getSource() instanceof RectangleFigure){
					location = -1;
					RectangleFigure rectangle = (RectangleFigure)e.getSource();
					for (int j = 0; j < lengths.size(); j++){
						if (((RectangleFigure)rectangles[j]).equals(rectangle)){
							location = j;	
							listener.handleEvent(new Event());
							return;						
						}
					}
					if (location == -1){
						for (int i = 0; i < highlights.length; i++){
							for (int k = 0; k < highlights[i].size(); k++){
								if (((RectangleFigure)highlights[i].elementAt(k)).equals(rectangle)){
									classNum = i;
									Point point = rectangle.getLocation();
									lineNum = ((Integer)lineNums[i].elementAt(k)).intValue();
									listener.handleEvent(new Event());
									return;
								}									
							}
						}	
					}
				}
			}
		};
	}
	
	/**
	 * A class which instantiates an AspectDrawing can add itself as a listener
	 * and is therefore alerted to selection changes. An Aspect Drawing can only
	 * have one listener.
	 */
	public void addListener (Listener listener){
		this.listener = listener;
	}
	
	/**
	 * Removes the listener.
	 */
	public void removeListener (Listener listener){
		if (this.listener == listener) this.listener = null;
	}
	
	/**
	 * Draw is the main method.  It is passed a canvas and some data and it 
	 * calls subsidiary methods to display the data graphically on the canvas.
	 */
	public void draw (Vector names, Vector tooltips, Vector lengths, Vector aspects, Map[] map, Color[] colors, Canvas canvas){
		clearPanel();
		this.lengths = lengths;
		this.names = names;
		this.tooltips = tooltips;
		this.map = map;
		this.colors = colors;
		this.aspects = aspects;
		int size = names.size();
		rectangles = new Shape[size];
		highlights = new Vector[size];
		lineNums = new Vector[size];
		org.eclipse.draw2d.Button[] buttons;
		panel = new Figure();
		FlowLayout layout= new FlowLayout();
		layout.setMinorSpacing(spacing);
		panel.setLayoutManager(layout);
		ScrollPane scrollpane = new ScrollPane();
		panel.setBorder(new LineBorder(ColorConstants.menuBackground, 5));
		viewsize = canvas.getClientArea();
		if (zoom > 1) {
			viewsize.width = (int)((float)viewsize.width * zoom);
			viewsize.height = (int)((float)viewsize.height * zoom);
		}
		int max_depth = viewsize.height-30;
		LightweightSystem lws = new LightweightSystem(canvas);
		scrollpane.setView(panel);
		lws.setContents(scrollpane);
		addShapes(max_depth, size, viewsize);
		scrollpane.setBounds(panel.getBounds());
		scrollpane.add(panel);
	}
	
	/**
	 * This draw function is used to display blank rectangles on the canvas
	 * without aspect lines.
	 */
	public void draw (Vector names, Vector tooltips, Vector lengths, Canvas canvas){
		clearPanel();
		this.lengths = lengths;
		this.names = names;
		this.tooltips = tooltips;
		int size = lengths.size();
		rectangles = new Shape[size];
		highlights = new Vector[size];
		lineNums = new Vector[size];
		org.eclipse.draw2d.Button[] buttons;
		FlowLayout layout= new FlowLayout();
		layout.setMinorSpacing(spacing);
		panel.setLayoutManager(layout);
		panel.setBorder(new LineBorder(ColorConstants.menuBackground, 5));
		ScrollPane scrollpane = new ScrollPane();
		viewsize = canvas.getClientArea();
		if (zoom > 1) {
			viewsize.width = (int)((float)viewsize.width * zoom - 30);
			viewsize.height = (int)((float)viewsize.height * zoom - 30);
		}
		int max_depth = viewsize.height-30;
		LightweightSystem lws = new LightweightSystem(canvas);
		scrollpane.setView(panel);
		lws.setContents(scrollpane);
		addShapes(max_depth, size, viewsize);
		scrollpane.setBounds(panel.getBounds());
		scrollpane.add(panel);
		
	}

	/**
	 * AddShapes is called by draw to add the shapes to the panel.
	 */
	private void addShapes (int max_depth, int size, org.eclipse.swt.graphics.Rectangle viewsize){
		
		int width = 0;
		boolean hScrollBar = false;
		int maximum = 0;
		int num = 0;
		
		Point[] points = new Point[size];
		buttons = new org.eclipse.draw2d.Button[size];
		
		/* Calculates the maximum rectangle depth from all rectangles being drawn. */
		for (int i = 0; i < size; i++){ 
			if (limit_mode){
				if (!(map[i].isEmpty())){
					Collection col = map[i].values();
					boolean selected = false;
					Iterator iterator = col.iterator();
					while (iterator.hasNext()){
						if(getColor((String)((List)iterator.next()).get(0))!=null)
							selected = true;
					}
					if (selected){
						num+=1;
						if (((Integer)lengths.elementAt(i)).intValue() > maximum)
						maximum = ((Integer)lengths.elementAt(i)).intValue();
					}
				}
			} else {
				if (((Integer)lengths.elementAt(i)).intValue() > maximum)
					maximum = ((Integer)lengths.elementAt(i)).intValue();
			}
		}
		
		/* 
		 * Calculates width and scale based on number of rectangles
		 * and maximum rectangle depth respectively, and viewsize.
		 */
		if (!limit_mode){
			num = size;
		}
		if (num*(minBarWidth+spacing) > (viewsize.width) - 10){
			width = minBarWidth;
			hScrollBar = true;
		} else if (num*(maxBarWidth+spacing) < (viewsize.width) - 10){
			width = maxBarWidth;
		} else width = (int)((float)viewsize.width - 10) / num - spacing;
		
		if (hScrollBar){
			max_depth = max_depth - 20;
		}
		scale = (float)max_depth / (float)maximum;
		
		/* Add shapes to the panel. */
		for (int i = 0; i < size; i++){
			rectangles[i] = new RectangleFigure();
			highlights[i] = new Vector();
			lineNums[i] = new Vector();
			rectangles[i].setSize(width, ((int)(((float)((Integer)lengths.elementAt(i)).intValue()) * scale)) + 20);
			rectangles[i].addMouseListener(mouseListener);
			rectangles[i].setToolTip(new org.eclipse.draw2d.Label((String)tooltips.elementAt(i)));
			buttons[i] = new org.eclipse.draw2d.Button((String)names.elementAt(i));
			buttons[i].setBackgroundColor(ColorConstants.buttonDarker);
			buttons[i].setSize(rectangles[i].getSize().width, 20);
			buttons[i].setStyle(org.eclipse.draw2d.Clickable.STYLE_TOGGLE);
			rectangles[i].add(buttons[i]);
			Point point = rectangles[i].getLocation();
			if (map != null){
				Set mapSet = map[i].entrySet();
				rectangles[i].setBackgroundColor(ColorConstants.buttonDarkest);
				if (!(mapSet.isEmpty())){
					Iterator iterator = mapSet.iterator();
					while (iterator.hasNext()){
						Map.Entry mapEntry = (Map.Entry)iterator.next();
						List aspectList = (List)mapEntry.getValue();
						int numStripes = 0;
						int first = -1;
						
						for (int k = 0; k < aspectList.size(); k++){
							Color newColor = getColor((String)aspectList.get(k));
							if (newColor != null) {
								if (first == -1){
									first = k;
								}
								numStripes++;
							}
						}
						if (first != -1){
							Color newColor = getColor((String)aspectList.get(first));
							
							if (rectangles[i].getBackgroundColor() != ColorConstants.white){
								rectangles[i].setBackgroundColor(ColorConstants.white);
							}
							Integer lineNum = (Integer)mapEntry.getKey();
							RectangleFigure f = new RectangleFigure();
							f.setLocation(new Point(point.x + 1, ((float)(lineNum.intValue()) * scale) + 20 + point.y));
							if (f.getLocation().y+ highlightDepth >= rectangles[i].getLocation().y+rectangles[i].getSize().height){
								f.setLocation(new Point(point.x + 1, rectangles[i].getLocation().y+rectangles[i].getSize().height-highlightDepth-1 )); 
							}
							if (rectangles[i].findFigureAt(f.getLocation()) != rectangles[i] || rectangles[i].findFigureAt(f.getLocation().x, f.getLocation().y+ highlightDepth-1) != rectangles[i]){
								f.setSize(width - 2, highlightDepth * 2 / 3);
							} else {
								f.setSize(width - 2, highlightDepth);
							}
							f.setOutline(false);
							f.setBackgroundColor(newColor);						
							f.addMouseListener(mouseListener);
							rectangles[i].add(f);
							highlights[i].add(f);
							lineNums[i].add(lineNum);
							if (numStripes >= 1) {
								int stripeNum = 1;
								for (int j = first; j < aspectList.size(); j++){
									Color color = getColor((String)aspectList.get(j));
									if (color != null){						
										RectangleFigure stripe = new RectangleFigure();
										stripe.setBackgroundColor(color);
										stripe.setLocation(new Point(f.getLocation().x + ((f.getSize().width / numStripes * (stripeNum - 1))), f.getLocation().y));
										stripe.setSize(f.getSize().width / numStripes, f.getSize().height);
										stripe.setOutline(false);
										f.add(stripe);
										stripeNum++;
									}
								}
							}
						}
					}
				}
			}
			if (limit_mode){
				if (rectangles[i].getBackgroundColor() == ColorConstants.white){
					panel.add(rectangles[i]);
				}
			}
			else {
				panel.add(rectangles[i]);
			}
		}		
	}
	
	/**
	 * Called in subselect mode so that only selected bars are redrawn.
	 */
	public boolean subSelect(Canvas canvas){
		if (map != null){
			int count = 0;
			for (int i = 0; i < lengths.size(); i++){
				if (buttons[i].isSelected()){
					count += 1;
				}
			}
			if (count == 0){
				return false;
			} else {
				Vector vectorSelection = new Vector();
				Vector elementSelection = new Vector();
				Vector tooltipSelection = new Vector();
				Map[] mapSelection = new Map[count];
				count = 0;
				for(int i = 0; i < lengths.size(); i++){
					if (buttons[i].isSelected()){
						vectorSelection.add(lengths.elementAt(i));
						elementSelection.add(names.elementAt(i));
						tooltipSelection.add(tooltips.elementAt(i));
						mapSelection[count] = map[i];
						count += 1;
					}
				}
				draw(elementSelection, tooltipSelection, vectorSelection, aspects, mapSelection, colors, canvas);
				return true;
			}
			
			
		} else {	
			int count = 0;
			for (int i = 0; i < lengths.size(); i++){
				if (buttons[i].isSelected()){
					count += 1;
				}
			}
			if (count == 0){
				return false;
			} else {
				Vector vectorSelection = new Vector();
				Vector elementSelection = new Vector();
				Vector tooltipSelection = new Vector();
				count = 0;
				for (int i = 0; i < lengths.size(); i++){
					if (buttons[i].isSelected()){
						vectorSelection.add(lengths.elementAt(i));
						elementSelection.add(names.elementAt(i));
						tooltipSelection.add(tooltips.elementAt(i));
						count += 1;
					}
				}
				draw(elementSelection, tooltipSelection, vectorSelection, canvas);
				return true;
			}
		}
	}
	
	/**
	 * Called when entering or leaving limit mode to draw only affected rectangles.
	 */
	public void Limit (Canvas canvas, boolean limit_mode){
		this.limit_mode = limit_mode;
		draw(names, tooltips, lengths, aspects, map, colors, canvas);		
	}
	
	/**
	 * Used to clear the panel
	 */
	private void clearPanel (){
		java.util.List children = panel.getChildren();
		if (children.size() > 0){
			for (int i = 0; i < children.size();){
				panel.remove((IFigure)children.get(i));
			}	
		}
	}
	
	/**
	 * Increases magnification by 0.5 (up to max_zoom) and redraws.
	 * Returns true if further zoom in is possible.
	 */
	public boolean zoomIn (Canvas canvas){
		if (zoom + (float)0.5 <= max_zoom){
			float old_zoom = zoom;
	 		zoom = zoom + (float)0.5;
	 		maxBarWidth = (int)(((float)maxBarWidth / old_zoom) * zoom);
			minBarWidth = (int)(((float)minBarWidth / old_zoom) * zoom);
			highlightDepth = (int)(((float)highlightDepth / old_zoom) * zoom);	
			draw(names, tooltips, lengths, aspects, map, colors, canvas);
		}
		return(zoom + (float)0.5 <= max_zoom);
	}
	
	/**
	 * Decreases magnification by 0.5 (down to 1) and redraws.
	 * Returns true if further zoom out is possible.
	 */
	public boolean zoomOut (Canvas canvas){
		if (zoom - (float)0.5 <= 1){
			zoom = 1;
			maxBarWidth = 60;
			minBarWidth = 20;
			highlightDepth = 3;
		} else {
			float old_zoom = zoom;
			zoom = zoom - (float)0.5;
			maxBarWidth = (int)(((float)maxBarWidth / old_zoom) * zoom);
			minBarWidth = (int)(((float)minBarWidth / old_zoom) * zoom);
			highlightDepth = (int)(((float)highlightDepth / old_zoom) * zoom);	
		}
		draw(names, tooltips, lengths, aspects, map, colors, canvas);
		return(zoom > 1);
	}
	
	/**
	 * Given the name of an aspect getColor returns the colour associated with that
	 * aspect.
	 */
	private Color getColor (String aspect){
		if (aspects != null && colors != null){
			for (int i = 0; i < aspects.size(); i++){
				if (aspect.equals((String)aspects.elementAt(i)) && colors.length >= i){
					return colors[i];
				} 
			}
		}
		return null;
	}

}
