/**********************************************************************
Copyright (c) 2002 IBM Corporation and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html
Contributors:
Adrian Colyer, Andy Clement, Tracy Gardner - initial version
Ian McGrath - added ability to use existing project structures
...
**********************************************************************/

package org.eclipse.ajdt.internal.ui.wizards;

//import org.eclipse.jdt.internal.ui.wizards.*;

import java.lang.reflect.InvocationTargetException;
import java.io.File;

import org.eclipse.ajdt.internal.core.AJDTEventTrace;
import org.eclipse.ajdt.internal.core.AJDTUtils;
import org.eclipse.ajdt.internal.core.resources.AspectJImages;
import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExecutableExtension;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jdt.internal.ui.util.ExceptionHandler;
import org.eclipse.jdt.ui.wizards.NewJavaProjectWizardPage;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.actions.WorkspaceModifyDelegatingOperation;
import org.eclipse.ui.wizards.newresource.BasicNewProjectResourceWizard;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IPath;

/**
 * Wizard to create a new AspectJ project
 */
public class AspectJProjectWizard
	extends BasicNewProjectResourceWizard
	implements IExecutableExtension {

	protected NewJavaProjectWizardPage fJavaPage;
	protected AspectJWizardNewProjectCreationPage fMainPage;
	private IConfigurationElement fConfigElement;

	/**
	 * Create a new wizard
	 */
	public AspectJProjectWizard() {
		super();
		setDefaultPageImageDescriptor(AspectJImages.W_NEW_AJ_PRJ.getImageDescriptor());
		setDialogSettings(AspectJPlugin.getDefault().getDialogSettings());
		setWindowTitle(AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.title"));
	}

	/*
	 * @see Wizard#addPages
	 */
	public void addPages() {
		
		//super.addPages();
		fMainPage = new AspectJWizardNewProjectCreationPage("NewAspectjProjectCreationWizard");
		
		fMainPage.setTitle(
			AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.MainPage.title"));
		fMainPage.setDescription(
			AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.MainPage.description"));
		fMainPage.setImageDescriptor(AspectJImages.W_NEW_AJ_PRJ.getImageDescriptor());
		
		addPage(fMainPage);
		IWorkspaceRoot root = AspectJPlugin.getWorkspace().getRoot();
		fJavaPage = new NewJavaProjectWizardPage(root, fMainPage);
		addPage(fJavaPage);
	}

	/*
	 * @see Wizard#performFinish
	 */
	public boolean performFinish() {

		IProject project = fJavaPage.getNewJavaProject().getProject();

		IWorkspaceRoot wRoot = AspectJPlugin.getWorkspace().getRoot();
		File projectPath = wRoot.getLocation().append(project.getName()).toFile();
		File classpath = wRoot.getLocation().append(project.getName() + IPath.SEPARATOR + ".classpath").toFile();
		File projectDetailsFile = wRoot.getLocation().append(project.getName() + IPath.SEPARATOR + ".project").toFile();
		boolean projectFolderExists = projectPath.exists();
		boolean classpathExists = classpath.exists();
		boolean projectFileExists = projectDetailsFile.exists();
		boolean useExisting = false;
		
		if(projectFolderExists && classpathExists && projectFileExists) {  // enough data to reconstruct the project
			useExisting = true;
		}
		
		if(!useExisting) {      //overwrite any existing structure

			IRunnableWithProgress op =
				new WorkspaceModifyDelegatingOperation(fJavaPage.getRunnable());
			try {
				getContainer().run(false, true, op);
			} catch (InvocationTargetException e) {
				String title =
					AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.op_error.title");
				String message =
					AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.op_error.message");
				ExceptionHandler.handle(e, getShell(), title, message);
				
				return false;
			} catch (InterruptedException e) {
				return false;
			}
			
			BasicNewProjectResourceWizard.updatePerspective(fConfigElement);
			project = fJavaPage.getNewJavaProject().getProject();
		}

		else {            //use existing structure
			IProgressMonitor monitor = new NullProgressMonitor();
			try {
				project.create(monitor);
				project.open(monitor);
			} catch (CoreException e) {	
		        String title = AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.build_error.title");
		        String message = AspectJPlugin.getResourceString("NewAspectjProjectCreationWizard.build_error.message");
		        ExceptionHandler.handle(e, getShell(), title, message);
		        return false;
			}
		}

		boolean completed = finalizeNewProject(project, useExisting);
		return completed;
	}

	
	/**
	 * Builds and adds the necessary properties to the new project and updates the workspace view
	 */
	private boolean finalizeNewProject(IProject project, boolean alreadyExists) {
		
		BasicNewProjectResourceWizard.updatePerspective(fConfigElement);
		
        // Bugzilla 46271
        // Force a build of the new AspectJ project using the Java builder
        // so that project state can be created. The creation of project 
        // state means that Java projects can reference this project on their
        // build path and successfully continue to build.
		final IProject thisProject = project;
		ProgressMonitorDialog dialog = new ProgressMonitorDialog(getShell());
		try {
			dialog.run(true, true, new IRunnableWithProgress() {
				public void run(IProgressMonitor monitor)
						throws InvocationTargetException {
					monitor.beginTask("", 2);
					try {
						monitor
								.setTaskName(AspectJPlugin
										.getResourceString("OptionsConfigurationBlock.buildproject.taskname"));
						thisProject.build(
								IncrementalProjectBuilder.FULL_BUILD,
								new SubProgressMonitor(monitor, 2));
					} catch (CoreException e) {
						System.err.println("Failed initial Java build of new project "
								+ thisProject.getName() + " : " + e);
						AspectJPlugin.getDefault().getLog().log(e.getStatus());
					} finally {
						monitor.done();
					}
				}
			});
			
			// The nature to add is the PluginID+NatureID - it is not the
			// name of the class implementing IProjectNature !!
			// When the nature is attached, the project will be driven through
			// INatureProject.configure() which will replace the normal javabuilder
			// with the aspectj builder.
			if(!alreadyExists) {
				AJDTUtils.addAspectJNature(project);
			}
	        
		} catch (InterruptedException e) {
			// build cancelled by user
			return false;
			
		} catch (Throwable e) {
			System.out.println("> Error creating new project: " + e);
			e.printStackTrace();
			return false;
		}
		
		project = thisProject;
		AspectJPlugin.getDefault().setCurrentProject( project );
		selectAndReveal(project);
		AJDTEventTrace.newProjectCreated( project );
		return true;
	}
	
	/*
	 * Stores the configuration element for the wizard.  The config element will be used
	 * in <code>performFinish</code> to set the result perspective.
	 */
	public void setInitializationData(
		IConfigurationElement cfig,
		String propertyName,
		Object data) {
		fConfigElement = cfig;
	}
}