/*********************************************************************
 Copyright (c) August 2002 IBM Corporation and others.
 All rights reserved.   This program and the accompanying materials
 are made available under the terms of the Common Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/cpl-v10.html 
 Contributors: Sian Whiting - initial version
 **********************************************************************/

package org.eclipse.ajdt.ui.visualiser.views;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Vector;

import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.ajdt.ui.visualiser.AspectVisualiserPlugin;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.ViewPart;

/**
 * This class represents the main view of the Aspect Visualiser.
 * It responds to selection change in the workbench.
 * 
 * @author Sian Whiting
 */

public class AV extends ViewPart implements ISelectionListener{

	AspectVisualiserPlugin plugin = AspectVisualiserPlugin.getDefault();
	AspectDrawing aspectDrawing = new AspectDrawing();
	private Action package_view;
	private Action class_view;
	private Action sub_select;
	private Action doubleClickAction;
	private Action limit;
	private Action zoom_in;
	private Action zoom_out;
	private IWorkbenchPart part;
	private ISelection sel;
	boolean in_package_view = false;
	boolean in_sub_select_mode = false;
	boolean limit_mode = false;
	private Object nowShowing = null;
	Canvas canvas;
	Vector vector;
	Vector elements;
	Vector aspects;
	Color[] colors;
	Vector names;
	Vector tooltips;
	Map[] map;
	public boolean justOpenedFile = false;
	org.eclipse.swt.widgets.Listener canvasListener;
	Set aspectSet;
	java.util.List list;
	org.eclipse.swt.graphics.Rectangle viewsize;


	/**
	 * The constructor. 
	 */
	public AV (){

			plugin.setVisualiser(this);
	}

	/**
	 * This is a callback that will allow us
	 * to create the view and initialize it.
	 * It also implements a listener to the canvas which is
	 * drawn by aspectDrawing in order to be able to respond to
	 * selection changes there.
	 */
	public void createPartControl (Composite parent) {

		canvas = new Canvas(parent, SWT.NONE);
		vector = new Vector();
		elements = new Vector();
		aspects = new Vector();
		names = new Vector();
		tooltips = new Vector();
		
		makeActions();
		class_view.setChecked(true);
		zoom_out.setEnabled(false);
		hookContextMenu();
		contributeToActionBars();
		getSite().getPage().addSelectionListener(this);
		
		/* canvasListener listens to aspectDrawing and responds to events. */
		canvasListener = new Listener(){
			public void handleEvent (Event e){
				int location = aspectDrawing.location;
				
				if (!(in_package_view)){
					IMarker marker;
					if (location != -1){	
						marker = getMarker(location);
					} else {
						marker = getMarker(aspectDrawing.classNum, aspectDrawing.lineNum);
					}
					if (marker != null){
						try {
							IDE.openEditor(part.getSite().getWorkbenchWindow().getActivePage( ),marker,true);
						} catch (Exception ex ) {
							System.err.println(ex);
						}
					}
				} else { // in package view
					if (location >= 0){
						in_package_view = false;
						class_view.setChecked(true);
						package_view.setChecked(false);
						in_package_view = false;
						in_sub_select_mode = true;
						sub_select.setChecked(true);
						Vector v = new Vector();
						v.add(elements.elementAt(location));
						list = plugin.getSelectedClasses(v);
						vector.removeAllElements();
						elements.removeAllElements();
						aspects.removeAllElements();
						names.removeAllElements();
						tooltips.removeAllElements();
	
						map = new Map[list.size()];
						for (int i = 0; i < list.size(); i++) {
							Object[] info = new Object[3];
							info = (Object[]) list.get(i);
							addToVectors(
								((Integer) info[1]).intValue(),
								(IResource) info[0]);
							map[i] = (Map) info[2];
						}
						IJavaProject jp = JavaCore.create(AspectJPlugin.getDefault().getCurrentProject());
						aspectStuff(jp);
					}
				}
			}
		};
		
		aspectDrawing.addListener(canvasListener);
			
		/* Prime the selection. */
		selectionChanged(null, getSite().getPage().getSelection());
		
		/* Resize listener - redraws when resized. */
		canvas.addControlListener(new ControlAdapter() {
			public void controlResized (ControlEvent e) {
				if (in_sub_select_mode){
					selectionChanged(part,sel);	
				} else if (part != null && sel != null){
					aspectDrawing.draw(names, tooltips, vector, aspects, map, colors, canvas);
				}
			}
		});
	}
	
	/**
	 * Called by AspectMenu when the colour selections or aspects selection
	 * in the menu has changed.
	 */
	public void draw (Color[] colors){

		this.colors = colors;
        if (map != null && aspects != null){
			aspectDrawing.draw(names, tooltips, vector, aspects, map, colors, canvas);
		} else {
			aspectDrawing.draw(names, tooltips, vector, canvas);
        }
	}
	
	/**
	 * Selection changed is called whenever the selection in the workbench changes.
	 * It in turn calls methods in AspectVisualsierPlugin to get the data and
	 * methods in AspectDrawing to draw the visualisation.
	 */
	public void selectionChanged (IWorkbenchPart part, ISelection sel) {

		if (!(sel instanceof IStructuredSelection)){
			return;
		} else {		
			/* Store part and sel for manual refreshing. */
			this.part = part;
			this.sel = sel;
		}
						
		/* If in subselect mode then do not refresh the view. */
		if (in_sub_select_mode){
			return;
		}
				
		/* If the selection is structured add packages or classes to list. */
      	IStructuredSelection structured = (IStructuredSelection)sel;

	    Object o = structured.getFirstElement();
	    if ( o instanceof IJavaElement) {
	    	nowShowing = o;      
			updateDisplay();
	    }
	}

	private void updateDisplay() {
		if (null == nowShowing) {
			nowShowing = JavaCore.create(AspectJPlugin.getDefault().getCurrentProject());			
		}
		
		if (in_package_view) {
		
			if (nowShowing instanceof IPackageFragment){
				clearVectors();
				IPackageFragment PF = (IPackageFragment)nowShowing;
		  		list = plugin.getAllPackages(PF);
		  		map = new Map[list.size()];
				for (int i = 0; i < list.size(); i++){
					Object[] info = new Object[3];
					info = (Object[]) list.get(i);
					addToVectors(((Integer)info[1]).intValue(), (IResource)info[0]);
					map[i] = (Map)info[2];
				}	
				aspectStuff(PF.getJavaProject());
			}
		
			else if (nowShowing instanceof IJavaProject){
				clearVectors();
				IJavaProject JP = (IJavaProject)nowShowing;
				aspectSet = plugin.getAllAspects(JP);
				list = plugin.getAllPackages(JP);
				map = new Map[list.size()];
				for(int i = 0; i < list.size(); i++){
					Object[] info = new Object[3];
					info = (Object[]) list.get(i);
					addToVectors(((Integer)info[1]).intValue(), (IResource)info[0]);
					map[i] = (Map)info[2];
				}
				aspectStuff(JP.getJavaProject());	
			}
		
			else if (nowShowing instanceof ICompilationUnit){
				clearVectors();
				in_package_view = false;
				package_view.setChecked(false);
				class_view.setChecked(true);
				ICompilationUnit CU = (ICompilationUnit)nowShowing;
				aspectSet = plugin.getAllAspects(CU.getJavaProject());
				list = plugin.getAllClasses(CU);
				map = new Map[list.size()];
				for (int i = 0; i < list.size(); i++){
					Object[] info = new Object[3];
		  			info = (Object[]) list.get(i);
		  			addToVectors(((Integer)info[1]).intValue(), (IResource)info[0]);
		  			map[i] = (Map)info[2];
				}
				aspectStuff(CU.getJavaProject());		
		  } else return;
		}
		
		else if (!(in_package_view)){ // We are in class view
		
		    if (nowShowing instanceof ICompilationUnit) {
				clearVectors();
				ICompilationUnit CU = (ICompilationUnit) nowShowing;
				aspectSet = plugin.getAllAspects(CU.getJavaProject());
				list = plugin.getAllClasses(CU);
				map = new Map[list.size()];
				for (int i = 0; i < list.size(); i++) {
					Object[] info = new Object[3];
					info = (Object[]) list.get(i);
					addToVectors(((Integer) info[1]).intValue(), (IResource) info[0]);
					map[i] = (Map) info[2];
				}
				aspectStuff(CU.getJavaProject());
			}
		
			if (nowShowing instanceof IPackageFragment) {
				clearVectors();
				IPackageFragment PF = (IPackageFragment) nowShowing;
				list = plugin.getAllClasses(PF);
				aspectSet = plugin.getAllAspects(PF.getJavaProject());
				map = new Map[list.size()];
				for (int i = 0; i < list.size(); i++) {
					Object[] info = new Object[3];
					info = (Object[]) list.get(i);
					addToVectors(((Integer) info[1]).intValue(), (IResource) info[0]);
					map[i] = (Map) info[2];
				}
				aspectStuff(PF.getJavaProject());
			
			} else if (nowShowing instanceof IJavaProject) {
				clearVectors();
				IJavaProject JP = (IJavaProject) nowShowing;
				aspectSet = plugin.getAllAspects(JP);
				list = plugin.getAllClasses(JP);
				map = new Map[list.size()];
				for (int i = 0; i < list.size(); i++) {
					Object[] info = new Object[3];
					info = (Object[]) list.get(i);
					addToVectors(((Integer) info[1]).intValue(), (IResource) info[0]);
					map[i] = (Map) info[2];
				}
				aspectStuff(JP);
			}
		}
	}  

	/**
	 * Private method to remove all elements from all vectors and the map.
	 */
	private void clearVectors (){

		map = null;
		vector.removeAllElements();
		elements.removeAllElements();
		aspects.removeAllElements();
		names.removeAllElements();
		tooltips.removeAllElements();
	}
	
	/**
	 * Private method to add aspects to the vector aspects and the menu.
	 */
	private void aspectStuff (IJavaProject project){

		if (aspectSet != null) {
			int num = aspectSet.size();
      		Object[] aspectArray = aspectSet.toArray();
      		IPath[] aspectPaths = new IPath[num];
      		for (int k = 0; k < num; k++){      		
      			if (aspectArray[k] instanceof IResource) {
      				aspectPaths[k] = ((IResource)aspectArray[k]).getLocation();
      				aspectPaths[k] = aspectPaths[k].removeFileExtension();
	     	 		aspects.add(aspectPaths[k].lastSegment());
      			}
      		}
     	}
     	if (!(plugin.menu == null)){
      		plugin.menu.addAspects(project, aspects);
     	}
     	if (map != null && aspects != null){
	      aspectDrawing.draw(names, tooltips, vector, aspects, map, colors, canvas);
      	} 
    }
	
	/**
	 * Called by the menu to open an aspect's source file in the editor.
	 */	
	public void openAspectSource (int index){
		
		Iterator iterator = aspectSet.iterator();
		int count = 0;
		
		while (iterator.hasNext()){
			IResource ir = (IResource)iterator.next();
			if (index == count){
				if (ir != null){
					try {
						IMarker marker = ir.createMarker(IMarker.MARKER);
						if (marker != null){
							IDE.openEditor(	part.getSite().getWorkbenchWindow().getActivePage( ),marker,true );
						}
					} catch (Exception ex) {
						System.err.println(ex);
					}
					return;
				}				
			} else {
				count++;
			}
			selectionChanged(part, sel);
		}					
	}
	
	/**
	 * Private method to get a marker for a source file given it's location in the vector elements.
	 */
	private IMarker getMarker (int location){

		IMarker marker = null;

		if (elements.elementAt(location) instanceof IResource){
			IResource ir = (IResource)elements.elementAt(location);			
			if (ir != null) {
				try {
					marker = ir.createMarker(IMarker.MARKER);
				}
				catch (CoreException coreEx) {
					System.err.println( coreEx );
				}
			}
		}
		return marker;
	}
	
	/**
	 * Private method to get a marker for a source file given location and required line number.
	 */
	private IMarker getMarker (int location, int lineNum){

		IMarker marker = null;

		if (elements.elementAt(location) instanceof IResource){
			IResource ir = (IResource)elements.elementAt(location);			
			if (ir != null) {
				try {
					marker = ir.createMarker(IMarker.MARKER);
					marker.setAttribute(IMarker.LINE_NUMBER, lineNum);
				}
				catch (CoreException coreEx) {
					System.err.println( coreEx );
				}
			}
		}
		return marker;
	}
	
	/**
	 * Private method to add sizes, javaElements, names and tooltips to vectors.
	 */
	private void addToVectors (int size, IJavaElement javaElement){

		if (size > 0 && javaElement != null){
			Integer Isize = new Integer(size);
    	 	vector.addElement(Isize);
    	 	elements.addElement(javaElement);
    	 	names.addElement(javaElement.getElementName());
    	 	tooltips.addElement(javaElement.getPath().toString());
      	}
	}
	
	/**
	 * Private method to add sizes, resources, names and tooltips to vectors. 
	 */
	private void addToVectors (int size, IResource resource){

		if (size > 0 && resource != null){
			Integer Isize = new Integer(size);
    	 	vector.addElement(Isize);
    	 	elements.addElement(resource);
    	 	if (in_package_view){
    	 		try {
    	 			IJavaProject JP = JavaCore.create(resource.getProject());
    	 			IClasspathEntry[] classpathArray = JP.getRawClasspath();
    	 			IClasspathEntry classpath = classpathArray[0];
    	 			String classpathS = classpath.toString();
    	 			int index = classpathS.indexOf('[');
    	 			classpathS = new String(classpathS.substring(0,index));
    	 			String resourcePath = resource.getFullPath().toString();
  	 				String path;

					/* 
					 * This is a nasty hack because the classpath seems to be different
					 * for projects which have apsects in them.
					 */
    	 			if(classpathS.endsWith("/")){
    	 				path = resourcePath.substring(classpathS.length());
    	 			} else {
    	 				path = resourcePath.substring(classpathS.length()+1);
    	 			}
    	 			
    	 			String string = new String(path.replace('/','.'));
    	 			names.addElement(string);
    	 		} catch (JavaModelException e) {
    	 			System.err.println(e);
    	 		}
    	 	} else {
    	 		names.addElement(resource.getName());
    	 	}
    	 	tooltips.addElement(resource.getFullPath().toString());
      	}
	}

	/**
	 * Adds actions to the context menu.
	 */
	private void hookContextMenu () {

		MenuManager menuMgr = new MenuManager("#PopupMenu"); //$NON-NLS-1$
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow (IMenuManager manager) {
				AV.this.fillContextMenu(manager);
			}
  		});
	}
	
	/**
	 * Adds actions to the action bar.
	 */
	private void contributeToActionBars () {

		IActionBars bars = getViewSite().getActionBars();
		fillLocalPullDown(bars.getMenuManager());
		fillLocalToolBar(bars.getToolBarManager());
	}

	/**
	 * Adds actions to local pull down menu.
	 */
	private void fillLocalPullDown (IMenuManager manager) {

		manager.add(zoom_in);
		manager.add(new Separator());
		manager.add(zoom_out);
		manager.add(new Separator());
		manager.add(limit);
		manager.add(new Separator());
		manager.add(sub_select);
		manager.add(new Separator());
		manager.add(package_view);
		manager.add(new Separator());
		manager.add(class_view);
	}

	/**
	 * Called by hookContextMenu to add actions.
	 */
	private void fillContextMenu (IMenuManager manager) {

		manager.add(zoom_in);
		manager.add(new Separator());
		manager.add(zoom_out);
		manager.add(new Separator());
		manager.add(limit);
		manager.add(new Separator());
		manager.add(sub_select);
		manager.add(new Separator());		
		manager.add(package_view);
		manager.add(new Separator("Additions")); //$NON-NLS-1$
		manager.add(class_view);
	}
	
	/**
	 * Called by contricuteToActionBars to add actions to local tool bar.
	 */
	private void fillLocalToolBar (IToolBarManager manager) {

		manager.add(zoom_in);
		manager.add(zoom_out);
		manager.add(limit);
		manager.add(sub_select);
		manager.add(package_view);
		manager.add(class_view);
	}

	/**
	 * Private method called by createPartControl to create and initialise actions.
	 */
	private void makeActions () {

		zoom_in = new Action(){
			public void run (){
				zoom_out.setEnabled(true);
				boolean zoom =	aspectDrawing.zoomIn(canvas);
				setEnabled(zoom);
			}
		};
		zoom_in.setText(Messages.getString("Zoom_In_3")); //$NON-NLS-1$
		zoom_in.setToolTipText(Messages.getString("Zooms_in_on_visualisation_4")); //$NON-NLS-1$
		ImageDescriptor Zoom_inID;
		Zoom_inID = ImageDescriptor.createFromFile(this.getClass(), Messages.getString("zoomIn.gif_5")); //$NON-NLS-1$
		zoom_in.setImageDescriptor(Zoom_inID);
		
		zoom_out = new Action(){
			public void run (){
				boolean zoom = aspectDrawing.zoomOut(canvas);
				setEnabled(zoom);
				zoom_in.setEnabled(true);
			}
		};
		zoom_out.setText(Messages.getString("Zoom_Out_6")); //$NON-NLS-1$
		zoom_out.setToolTipText(Messages.getString("Zooms_out_7")); //$NON-NLS-1$
		ImageDescriptor Zoom_outID;
		Zoom_outID = ImageDescriptor.createFromFile(this.getClass(), Messages.getString("zoomOut.gif_8")); //$NON-NLS-1$
		zoom_out.setImageDescriptor(Zoom_outID);
		
		limit = new Action() {
			public int getStyle () { 
				return IAction.AS_CHECK_BOX; 
			}
			public void run () {
				if(!limit_mode){
					limit_mode = true;
					aspectDrawing.Limit(canvas, limit_mode);
					Vector newVector = aspectDrawing.lengths;
					Vector newElements = new Vector();
					int index = 0;
					for(int i = 0; i < vector.size(); i++){
						if(index < newVector.size()){
							if (vector.elementAt(i).equals(newVector.elementAt(index))){
								newElements.add(elements.elementAt(i));
								index++;
							}
						}
					}
					elements = newElements;
					vector = newVector;
					names = aspectDrawing.names;
					tooltips = aspectDrawing.tooltips;
				} else {
					limit_mode = false;
					aspectDrawing.Limit(canvas, limit_mode);
				}
			}
		};
		limit.setText(Messages.getString("Limit_view_9")); //$NON-NLS-1$
		limit.setToolTipText(Messages.getString("Limits_visualisation_to_affected_bars_only_10")); //$NON-NLS-1$
		ImageDescriptor LimitID;
		LimitID = ImageDescriptor.createFromFile(this.getClass(), "graph_mode.gif"); //$NON-NLS-1$
		limit.setImageDescriptor(LimitID);
		
		package_view = new Action() {
			public int getStyle () { 
				return IAction.AS_CHECK_BOX; 
			}
			public void run () {
				if (in_sub_select_mode){
					if (!(in_package_view)){
						setChecked(true);
						class_view.setChecked(false);
						in_package_view = true;
						list = plugin.getSelectedPackages(elements);
						vector.removeAllElements();
						elements.removeAllElements();
						aspects.removeAllElements();
						names.removeAllElements();
						tooltips.removeAllElements();
						map = null;

						map = new Map[list.size()];
						for (int i = 0; i < list.size(); i++){
							Object[] info = new Object[3];
      						info = (Object[]) list.get(i);
      						addToVectors(((Integer)info[1]).intValue(), (IResource)info[0]);
      						map[i] = (Map)info[2];
						}
						IJavaProject jp = JavaCore.create(AspectJPlugin.getDefault().getCurrentProject());
						aspectStuff(jp);
					} else setChecked(true);
				} else
				setChecked(true);
				class_view.setChecked(false);
				in_package_view = true;
				selectionChanged(part, sel);
			}
		};
		package_view.setText(Messages.getString("Package_View_12")); //$NON-NLS-1$
		package_view.setToolTipText(Messages.getString("Changes_to_package_view_13")); //$NON-NLS-1$
		ImageDescriptor PackageID;
		PackageID = ImageDescriptor.createFromFile(this.getClass(), "packages.gif"); //$NON-NLS-1$
		package_view.setImageDescriptor(PackageID);
		
		class_view = new Action() {
			public int getStyle () { 
				return IAction.AS_CHECK_BOX; 
			}
			public void run () {
				if (in_sub_select_mode){
					if (in_package_view){
						list = plugin.getSelectedClasses(elements);
						vector.removeAllElements();
						elements.removeAllElements();
						aspects.removeAllElements();
						names.removeAllElements();
						tooltips.removeAllElements();

						map = new Map[list.size()];
						for (int i = 0; i < list.size(); i++){
							Object[] info = new Object[3];
      						info = (Object[]) list.get(i);
      						addToVectors(((Integer)info[1]).intValue(), (IResource)info[0]);
      						map[i] = (Map)info[2];
						}
						IJavaProject jp = JavaCore.create(AspectJPlugin.getDefault().getCurrentProject());
						aspectStuff(jp);
					} 
				}
				setChecked(true);
				package_view.setChecked(false);
				in_package_view = false;
				selectionChanged(part, sel);
			}
		};
		class_view.setText(Messages.getString("Class_View_15")); //$NON-NLS-1$
		class_view.setToolTipText(Messages.getString("Changes_to_class_view_16")); //$NON-NLS-1$
		ImageDescriptor ClassID;
		ClassID = ImageDescriptor.createFromFile(this.getClass(), "class_obj.gif"); //$NON-NLS-1$
		class_view.setImageDescriptor(ClassID);
		
		sub_select = new Action() {
			public int getStyle () { 
				return IAction.AS_CHECK_BOX; 
			}
			public void run () {
				boolean sub = aspectDrawing.subSelect(canvas);
				
				sub_select.setChecked(sub);
				in_sub_select_mode = sub;
				if (!sub){					
					selectionChanged(null, getSite().getPage().getSelection());
				} else {
					Vector newVector = aspectDrawing.lengths;
					Vector newElements = new Vector();
					int index = 0;
					for (int i = 0; i < vector.size(); i++){
						if (index < newVector.size()){
							if (vector.elementAt(i).equals(newVector.elementAt(index))){
								newElements.add(elements.elementAt(i));
								index++;
							}
						}
					}
					elements = newElements;
					vector = newVector;
					names = aspectDrawing.names;
					tooltips = aspectDrawing.tooltips;
				}
			}
		};
		sub_select.setText(Messages.getString("Subselect_18")); //$NON-NLS-1$
		sub_select.setToolTipText(Messages.getString("Subselects_chosen_packages/classes_19")); //$NON-NLS-1$
		ImageDescriptor SubselectID;
		SubselectID = ImageDescriptor.createFromFile(this.getClass(), "next_nav.gif"); //$NON-NLS-1$
		sub_select.setImageDescriptor(SubselectID);

	}	
	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus () {
	}
	
	/**
	 * Dispose of the Visualiser view when closed.
	 */
	public void dispose () {
		
		getSite().getPage().removeSelectionListener(this);
		aspectDrawing.removeListener(canvasListener);
		plugin.removeVisualiser();
	}

	/**
	 * @param p
	 */
	public void refresh(IJavaProject p) {	
		updateDisplay();
	}

}