/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.ajdt.buildconfigurator.tests;

import java.io.ByteArrayInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import junit.framework.TestCase;

import org.eclipse.ajdt.buildconfigurator.BuildConfiguration;
import org.eclipse.ajdt.buildconfigurator.BuildConfigurator;
import org.eclipse.ajdt.buildconfigurator.ProjectBuildConfigurator;
import org.eclipse.ajdt.internal.core.AJDTUtils;
import org.eclipse.ajdt.internal.ui.preferences.AspectJPreferences;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;

public class BuildConfiguratorTest extends TestCase {

	IProject javaProject = null;
	IProject ajProject = null;
	
	static int testNum = 0;
	
	IFile fileA;
	IFile fileB;
	IFile fileDef;
	
	protected void setUp() throws Exception {
		super.setUp();
		
		//create JavaProject
		TestProject tp = new TestProject("Java Project created by BuildConfiguratorTest " + testNum);
		javaProject = tp.getProject();
		
		TestProject tp2 = new TestProject("AJ Project created by BuildConfiguratorTest" + testNum++);
		ajProject = tp2.getProject();
		AspectJPreferences.setAJDTPrefConfigDone(true);
		AJDTUtils.addAspectJNature(ajProject);		
		
//		assertTrue( "There must be at least one AspectJ project in workspace for this test",
//				ajProject != null );
//		
//		assertTrue( "There must be at least one non-aj java project in workspace for this test",
//				ajProject != null );
	

		this.waitForJobsToComplete(ajProject);
		setupSandboxSourceFolder();
		waitForJobsToComplete(ajProject);
	}
	
	private void setupSandboxSourceFolder() throws Exception{
		IFolder src = ajProject.getFolder("testSrcPath");
		if (!src.exists()){
			src.create(true, true, null);
		}
		IJavaProject jp = JavaCore.create(ajProject);
		IClasspathEntry[] cpes = jp.getRawClasspath();
		IClasspathEntry[] newCpes = new IClasspathEntry[cpes.length + 1];
		
		boolean alreadyThere = false;
		for (int i=0; i<cpes.length; i++){
			newCpes[i] = cpes[i];
			if (cpes[i].getPath().equals(src.getFullPath()))
				alreadyThere = true;
		}
		if (!alreadyThere){
			newCpes[cpes.length] = JavaCore.newSourceEntry(src.getFullPath());
			jp.setRawClasspath(newCpes, null);
		}
		
		fileDef = src.getFile("InDefaultPack.java");
		if (!fileDef.exists()) {
			//fileDef.create(new StringBufferInputStream("public class InDefaultPack{}"), true, null);
			String content = "public class InDefaultPack{}";
			ByteArrayInputStream source = new ByteArrayInputStream(content.getBytes());
			fileDef.create(source,true,null);
		}
		IFolder pack = src.getFolder("package1");
		if (!pack.exists()){
			pack.create(true, true, null);
		}

		fileA = pack.getFile("A.java"); 
		if (!fileA.exists()) {
			//fileA.create(new StringBufferInputStream("package package1;\npublic class A{}"), true, null);
			String content = "package package1;\npublic class A{}";
			ByteArrayInputStream source = new ByteArrayInputStream(content.getBytes());
			fileA.create(source,true,null);
		}

		fileB = pack.getFile("B.java"); 
		if (!fileB.exists()) {
			//fileB.create(new StringBufferInputStream("package package1;\npublic class B{}"), true, null);
			String content = "package package1;\npublic class B{}";
			ByteArrayInputStream source = new ByteArrayInputStream(content.getBytes());
			fileB.create(source,true,null);
		}
	}
	
//	private void removeSandboxSourceFolder() throws Exception{
//		IJavaProject jp = JavaCore.create(ajProject);
//		IClasspathEntry[] cpes = jp.getRawClasspath();
//		IClasspathEntry[] newCpes = new IClasspathEntry[cpes.length - 1];
//		IFolder src = ajProject.getFolder("testSrcPath");
//		int k=0;
//		for (int i=0; i<cpes.length; i++){
//			if (cpes[i].getPath().equals(src.getFullPath())){
//				k=1;
//			} else {
//				newCpes[i-k] = cpes[i];
//			}
//		}
//		if (k==1){
//			jp.setRawClasspath(newCpes, null);
//		} else {
//			int x =0;
//		}
//
//		try {
//			src.delete(true, null);
//		} catch (CoreException e) {
//			//e.printStackTrace();
//		}		
//	}

	/*
	 * @see TestCase#tearDown()
	 */
	protected void tearDown() throws Exception {
		super.tearDown();
	}
	
	public void testGetBuildConfigurator(){
		//TODO: test could be improved using threads to do accesses simultanously
		BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
		BuildConfigurator conf2 = BuildConfigurator.getBuildConfigurator();
		if (conf != conf2){
			fail("Build Configurator not unique.");
		}
	}
	
	public void testGetProjectBuildConfigurator() throws CoreException {
			BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
			ProjectBuildConfigurator pbc;
			
			pbc = conf.getProjectBuildConfigurator(javaProject);
			if (pbc != null)
				fail("Could obtain a ProjectBuildConfigurator for non-aj project. This should not be possible.");
			
			ajProject.close(null);
			pbc = conf.getProjectBuildConfigurator(ajProject);
			if (pbc != null)
				fail("Could obtain a ProjectBuildConfigurator for closed project. This should not be possible.");
			
			ajProject.open(null);
			pbc = conf.getProjectBuildConfigurator(ajProject);
			if (pbc == null)
				fail("Could not get a ProjectBuildConfigurator for an aj project.");
			
			//test does not work. buildConfigurator gets not notified when selection changes...
//			PackageExplorerPart.getFromActivePerspective().selectAndReveal(ajProject);
//			ProjectBuildConfigurator pbc2 = conf.getActiveProjectBuildConfigurator();
//			if (pbc2 != pbc){
//					fail("getActiveProjectBuildConfigurator did not return the pbc of the selected project.");
//			}
	}
	
	public void testGetBuildConfiguration(){
		BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
		ProjectBuildConfigurator pbc;
		
		pbc = conf.getProjectBuildConfigurator(ajProject);
		
		BuildConfiguration bc = pbc.getActiveBuildConfiguration();
		if (bc == null)
			fail("Could not get active BuildConfiguration.");
		
		Collection bcs = new ArrayList(pbc.getBuildConfigurations());
		
		Iterator iter = bcs.iterator();
		while(iter.hasNext()){
			BuildConfiguration tempbc = (BuildConfiguration)iter.next();
			pbc.removeBuildConfiguration(tempbc);
		}
		
		bc = pbc.getActiveBuildConfiguration();
		
		//recover old configuration
		iter = bcs.iterator();
		boolean isIncluded = false;
		while(iter.hasNext()){
			BuildConfiguration tempbc = (BuildConfiguration)iter.next();
			pbc.addBuildConfiguration(tempbc);
			if (tempbc.getFile().equals(bc.getFile())){
				isIncluded = true; 
			}
		}
		if (!isIncluded)
			pbc.removeBuildConfiguration(bc);	

		if (bc == null)
			fail("No new build configuration was created when removing all old ones.");
	}
	
	public void testBuildConfigurationIsincluded(){
		BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
		ProjectBuildConfigurator pbc;
		
		pbc = conf.getProjectBuildConfigurator(ajProject);
		BuildConfiguration bc = pbc.getActiveBuildConfiguration();
		
		//test: are all new files included?
		waitForJobsToComplete(ajProject);
		if (!bc.isIncluded(fileA))
			fail("A.java was not included.");
		if (!bc.isIncluded(fileB))
			fail("B.java was not included.");
		if (!bc.isIncluded(fileDef))
			fail("InDefaultPackage.java was not included.");
		
	}
	
	public void testBuildConfigurationExclude() throws CoreException{
		BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
		ProjectBuildConfigurator pbc;
		
		pbc = conf.getProjectBuildConfigurator(ajProject);
		BuildConfiguration bc = pbc.getActiveBuildConfiguration();
		
		//prerequisite
		testBuildConfigurationIsincluded();
			
			waitForJobsToComplete(ajProject);
			List l = new ArrayList(3);
			l.add(fileA);
			bc.excludeFiles(l);
			waitForJobsToComplete(ajProject);
			if (bc.isIncluded(fileA)){
				fail("Exclude failed. A.java still included.");
			}
			if (!bc.isIncluded(fileB))
				fail("Exclude failed. B.java should be included.");
			if (!bc.isIncluded(fileDef))
				fail("Exclude failed. InDefaultPackage.java should be included.");
			
			l.clear();
			l.add(fileA.getParent());
			bc.excludeFiles(l);
			waitForJobsToComplete(ajProject);
			if (bc.isIncluded(fileA))
				fail("Exclude failed. A.java still included.");
			if (bc.isIncluded(fileB))
				fail("Exclude failed. B.java still included.");
			if (!bc.isIncluded(fileDef))
				fail("Exclude failed. InDefaultPackage.java should be included.");
			
			
			bc.includeFiles(l);
			waitForJobsToComplete(ajProject);
			if (!bc.isIncluded(fileA))
				fail("Reinclude failed. A.java should be included.");
			if (!bc.isIncluded(fileB))
				fail("Reinclude failed. B.java should be included.");
			if (!bc.isIncluded(fileDef))
				fail("Reinclude failed. InDefaultPackage.java should be included.");
		
			l.clear();
			l.add(fileA);
			l.add(fileB);
			bc.excludeFiles(l);
			waitForJobsToComplete(ajProject);
			if (bc.isIncluded(fileA))
				fail("Exclude failed. A.java still included.");
			if (bc.isIncluded(fileB))
				fail("Exclude failed. B.java still included.");
			if (!bc.isIncluded(fileDef))
				fail("Exclude failed. InDefaultPackage.java should be included.");
			
	}
	
	public void testBuildConfigurationInclude() throws CoreException{
		BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
		ProjectBuildConfigurator pbc;
		
		pbc = conf.getProjectBuildConfigurator(ajProject);
		BuildConfiguration bc = pbc.getActiveBuildConfiguration();
			
			waitForJobsToComplete(ajProject);
			List l = new ArrayList(3);
			l.add(fileA.getParent());
			l.add(fileDef);
			bc.excludeFiles(l);
			waitForJobsToComplete(ajProject);
			if (bc.isIncluded(fileA))
				fail("Exclude failed. A.java should be excluded.");
			if (bc.isIncluded(fileB))
				fail("Exclude failed. B.java should be excluded.");
			if (bc.isIncluded(fileDef))
				fail("Exclude failed. InDefaultPackage.java should be excluded.");
			
			l.clear();
			l.add(fileDef);
			bc.includeFiles(l);
			waitForJobsToComplete(ajProject);
			if (bc.isIncluded(fileA))
				fail("Include failed. A.java should be excluded.");
			if (bc.isIncluded(fileB))
				fail("Include failed. B.java should be excluded.");
			if (!bc.isIncluded(fileDef))
				fail("Include failed. InDefaultPackage.java should be included.");
			

			l.clear();
			l.add(fileA.getParent());
			bc.includeFiles(l);
			waitForJobsToComplete(ajProject);
			if (!bc.isIncluded(fileA))
				fail("Include failed. A.java is not included.");
			if (!bc.isIncluded(fileB))
				fail("Include failed. B.java is not included.");
			if (!bc.isIncluded(fileDef))
				fail("Include failed. InDefaultPackage.java is not included.");
				
	}
	
	public void testNatureConversion() throws CoreException{
			BuildConfigurator conf = BuildConfigurator.getBuildConfigurator();
			ProjectBuildConfigurator pbc;
			AJDTUtils.removeAspectJNature(ajProject);
			this.waitForJobsToComplete(ajProject);
			
			pbc = conf.getProjectBuildConfigurator(ajProject);
			if (pbc != null){
				AJDTUtils.addAspectJNature(ajProject);
				fail ("could obtain pbc despite of removed aj nature");
			}
			
			IResource[] mems = ajProject.members();
			for (int i=0; i<mems.length; i++){
				if (mems[i].getName().endsWith(BuildConfiguration.EXTENSION))
					mems[i].delete(true, null);
			}
			
			IJavaProject jp= JavaCore.create(ajProject);
			IClasspathEntry[] cpes = jp.getRawClasspath();
			for (int i=0; i<cpes.length; i++){
				if (cpes[i].getPath().equals(fileDef.getParent().getFullPath())){
					IPath[] expats = new IPath[1];
					expats[0] = fileA.getFullPath().removeFirstSegments(cpes[i].getPath().matchingFirstSegments(fileA.getFullPath()));
					cpes[i] = JavaCore.newSourceEntry(cpes[i].getPath(), expats);
				}
			}
			jp.setRawClasspath(cpes, null);
			
			AJDTUtils.addAspectJNature(ajProject);
			
			this.waitForJobsToComplete(ajProject);
			
			pbc = conf.getProjectBuildConfigurator(ajProject);
			if (pbc == null){
				fail("No ProjectBuildConfigurator was created after adding aj nature.");
			}
			
			BuildConfiguration bc = pbc.getActiveBuildConfiguration();
			if (bc == null){
				fail("No active build configuration was created when added aj nature to project.");
			}
			
			
			this.waitForJobsToComplete(ajProject);
			//this cannot be tested because the file writing thread may still not have written its file
			if (bc.isIncluded(fileA)){
				fail("jdt exclusion filter not taken into account when converted to aj project.");
			}
			
			if (!bc.isIncluded(fileDef) || !bc.isIncluded(fileB)){
				fail("not all desired files included after conversion to aj project.");
			}
			
			if (!bc.getFile().exists()){
				fail("New build configuration was created when adding aj nature, but file not written.");
			}
			
			jp.getRawClasspath();
			cpes = jp.getRawClasspath();
			for (int i=0; i<cpes.length; i++){
				if (cpes[i].getPath().equals(fileDef.getParent().getFullPath())){
					if (cpes[i].getExclusionPatterns().length > 0)
						fail("Exclusion patterns not reset when converting to aj project.");
				}
			}
		
	}
	
	private void waitForJobsToComplete(IProject pro){
		Job job = new Job("Dummy Job"){
			public IStatus run(IProgressMonitor m){
				return Status.OK_STATUS;
			}
		};
		job.setPriority(Job.DECORATE);
		job.setRule(pro);
	    job.schedule();
	    try {
			job.join();
		} catch (InterruptedException e) {
			// Do nothing
		}
	}
}

