/**********************************************************************
Copyright (c) 2002 IBM Corporation and others.
All rights reserved. This program and the accompanying materials
are made available under the terms of the Common Public License v1.0
which accompanies this distribution, and is available at
http://www.eclipse.org/legal/cpl-v10.html
Contributors:
Adrian Colyer, Andy Clement, Tracy Gardner - initial version
...
**********************************************************************/
package org.eclipse.ajdt.ui.tests;


import junit.framework.TestCase;

import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.osgi.framework.Bundle;

public class AspectJPluginTest extends TestCase {

	public AspectJPluginTest(String name) {
		super(name);
	}

	/**
	 * @see TestCase#setUp()
	 */
	protected void setUp() throws Exception {
		super.setUp();
	}

	// test for AspectJPlugin.getDefault( )
	public void testGetDefault( ) {
		AspectJPlugin p = AspectJPlugin.getDefault( );
		assertNotNull( "AspectJPlugin default must be set", p );
		AspectJPlugin p2 = AspectJPlugin.getDefault( );
		assertEquals( "AspectJPlugin must be a singleton", p, p2 );
			
	}
	
	/** Test that descriptor ids match constants in AJPlugin */
	// Replaced this with testBundle() for 3.0 compatibility
//	public void testDescriptor( ) {
//		IPluginDescriptor desc = 
//			Platform.getPluginRegistry().getPluginDescriptor( AspectJPlugin.PLUGIN_ID);
//		assertNotNull( "Plugin descriptor must be available", desc );
//	}
	
	/** Test that bundle is available */
	public void testBundle() {
	    Bundle bundle = AspectJPlugin.getDefault().getBundle();
	    assertNotNull("Plugin bundle must be available", bundle);
	}
	
	/** Test resource bundle */
	public void testResourceBundle( ) {
		String value = AspectJPlugin.getResourceString( "test.message" );	
		assertEquals( "Resource key lookup failed", value, "A test message" );
		String key = "missing.key";
		assertEquals( "Missing resource key lookup failed", key,
						AspectJPlugin.getResourceString( key ) ); 
	}
	
	/** Test get/set current project */
	public void testProjectBuildFile( ) {
		AspectJPlugin p = AspectJPlugin.getDefault();
		p.setCurrentProject( "Examples" );
		IProject project = p.getCurrentProject();
		assertEquals( "Project should be set to 'Examples'", "Examples", 
						project.getName() );
						
		p.setCurrentProject( "The unknown project" );
		
		// verify that a warning dialog was produced
		assertTrue( AllAJPluginTests.confirmWithUser(
					"Did you see a warning dialog for the unknown project?\n" + 
					"(look behind this dialog! ;-))") );	
		
		project = p.getCurrentProject();
		assertEquals( "Project should be set to 'Examples'", "Examples", 
						project.getName() );										
	}
	
	public void testMultiBuildFilesInSingleProject( ) {
		IProject[] projects = AspectJPlugin.getWorkspace().getRoot().getProjects();
		assertTrue( "Must be at least one project in workspace for this test",
		    		projects.length > 0 );
        IProject p = projects[0];
        assertEquals( "Project should be set to 'Examples'", "Examples", 
						p.getName() );
		
		IResource defaultConfig = findRequiredResource( p, ".generated.lst" );
		IResource customConfig1 = findRequiredResource( p, "custom1.lst" );
		IResource customConfig2 = findRequiredResource( p, "custom2.lst" );
		
		AspectJPlugin plugin = AspectJPlugin.getDefault();
		AspectJPlugin.setBuildConfigurationFile( p, defaultConfig );
		String config = AspectJPlugin.getBuildConfigurationFile( p );
		String workspaceLoc = AspectJPlugin.getWorkspace().getRoot().getLocation().toOSString();
		String defaultConfigLoc = workspaceLoc + defaultConfig.getFullPath().toOSString();
		assertEquals( "Project should be using default config",
						config, defaultConfigLoc );
						
		AspectJPlugin.setBuildConfigurationFile( p, customConfig1 );
		config = AspectJPlugin.getBuildConfigurationFile( p );
		assertEquals( "Project should be using custom1.lst", config,
						workspaceLoc + customConfig1.getFullPath().toOSString() );						
						
		AspectJPlugin.setBuildConfigurationFile( p, customConfig2 );
		config = AspectJPlugin.getBuildConfigurationFile( p );
		assertEquals( "Project should be using custom2.lst", config,
						workspaceLoc + customConfig2.getFullPath().toOSString() );						
						
	}


	public void testMultiBuildFilesInMultiProjects() {
		IProject[] projects = AspectJPlugin.getWorkspace().getRoot().getProjects();
		assertTrue( "Must be at least two projects in workspace for this test",
		    		projects.length >= 2 );
        IProject p1 = projects[0];
        assertEquals( "Project one should be set to 'Examples'", "Examples", 
						p1.getName() );
        IProject p2 = projects[1];
        assertEquals( "Project one should be set to 'More Examples'", "More Examples", 
						p2.getName() );
		
		IResource customConfig1 = findRequiredResource( p1, "custom1.lst" );
		IResource customConfig3 = findRequiredResource( p2, "custom3.lst" );

		AspectJPlugin plugin = AspectJPlugin.getDefault();
		AspectJPlugin.setBuildConfigurationFile( p1, customConfig1 );
		String config = AspectJPlugin.getBuildConfigurationFile( p1 );
		String workspaceLoc = AspectJPlugin.getWorkspace().getRoot().getLocation().toOSString();
		String configLoc = workspaceLoc + customConfig1.getFullPath().toOSString();
		assertEquals( "Project should be using custom config1",
						config, configLoc );

		AspectJPlugin.setBuildConfigurationFile( p2, customConfig3 );
		config = AspectJPlugin.getBuildConfigurationFile( p2 );
		configLoc = workspaceLoc + customConfig3.getFullPath().toOSString();
		assertEquals( "Project should be using custom config3",
						config, configLoc );

		config = AspectJPlugin.getBuildConfigurationFile( p1 );
		configLoc = workspaceLoc + customConfig1.getFullPath().toOSString();
		assertEquals( "Project should be using custom config1",
						config, configLoc );
		
	}
	
	private IResource findRequiredResource( IProject p, String fName ) {
		IResource retVal = null;
		IResource[] projectMembers = null;
        try {						
			projectMembers = p.members();
        } catch ( CoreException cEx ) {
        	fail( "Exception listing project contents: " + cEx.getMessage() );
        }
        
        for ( int i = 0; i < projectMembers.length; i++ ) {
			IResource resource = projectMembers[i];
			if ( resource.getName( ).endsWith( fName ) ) {
				retVal = resource;
				break;
			}
		}	
		
		if ( retVal == null ) {
			fail( "Missing required project resource: " + fName + 
				  "in project " + p.getName() );
		}
		
		return retVal;
	}
}
