/*********************************************************************
 Copyright (c) August 2002 IBM Corporation and others.
 All rights reserved.   This program and the accompanying materials
 are made available under the terms of the Common Public License v1.0
 which accompanies this distribution, and is available at
 http://www.eclipse.org/legal/cpl-v10.html 
 Contributors: 
 Rory Slaney - Majority of initial version 
 Sian Whiting - setVisualiser, removeVisualiser, setMenu, removeMenu in initial version.
 **********************************************************************/

package org.eclipse.ajdt.ui.visualiser;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.TreeSet;

import org.aspectj.ajde.Ajde;
import org.aspectj.asm.IProgramElement;
import org.eclipse.ajdt.internal.core.AJDTEventTrace;
import org.eclipse.ajdt.internal.ui.ajde.ProjectProperties;
import org.eclipse.ajdt.ui.AspectJPlugin;
import org.eclipse.ajdt.ui.visualiser.menu.Menu;
import org.eclipse.ajdt.ui.visualiser.views.AV;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;

/**
 * The main plugin class to be used in the desktop.
 * 
 * @author Rory Slaney, Sian Whiting
 */
public class AspectVisualiserPlugin {
	// ASC - Temporary fix, removed the extends line since the the visualiser
	// is not running as a standalone plugin.  This also means I had to invoke
	// the AspectVisualiserPlugin ctor directly and I've done that from
	// AspectJPlugin for now.
	//extends AbstractUIPlugin {

	//The shared instance.
	private static AspectVisualiserPlugin plugin;
	//Resource bundle.
	private ResourceBundle resourceBundle;
	

	public AV visualiser;
	public Menu menu;

	/**
	 * The constructor.
	 */
	public AspectVisualiserPlugin() {
		// See comment at top of this class about the visualiser not currently
		// being a standalone plugin.
		//super(descriptor);
		plugin = this;
		try {
			resourceBundle =
				ResourceBundle.getBundle(
					"aspectVisualiser.aspectVisualiserPluginResources");
		} catch (MissingResourceException x) {
			resourceBundle = null;
		}
	}

	/**
	 * Returns the shared instance.
	 */
	public static AspectVisualiserPlugin getDefault() {
		return plugin;
	}

	/**
	 * Returns the workspace instance.
	 */
	public static IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	/**
	 * Returns the string from the plugin's resource bundle,
	 * or 'key' if not found.
	 */
	public static String getResourceString(String key) {
		ResourceBundle bundle =
			AspectVisualiserPlugin.getDefault().getResourceBundle();
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the plugin's resource bundle,
	 */
	public ResourceBundle getResourceBundle() {
		return resourceBundle;
	}

	/**
	 * Refresh the view
	 * @param visualiser
	 */
	public void refreshView() {
		if (visualiser != null) {
			IJavaProject p = JavaCore.create(AspectJPlugin.getDefault().getCurrentProject());
			visualiser.refresh(p);
		}
	}

	/**
	 * Set and remove the Aspect Visualiser view
	 */
	public void setVisualiser(AV visualiser) {
		this.visualiser = visualiser;
	}

	public void removeVisualiser() {
		this.visualiser = null;
	}

	/**
	 * Set and remove the Aspect menu view
	 */
	public void setMenu(Menu menu) {
		this.menu = menu;
	}

	public void removeMenu() {
		this.menu = null;
	}

	//aspectvisualiser backend

	String lastLoadedConfigFile = null;
	IProject project;

	/**
	 * Iterates through all the packages in a project and returns a Set of IResource containing
	 * all the Aspects in a project that are currently active.
	 */
	public Set getAllAspects(IJavaProject JP) {

		project = JP.getProject();

		initialiseAJDE();

		List packages = StructureModelUtil.getPackagesInModel();

		Set aspects = new HashSet();

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] progNodes = (Object[]) it.next();
			
			IProgramElement packageNode = (IProgramElement) progNodes[0];//it.next();

			Set temp =
				StructureModelUtil.getAspectsAffectingPackage(packageNode);

			aspects.addAll(temp);
		}

		return changeSet(aspects);
	}

	/**
	 * This method returns a list of all the classes in an IJavaProject
	 * as a list of three element arrays.
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] is an IResource corresponding to a class
	 * 			Object[1] is the number of lines in the class
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getAllClasses(IJavaProject JP) {

		LinkedList returningClasses = new LinkedList();
		project = JP.getProject();

		initialiseAJDE();

		List unsortedPackages = StructureModelUtil.getPackagesInModel();

		List packages = sortArray(unsortedPackages);

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] o = (Object[])it.next();
					
			IProgramElement packageNode = (IProgramElement) o[0];

			List unsortedFiles =
				StructureModelUtil.getFilesInPackage(packageNode);

			List files = sortElements(unsortedFiles);

			Iterator it2 = files.iterator();
			while (it2.hasNext()) {
				IProgramElement progNode = (IProgramElement) it2.next();

				try {
					org.aspectj.bridge.ISourceLocation isl =
						progNode.getSourceLocation();

					String fullpath =
						isl.getSourceFile().getAbsolutePath();

					Map lineAdviceMap =
						StructureModelUtil.getLinesToAspectMap(fullpath);

					Object entry[] = new Object[3];

					IResource res =
						AspectJPlugin
							.getDefault()
							.getAjdtProjectProperties()
							.findResource(
							fullpath, project);

					int endLine = isl.getEndLine();

					entry[0] = res;

					entry[1] = new Integer(endLine);

					entry[2] = changeMap(lineAdviceMap, 0);

					returningClasses.add(entry);
				} catch (NumberFormatException e) {
					e.printStackTrace();
				}
			}
		}

		return returningClasses;
	}

	/**
	 * This method returns a list of all the classes in an IPackageFragment
	 * as a list of three element arrays.
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] is an IResource corresponding to a class
	 * 			Object[1] is the number of lines in the class
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getAllClasses(IPackageFragment PF) {
		IJavaProject JP = PF.getJavaProject();
		project = JP.getProject();

		initialiseAJDE();

		LinkedList returningClasses = new LinkedList();
		List packages = StructureModelUtil.getPackagesInModel();
		
		String pf_string = PF.toString();
		
		
		boolean defaultPackage = false;
		
		String test = "";
		if (pf_string.startsWith("<default>")) {
			defaultPackage = true;
		} else {
			int end = pf_string.indexOf("[");
			test = PF.toString().substring(0, end - 1);
			if(test.indexOf("(not open)") != -1) {
				test = test.substring(0, test.length() - 10).trim();
			}
		}

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] o = (Object[]) it.next();

			if ((defaultPackage && o[1].equals("<default>")) || test.equals(o[1])) {

				IProgramElement packageNode = (IProgramElement) o[0];

				List unsortedFiles =
					StructureModelUtil.getFilesInPackage(packageNode);

				List files = sortElements(unsortedFiles);

				for (int j = 0; j < files.size(); j++) {
					try {
						IProgramElement fileNode =
							(IProgramElement) files.get(j);

						org.aspectj.bridge.ISourceLocation isl =
							fileNode.getSourceLocation();

						String fullpath =
							isl.getSourceFile().getAbsolutePath();

						Map lineAdviceMap =
							StructureModelUtil.getLinesToAspectMap(fullpath);

						Object entry[] = new Object[3];

						IResource res =
							AspectJPlugin
								.getDefault()
								.getAjdtProjectProperties()
								.findResource(
								fullpath, project);

						int endLine = isl.getEndLine();

						entry[0] = res;

						entry[1] = new Integer(endLine);

						entry[2] = changeMap(lineAdviceMap, 0);

						returningClasses.add(entry);
					} catch (NumberFormatException e) {
						e.printStackTrace();
					}
				}

			}
		}

		return returningClasses;
	}

	/**
	 * This method returns a list of all the classes in an ICompilationUnit
	 * as a list of three element arrays.
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] is an IResource corresponding to a class
	 * 			Object[1] is the number of lines in the class
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getAllClasses(ICompilationUnit CU) {

		project = CU.getJavaProject().getProject();

		initialiseAJDE();

		LinkedList returningClasses = new LinkedList();
		List packages = StructureModelUtil.getPackagesInModel();

        String cu_parent = CU.getParent().toString();
        
		// If the CompilationUnit is in the default package then the cu_parent string will start:
		// [default] [in ......
        // This differs from other cases when the cu is in a real package because it will then start:
        // figures.gui [in ......
        // So, this next piece of code, which relies on the position of that first '[' will break ....
        
		int end;
		
		if (cu_parent.startsWith("<default>")) {
			// Jump over the first [ and find the index of the second one !
			end = cu_parent.substring(1).indexOf("[")+1;
		} else {
			end = cu_parent.indexOf("[");
		}
		
		String test = cu_parent.substring(0, end - 1);
		if(test.indexOf("(not open)") != -1) {
			test = test.substring(0, test.length() - 10).trim();
		}
		String path = CU.getPath().toString();

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] o = (Object[]) it.next();

			if (test.equals(o[1])) {

				IProgramElement packageNode = (IProgramElement) o[0];
				List files = StructureModelUtil.getFilesInPackage(packageNode);

				Iterator it2 = files.iterator();
				while (it2.hasNext()) {

					IProgramElement file = (IProgramElement) it2.next();
					
					org.aspectj.bridge.ISourceLocation isl =
						file.getSourceLocation();

					String testpath = isl.getSourceFile().getAbsolutePath();
					testpath = testpath.replace('\\','/');

					if (testpath.endsWith(path)) {
						try {
							String fullpath =
								isl.getSourceFile().getAbsolutePath();

							Map lineAdviceMap =
								StructureModelUtil.getLinesToAspectMap(
									fullpath);

							Object entry[] = new Object[3];

							IResource res =
								AspectJPlugin
									.getDefault()
									.getAjdtProjectProperties()
									.findResource(
									fullpath, project);

							int endLine = isl.getEndLine();

							entry[0] = res;
							entry[1] = new Integer(endLine);
							entry[2] = changeMap(lineAdviceMap, 0);

							returningClasses.add(entry);
						} catch (NumberFormatException e) {
							e.printStackTrace();
						}

					}
				}
			}
		}

		return returningClasses;
	}

	/**
	 * This method iterates through every class in a project and returns a list of packages.
	 * Line numbers are accumulated from class to class.
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] an IResource for the project
	 * 			Object[1] an Integer with the size of the package
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getAllPackages(IJavaProject JP) {

		project = JP.getProject();

		initialiseAJDE();

		List returningPackages = new LinkedList();
		List packages = StructureModelUtil.getPackagesInModel();

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] o = (Object[])it.next();
			
			IProgramElement progNode = (IProgramElement) o[0];

			int start = 0;
			Map map = new HashMap();

			List files = StructureModelUtil.getFilesInPackage(progNode);

			files = sortElements(files);

			String path = null;

			for (int j = 0; j < files.size(); j++) {
				IProgramElement file = (IProgramElement) files.get(j);

				org.aspectj.bridge.ISourceLocation isl =
					file.getSourceLocation();

				path = isl.getSourceFile().getAbsolutePath();

				Map fileMap = StructureModelUtil.getLinesToAspectMap(path);
				fileMap = changeMap(fileMap, start);
				map.putAll(fileMap);

				start = start + isl.getEndLine();
			}

			if (path != null) {

				Object[] packageObj = new Object[3];
				int last = path.lastIndexOf(java.io.File.separator);
				path = new String(path.substring(0, last));

				IResource resource =
					AspectJPlugin
						.getDefault()
						.getAjdtProjectProperties()
						.findResource(
						path, project);

				packageObj[0] = resource;
				packageObj[1] = new Integer(start);
				packageObj[2] = map;

				returningPackages.add(packageObj);
			}
		}
		return returningPackages;
	}

	/**
	 * This returns the package data for a IPackageFragment ie just one package
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] an IResource for the project
	 * 			Object[1] an Integer with the size of the package
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getAllPackages(IPackageFragment PF) {

		IJavaProject JP = PF.getJavaProject();
		project = JP.getProject();

		initialiseAJDE();

		List returningPackages = new LinkedList();

		List packages = StructureModelUtil.getPackagesInModel();

		int end = PF.toString().indexOf("[");
		String test = PF.toString().substring(0, end - 1);

		Iterator it = packages.iterator();
		while (it.hasNext()) {
			Object[] Obj = (Object[]) it.next();

			String testpackage = (String) Obj[1];

			if (test.equals(testpackage)) {

				IProgramElement progNode = (IProgramElement) Obj[0];

				int start = 0;
				Map map = new HashMap();

				List files = StructureModelUtil.getFilesInPackage(progNode);
				files = sortElements(files);

				String path = null;

				for (int j = 0; j < files.size(); j++) {
					IProgramElement file = (IProgramElement) files.get(j);

					org.aspectj.bridge.ISourceLocation isl =
						file.getSourceLocation();

					path = isl.getSourceFile().getAbsolutePath();

					Map fileMap = StructureModelUtil.getLinesToAspectMap(path);
					fileMap = changeMap(fileMap, start);
					map.putAll(fileMap);

					start = start + isl.getEndLine();
				}

				if (path != null) {

					Object[] packageObj = new Object[3];
					int last = path.lastIndexOf(java.io.File.separator);
					path = new String(path.substring(0, last));

					IResource resource =
						AspectJPlugin
							.getDefault()
							.getAjdtProjectProperties()
							.findResource(
							path, project);

					packageObj[0] = resource;
					packageObj[1] = new Integer(start);
					packageObj[2] = map;

					returningPackages.add(packageObj);
				}
			}
		}
		return returningPackages;
	}

	/**
	 * This method returns a list of classes belonging to the packages past in as a List.
	 * 
	 * @param List of IResources representing packages
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] an IResource for the project
	 * 			Object[1] an Integer with the size of the package
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getSelectedClasses(List packages) {

		ProjectProperties prop =
			AspectJPlugin.getDefault().getAjdtProjectProperties();

		List returningClasses = new LinkedList();

		Iterator it = packages.iterator();

		while (it.hasNext()) {
			try {
				IResource res = (IResource) it.next();

				IJavaProject JP = JavaCore.create(res.getProject());
				IClasspathEntry[] classpathArray = JP.getRawClasspath();
				IClasspathEntry classpath = classpathArray[0];

				String clazzpath = classpath.toString();
				int index = clazzpath.indexOf('[');
				clazzpath = new String(clazzpath.substring(0, index));

				String respath = res.getFullPath().toString();

				String path;

				//this is a nasty hack				
				if(clazzpath.endsWith("/")){
					path = respath.substring(clazzpath.length());					
				} else {
					path = respath.substring(clazzpath.length() + 1);					
				}

				String packagename = new String(path.replace('/', '.'));

				List modelpackages = StructureModelUtil.getPackagesInModel();

				Iterator it2 = modelpackages.iterator();
				while (it2.hasNext()) {
					Object[] o = (Object[]) it2.next();

					String testname = (String) o[1];

					if (testname.equals(packagename)) {

						IProgramElement packageNode =
							(IProgramElement) o[0];

						List unsortedFiles =
							StructureModelUtil.getFilesInPackage(packageNode);

						List files = sortElements(unsortedFiles);

						for (int j = 0; j < files.size(); j++) {
							try {
								IProgramElement fileNode =
									(IProgramElement) files.get(j);

								org.aspectj.bridge.ISourceLocation isl =
									fileNode.getSourceLocation();

								String fullpath =
									isl.getSourceFile().getAbsolutePath();

								Map lineAdviceMap =
									StructureModelUtil.getLinesToAspectMap(
										fullpath);

								Object entry[] = new Object[3];

								IResource resource =
									AspectJPlugin
										.getDefault()
										.getAjdtProjectProperties()
										.findResource(
										fullpath);

								int endLine = isl.getEndLine();

								entry[0] = resource;

								entry[1] = new Integer(endLine);

								entry[2] = changeMap(lineAdviceMap, 0);

								returningClasses.add(entry);
							} catch (NumberFormatException e) {
								e.printStackTrace();
							}
						}

						break;
					}
				}
			} catch (JavaModelException e) {
				e.printStackTrace();
			}
		}

		return returningClasses;
	}

	/**
	 * This method returns a List of packages which contain the selected classes.
	 * 
	 * @param List of IResources corresponding to selected packages
	 * 
	 * @return Object[] of length 3
	 * 			Object[0] an IResource for the project
	 * 			Object[1] an Integer with the size of the package
	 * 			Object[2] is a map from Integers to a List of Strings representing 
	 * 						the names of the aspects affecting the line
	 */
	public List getSelectedPackages(List classes) {

		List returningPackages = new LinkedList();

		Set temp = new TreeSet();

		Iterator it = classes.iterator();
		while (it.hasNext()) {
			try {
				IResource res = (IResource) it.next();

				IJavaProject JP = JavaCore.create(res.getProject());
				IClasspathEntry[] classpathArray = JP.getRawClasspath();
				IClasspathEntry classpath = classpathArray[0];

				String clazzpath = classpath.toString();
				int index = clazzpath.indexOf('[');
				clazzpath = new String(clazzpath.substring(0, index));

				IPath respath = res.getFullPath();
				respath = respath.removeLastSegments(1);

				String path;

				//this is a hack				
				if(clazzpath.endsWith("/")){
					path = respath.toString().substring(clazzpath.length());
				} else {
					path = respath.toString().substring(clazzpath.length() + 1);					
				}

				String packagename = new String(path.replace('/', '.'));

				temp.add(packagename);
			} catch (JavaModelException e) {
				e.printStackTrace();
			}
		}

		List packagez = StructureModelUtil.getPackagesInModel();
		Iterator it2 = packagez.iterator();
		while (it2.hasNext()) {
			Object[] Obj = (Object[]) it2.next();

			String testpackage = (String) Obj[1];

			if (temp.contains(testpackage)) {
				IProgramElement progNode = (IProgramElement) Obj[0];

				int start = 0;
				Map map = new HashMap();

				List files = StructureModelUtil.getFilesInPackage(progNode);
				files = sortElements(files);

				String path = null;

				for (int j = 0; j < files.size(); j++) {
					IProgramElement file =
						(IProgramElement) files.get(j);
					
					org.aspectj.bridge.ISourceLocation isl =
						file.getSourceLocation();
					
					path = isl.getSourceFile().getAbsolutePath();

					Map fileMap = StructureModelUtil.getLinesToAspectMap(path);
					fileMap = changeMap(fileMap, start);
					map.putAll(fileMap);

					start = start + isl.getEndLine();
				}

				if (path != null) {

					Object[] packageObj = new Object[3];
					int last = path.lastIndexOf(java.io.File.separator);
					path = new String(path.substring(0, last));

					IResource resource =
						AspectJPlugin
							.getDefault()
							.getAjdtProjectProperties()
							.findResource(
							path, project);

					packageObj[0] = resource;
					packageObj[1] = new Integer(start);
					packageObj[2] = map;

					returningPackages.add(packageObj);
				}
			}
		}
		return returningPackages;
	}
	//private support methods

	/**
	 * This method changes a set of ProgramElementNodes into a Set of IResources
	 * 
	 * @return a Set of IResources
	 */
	private Set changeSet(Set oldset) {
		Set newset = new HashSet();

		Iterator it = oldset.iterator();

		
		while (it.hasNext()) {
			Object obj = it.next();
			
			List aspects = (List)obj;//(List) it.next();
		
			Iterator iterator2 = aspects.iterator();
			

		while (iterator2.hasNext()) {

			Object progNodes = iterator2.next();

			IProgramElement progNode = (IProgramElement) progNodes;

			String path = progNode.getSourceLocation().getSourceFile().getAbsolutePath();

			IResource resource =
				AspectJPlugin
					.getDefault()
					.getAjdtProjectProperties()
					.findResource(
					path, project);

			newset.add(resource);
		}
	}

	return newset;
}

/**
 * This method changes a Map from line numbers to a List of ProgramElementNodes
 * into a Map from line numbers to a List of Strings representing the names of
 * the aspects affecting that line number
 * 
 * @return a Map from Integers to a List of Strings
 */
private Map changeMap(Map oldmap, int startline) {
	HashMap newmap = new HashMap();

	Set keys = oldmap.keySet();

	Iterator it = keys.iterator();
	while (it.hasNext()) {
		Integer line = (Integer) it.next();

		Object o = oldmap.get(line);

		List oldaspects = (List)o;

		List newaspects = new LinkedList();

		Iterator iterator2 = oldaspects.iterator();
		while (iterator2.hasNext()) {

			IProgramElement theaspect =
				(IProgramElement) iterator2.next();

			String fullpath = theaspect.getSourceLocation().getSourceFile().getAbsolutePath();

			Path path = new Path(fullpath);
			String aspectname = path.removeFileExtension().lastSegment();

			newaspects.add(aspectname);
		}

		if (!newaspects.isEmpty()) {
			line = new Integer(line.intValue() + startline);

			newmap.put(line, newaspects);
		}
	}

	return newmap;
}

/**
 * This method sets the current project and initialises AJDE
 */
private void initialiseAJDE() {

	String configFile = AspectJPlugin.getBuildConfigurationFile(project);
	if (!configFile.equals(lastLoadedConfigFile)) {
		AJDTEventTrace.generalEvent("initialiseAJDE: switching configs - from:"+lastLoadedConfigFile+" to:"+configFile);
		Ajde.getDefault().getConfigurationManager().setActiveConfigFile(
			configFile);
		lastLoadedConfigFile = configFile;
	}
}

/**
 * Helper function sorts a list of resources into alphabetical order
 */
private List sortElements(List oldElements) {

	Object[] temp = oldElements.toArray();
	SortingComparator comparator = new SortingComparator();

	Arrays.sort(temp, comparator);

	List newResources = Arrays.asList(temp);

	return newResources;
}

/**
 * Helper function that sorts a List containing arrays of program elements.
 */
private List sortArray(List oldElements) {
	Object[] temp = oldElements.toArray();
	SortArrayComparator comparator = new SortArrayComparator();

	Arrays.sort(temp, comparator);

	List newElements = Arrays.asList(temp);

	return newElements;
}

//--InnerClass---//

/**
 * Compares two ProgramElementNodes by their String names.
 */
private class SortingComparator implements Comparator {
	public int compare(Object o1, Object o2) {
		IProgramElement p1 = (IProgramElement) o1;
		IProgramElement p2 = (IProgramElement) o2;

		String name1 = p1.getName();
		String name2 = p2.getName();

		return name1.compareTo(name2);
	}
}

//May need this to sort arrays of packages but they seem to be sorted allready

/**
 * Compares two arrays with the first element as a ProgramElementNode by their names.
 */
private class SortArrayComparator implements Comparator {

	public int compare(Object o1, Object o2) {
		Object[] array1 = (Object[]) o1;
		Object[] array2 = (Object[]) o2;

		IProgramElement p1 = (IProgramElement) array1[0];
		IProgramElement p2 = (IProgramElement) array2[0];

		String name1 = p1.getName();
		String name2 = p2.getName();

		return name1.compareTo(name2);
	}
}

}
