/**********************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: Sian January - initial version
 * ...
 **********************************************************************/

package org.eclipse.contribution.visualiser.markerimpl;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;

import org.eclipse.contribution.visualiser.core.Stripe;
import org.eclipse.contribution.visualiser.interfaces.IGroup;
import org.eclipse.contribution.visualiser.interfaces.IMarkupKind;
import org.eclipse.contribution.visualiser.interfaces.IMember;
import org.eclipse.contribution.visualiser.interfaces.simpleImpl.SimpleMarkupKind;
import org.eclipse.contribution.visualiser.interfaces.simpleImpl.SimpleMarkupProvider;
import org.eclipse.contribution.visualiser.utils.JDTUtils;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.internal.editors.text.EditorsPlugin;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.texteditor.AnnotationPreference;
import org.eclipse.ui.texteditor.AnnotationPreferenceLookup;
import org.eclipse.ui.texteditor.MarkerAnnotation;

/**
 * Markup provider that accompanies the ResourceContentProvider and shows any markers attached to the
 * files being displayed.
 */
public class MarkerMarkupProvider extends SimpleMarkupProvider {
	
	private static final String WARNING_SUFFIX = " (warning)";
	private static final String ERROR_SUFFIX = " (error)";
	Map namesToKinds = new HashMap();
	List images = new ArrayList();
	
	
	/**
	 * Update the set of markups stored by this provider.
	 * @param groups
	 */
	protected void updateMarkups(List groups) {
		namesToKinds = new HashMap();
		resetColours();
		markups = new Hashtable();
		markupKinds = new TreeSet();
		for (Iterator iter = groups.iterator(); iter.hasNext();) {
			IGroup group = (IGroup) iter.next();
			for (Iterator iter2 = group.getMembers().iterator(); iter2.hasNext();) {
				IMember member = (IMember)iter2.next();
				if (member instanceof ResourceMember) {
					IResource res = ((ResourceMember)member).getResource();
					try {
						IMarker[] markers = res.findMarkers(null, true, IResource.DEPTH_INFINITE);
						for (int i = 0; i < markers.length; i++) {
							IMarker marker = markers[i];
							Integer lineNum = (Integer)marker.getAttribute(IMarker.LINE_NUMBER);
							if (lineNum != null) {
								int lineNumber = lineNum.intValue();
								String name = getLabel(marker);
								if(name == null) {
									name = marker.getType();
								}
//								if (marker.isSubtypeOf(IMarker.PROBLEM)) {
//									Integer severity = (Integer)marker.getAttribute(IMarker.SEVERITY);
//									if( severity != null) {
//										if (severity.intValue() == IMarker.SEVERITY_ERROR) {
//											name = name + ERROR_SUFFIX;
//										} else if (severity.intValue() == IMarker.SEVERITY_WARNING) {
//											name = name + WARNING_SUFFIX;
//										}
//									} 
//								}
								IMarkupKind kind;
								if (namesToKinds.get(name) instanceof IMarkupKind) {
									kind = (IMarkupKind)namesToKinds.get(name);
								} else {
									Image image = getImage(marker);
									images.add(image);
									kind = new SimpleMarkupKind(name, image);
									namesToKinds.put(name, kind);
									markupKinds.add(kind);
									Color color = getColor(marker);
									if(color != null) {
										colourMemory.put(name, color);
									}
								}
								boolean stripeOnLineAlready = false;
								if(markups != null) {
									List stripes = (List) markups.get(member.getFullname());
									if (stripes != null) {
										for (Iterator iter3 = stripes.iterator(); iter3.hasNext();) {
											Stripe stripe = (Stripe) iter3.next();
											if(stripe.getOffset() == lineNumber) {
												List kindList = Arrays.asList(new Object[] {kind});
												stripe.addKinds(kindList);
												stripeOnLineAlready = true;
											}
										}
									}
								} 
								if (!stripeOnLineAlready) {
									Stripe stripe = new StripeWithMarker(kind, lineNumber, marker);
									addMarkup(member.getFullname(), stripe);
								}
							}
						}
					} catch (CoreException e) {
						e.printStackTrace();
					}
				}
			}
		}
	}
	
	
	/**
	 * Get the Image for a marker, or null if none is defined
	 * @param marker
	 * @return
	 */
	private Image getImage(IMarker marker) {
		IWorkbenchAdapter adapter= (IWorkbenchAdapter)marker.getAdapter(IWorkbenchAdapter.class);
		if (adapter != null) {
		     ImageDescriptor descriptor = adapter.getImageDescriptor(marker);
		     if (descriptor != null) {
		          return descriptor.createImage();
		     }
		}
		Annotation annotation = new MarkerAnnotation(marker);
		if (annotation != null) {
			AnnotationPreferenceLookup lookup = EditorsPlugin.getDefault().getAnnotationPreferenceLookup();
			AnnotationPreference preference = lookup.getAnnotationPreference(annotation);
			if(preference != null) {
				ImageDescriptor id = preference.getImageDescriptor();
				if (id != null) {
					return id.createImage();
				} else if (preference.getSymbolicImageName().equals("bookmark")) {
					return PlatformUI.getWorkbench().getSharedImages().getImageDescriptor(IDE.SharedImages.IMG_OBJS_BKMRK_TSK).createImage();
				}
			}
		}
		return null;
	}
	
	
	/**
	 * Get the label for a marker, or null if a label can not be found
	 * 
	 * @param marker
	 * @return
	 */
	private String getLabel(IMarker marker) {
//		IWorkbenchAdapter adapter= (IWorkbenchAdapter)marker.getAdapter(IWorkbenchAdapter.class);
//		if (adapter != null) {
//		     String label = adapter.getLabel(marker);
//		     if(label != null && !(label.trim().equals(""))) {
//		         return label;
//		     }		   
//		}
		Annotation annotation = new MarkerAnnotation(marker);
		if (annotation != null) {
			AnnotationPreferenceLookup lookup = EditorsPlugin.getDefault().getAnnotationPreferenceLookup();
			AnnotationPreference preference = lookup.getAnnotationPreference(annotation);
			if(preference != null) {
				String id = preference.getPreferenceLabel();
				if (id != null && !(id.trim().equals(""))) {
					return id;
				}
			}
		}
		return null;
	}
	

	/**
	 * Get the Color for a marker, or null if none is defined
	 * @param marker
	 * @return
	 */
	private Color getColor(IMarker marker) {
		Annotation annotation = new MarkerAnnotation(marker);
		if (annotation != null) {
			AnnotationPreferenceLookup lookup = EditorsPlugin.getDefault().getAnnotationPreferenceLookup();
			AnnotationPreference preference = lookup.getAnnotationPreference(annotation);
			if(preference != null) {
				RGB rgb = preference.getColorPreferenceValue();
				if (rgb != null) {
					return new Color(null, rgb);
				}
			}
		}
		return null;
	}
	
	
	/**
	 * Process a mouse click on a stripe.  This implementation opens the editor at the 
	 * location of the marker
	 * @see org.eclipse.contribution.visualiser.interfaces.IMarkupProvider#processMouseclick(IMember, Stripe, String, int)
	 */
	public boolean processMouseclick(IMember member, Stripe stripe, int buttonClicked) {
		if(stripe instanceof StripeWithMarker) {
			IMarker marker = ((StripeWithMarker)stripe).getMarker();
			if(marker != null) {
				JDTUtils.openInEditor(marker);
			}
		}
		
		return false;
	}


	public Color getColorFor(IMarkupKind kind) {
		if(colourMemory.containsKey(kind.getName())) {
			return (Color)colourMemory.get(kind.getName());
		} else {
			return super.getColorFor(kind);
		}
	}
	
	
	/**
	 * Deactivate this provider - dispose of system resources
	 */
	public void deactivate() {
		super.deactivate();
		for (Iterator iter = images.iterator(); iter.hasNext();) {
			Image element = (Image) iter.next();
			if(element != null && ! element.isDisposed()) {
				element.dispose();
			}
			iter.remove();
		}
		namesToKinds = new HashMap();
		markupKinds = new TreeSet();
	}


	
	public class StripeWithMarker extends Stripe {
		
		IMarker marker;
		
		/**
		 * Stripe constructor when the stripe is of the minimum depth (1).
		 * 
		 * @param k The kind of stripe
		 * @param i The offset down the bar where the stripe starts
		 */
		public StripeWithMarker(IMarkupKind k, int i, IMarker marker) {
			super(k,i);
			this.marker = marker;
		}
		
		public IMarker getMarker() {
			return marker;
		}
		
	}

}
