/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.aspectj.org.eclipse.jdt.core.dom;

import java.util.List;

/**
 * Boolean literal node.
 * 
 * <pre>
 * BooleanLiteral:
 * 		<b>true</b>
 * 		<b>false</b>
 * </pre>
 * 
 * @since 2.0
 */
public class BooleanLiteral extends Expression {
	
	/**
	 * The "booleanValue" structural property of this node type.
	 * @since 3.0
	 */
	public static final SimplePropertyDescriptor BOOLEAN_VALUE_PROPERTY = 
		new SimplePropertyDescriptor(BooleanLiteral.class, "booleanValue", boolean.class, MANDATORY); //$NON-NLS-1$
	
	/**
	 * A list of property descriptors (element type: 
	 * {@link StructuralPropertyDescriptor}),
	 * or null if uninitialized.
	 */
	private static final List PROPERTY_DESCRIPTORS;
	
	static {
		createPropertyList(BooleanLiteral.class);
		addProperty(BOOLEAN_VALUE_PROPERTY);
		PROPERTY_DESCRIPTORS = reapPropertyList();
	}

	/**
	 * Returns a list of structural property descriptors for this node type.
	 * Clients must not modify the result.
	 * 
	 * @param apiLevel the API level; one of the
	 * <code>AST.JLS&ast;</code> constants

	 * @return a list of property descriptors (element type: 
	 * {@link StructuralPropertyDescriptor})
	 * @since 3.0
	 */
	public static List propertyDescriptors(int apiLevel) {
		return PROPERTY_DESCRIPTORS;
	}
			
	/**
	 * The boolean; defaults to the literal for <code>false</code>.
	 */
	private boolean value = false;

	/**
	 * Creates a new unparented boolean literal node owned by the given AST.
	 * <p>
	 * N.B. This constructor is package-private.
	 * </p>
	 * 
	 * @param ast the AST that is to own this node
	 */
	BooleanLiteral(AST ast) {
		super(ast);
	}

	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	final List internalStructuralPropertiesForType(int apiLevel) {
		return propertyDescriptors(apiLevel);
	}
	
	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	final boolean internalGetSetBooleanProperty(SimplePropertyDescriptor property, boolean get, boolean newValue) {
		if (property == BOOLEAN_VALUE_PROPERTY) {
			if (get) {
				return booleanValue();
			} else {
				setBooleanValue(newValue);
				return false;
			}
		}
		// allow default implementation to flag the error
		return super.internalGetSetBooleanProperty(property, get, newValue);
	}
	
	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	final int getNodeType0() {
		return BOOLEAN_LITERAL;
	}

	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	ASTNode clone0(AST target) {
		BooleanLiteral result = new BooleanLiteral(target);
		result.setSourceRange(this.getStartPosition(), this.getLength());
		result.setBooleanValue(booleanValue());
		return result;
	}

	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	final boolean subtreeMatch0(ASTMatcher matcher, Object other) {
		// dispatch to correct overloaded match method
		return matcher.match(this, other);
	}

	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	void accept0(ASTVisitor visitor) {
		visitor.visit(this);
		visitor.endVisit(this);
	}
	
	/**
	 * Returns the boolean value of this boolean literal node.
	 * 
	 * @return <code>true</code> for the boolean literal spelled
	 *    <code>"true"</code>, and <code>false</code> for the boolean literal 
	 *    spelled <code>"false"</code>.
	 */ 
	public boolean booleanValue() {
		return this.value;
	}
		
	/**
	 * Sets the boolean value of this boolean literal node.
	 * 
	 * @param value <code>true</code> for the boolean literal spelled
	 *    <code>"true"</code>, and <code>false</code> for the boolean literal 
	 *    spelled <code>"false"</code>.
	 */ 
	public void setBooleanValue(boolean value) {
		preValueChange(BOOLEAN_VALUE_PROPERTY);
		this.value = value;
		postValueChange(BOOLEAN_VALUE_PROPERTY);
	}

	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	int memSize() {
		return BASE_NODE_SIZE + 1 * 4;
	}
	
	/* (omit javadoc for this method)
	 * Method declared on ASTNode.
	 */
	int treeSize() {
		return memSize();
	}
}

