/* *******************************************************************
 * Copyright (c) 2002 Palo Alto Research Center, Incorporated (PARC).
 * All rights reserved. 
 * This program and the accompanying materials are made available 
 * under the terms of the Common Public License v1.0 
 * which accompanies this distribution and is available at 
 * http://www.eclipse.org/legal/cpl-v10.html 
 *  
 * Contributors: 
 *     PARC     initial implementation 
 * ******************************************************************/


package org.aspectj.ajdt.internal.compiler.ast;

import java.lang.reflect.Modifier;

import org.aspectj.ajdt.internal.compiler.lookup.EclipseFactory;
import org.aspectj.ajdt.internal.core.builder.EclipseSourceContext;
import org.aspectj.org.eclipse.jdt.core.compiler.CharOperation;
import org.aspectj.org.eclipse.jdt.internal.compiler.ClassFile;
import org.aspectj.org.eclipse.jdt.internal.compiler.CompilationResult;
import org.aspectj.org.eclipse.jdt.internal.compiler.ast.Annotation;
import org.aspectj.org.eclipse.jdt.internal.compiler.ast.Argument;
import org.aspectj.org.eclipse.jdt.internal.compiler.ast.CompilationUnitDeclaration;
import org.aspectj.org.eclipse.jdt.internal.compiler.ast.TypeDeclaration;
import org.aspectj.org.eclipse.jdt.internal.compiler.ast.TypeReference;
import org.aspectj.org.eclipse.jdt.internal.compiler.lookup.ClassScope;
import org.aspectj.org.eclipse.jdt.internal.compiler.lookup.TagBits;
import org.aspectj.org.eclipse.jdt.internal.compiler.parser.Parser;
import org.aspectj.weaver.AjAttribute;
import org.aspectj.weaver.ResolvedPointcutDefinition;
import org.aspectj.weaver.patterns.Pointcut;

/**
 * pointcut [declaredModifiers] [declaredName]([arguments]): [pointcutDesignator];
 * 
 * <p>No method will actually be generated for this node but an attribute
 * will be added to the enclosing class.</p>
 * 
 * @author Jim Hugunin
 */
public class PointcutDeclaration extends AjMethodDeclaration {
	public static final char[] mangledPrefix = "ajc$pointcut$".toCharArray();
	
	public PointcutDesignator pointcutDesignator;
	private int declaredModifiers;
	private String declaredName;
	private boolean generateSyntheticPointcutMethod = false;
	//private boolean mangleSelector = true;
    
    private ResolvedPointcutDefinition resolvedPointcutDeclaration = null;

	public PointcutDeclaration(CompilationResult compilationResult) {
		super(compilationResult);
		this.returnType = TypeReference.baseTypeReference(T_void, 0);
	}

	private Pointcut getPointcut() {
		if (pointcutDesignator == null) {
			return Pointcut.makeMatchesNothing(Pointcut.RESOLVED);
		} else {
			return pointcutDesignator.getPointcut();
		}
	}
	
	
	public void parseStatements(
		Parser parser,
		CompilationUnitDeclaration unit) {
		// do nothing
	}

	public void postParse(TypeDeclaration typeDec) {
		if (arguments == null) arguments = new Argument[0];
		this.declaredModifiers = modifiers;
		this.declaredName = new String(selector);
		// amc - if we set mangle selector to false, then the generated bytecode has the 
		// pointcut method name that the user of an @Pointcut would expect.
		// But then we will unpack it again in the weaver which may cause redundant
		// error messages to be issued. This seems the better trade-off...
		//if (mangleSelector) {  
			selector = CharOperation.concat(mangledPrefix, '$', selector, '$',
				                                         Integer.toHexString(sourceStart).toCharArray());
		//}
				
		if (Modifier.isAbstract(this.declaredModifiers)) {
			if (!(typeDec instanceof AspectDeclaration)) {
				typeDec.scope.problemReporter().signalError(sourceStart, sourceEnd, 
						"The abstract pointcut " + new String(declaredName) +
						" can only be defined in an aspect");
				ignoreFurtherInvestigation = true;
				return;
			} else if (!Modifier.isAbstract(typeDec.modifiers)) {
				typeDec.scope.problemReporter().signalError(sourceStart, sourceEnd, 
						"The abstract pointcut " + new String(declaredName) +
						" can only be defined in an abstract aspect");

				ignoreFurtherInvestigation = true;
				return;
			}
		}
		
		if (pointcutDesignator != null) {
			pointcutDesignator.postParse(typeDec, this);
		}
	}
    
	
	/**
	 * Called from the AtAspectJVisitor to create the @Pointcut annotation
	 * (and corresponding method) for this pointcut
	 *
	 */
	public void addAtAspectJAnnotations() {
		Annotation pcutAnnotation = AtAspectJAnnotationFactory.createPointcutAnnotation(getPointcut().toString(),declarationSourceStart);;
		if (annotations == null) {
			annotations = new Annotation[] { pcutAnnotation };
		} else {
			Annotation[] old = annotations;
			annotations = new Annotation[old.length +1];
			System.arraycopy(old,0,annotations,0,old.length);
			annotations[old.length] = pcutAnnotation;
		}		
		generateSyntheticPointcutMethod = true;
	}
	
	// coming from an @Pointcut declaration
	public void setGenerateSyntheticPointcutMethod() {
		generateSyntheticPointcutMethod = true;
		//mangleSelector = false;
	}
	
    public void resolve(ClassScope upperScope) {
		// we attempted to resolve annotations below, but that was too early, so we do it again
		// now at the 'right' time.
		if (binding!= null) {
			binding.tagBits -= TagBits.AnnotationResolved;
			resolveAnnotations(scope, this.annotations, this.binding);
		}
        // for the rest of the resolution process, this method should do nothing, use the entry point below...
    }

    public void resolvePointcut(ClassScope upperScope) {
        super.resolve(upperScope);
    }


	public void resolveStatements() {
		if (isAbstract()) {
			this.modifiers |= AccSemicolonBody;
		}
		
		
		if (binding == null || ignoreFurtherInvestigation) return;

		if (Modifier.isAbstract(this.declaredModifiers)&& (pointcutDesignator != null)) {
			scope.problemReporter().signalError(sourceStart, sourceEnd, "abstract pointcut can't have body");
			ignoreFurtherInvestigation = true;
			return;
		}
		
		if (pointcutDesignator != null) {
			pointcutDesignator.finishResolveTypes(this, this.binding, arguments.length, 
					scope.enclosingSourceType());
		}
        
        //System.out.println("resolved: " + getPointcut() + ", " + getPointcut().state);
		makeResolvedPointcutDefinition();
        resolvedPointcutDeclaration.setPointcut(getPointcut());
		super.resolveStatements();
	}
	

	public ResolvedPointcutDefinition makeResolvedPointcutDefinition() {
        if (resolvedPointcutDeclaration != null) return resolvedPointcutDeclaration;
		//System.out.println("pc: " + getPointcut() + ", " + getPointcut().state);
		resolvedPointcutDeclaration = new ResolvedPointcutDefinition(
            EclipseFactory.fromBinding(this.binding.declaringClass), 
            declaredModifiers, 
            declaredName,
			EclipseFactory.fromBindings(this.binding.parameters),
			getPointcut()); //??? might want to use null 
			
		resolvedPointcutDeclaration.setPosition(sourceStart, sourceEnd);
		resolvedPointcutDeclaration.setSourceContext(new EclipseSourceContext(compilationResult));
		return resolvedPointcutDeclaration;
	}


	public AjAttribute makeAttribute() {
		return new AjAttribute.PointcutDeclarationAttribute(makeResolvedPointcutDefinition());
	}

	
	/**
	 * A pointcut declaration exists in a classfile only as an attibute on the
	 * class.  Unlike advice and inter-type declarations, it has no corresponding
	 * method.
	 */
	public void generateCode(ClassScope classScope, ClassFile classFile) {
		if (ignoreFurtherInvestigation) return ;
		classFile.extraAttributes.add(new EclipseAttributeAdapter(makeAttribute()));
		if (generateSyntheticPointcutMethod) {
			super.generateCode(classScope,classFile);
		}
		return;
	}
	
	protected int generateInfoAttributes(ClassFile classFile) {
		return super.generateInfoAttributes(classFile,true);
	}
	
	public StringBuffer printReturnType(int indent, StringBuffer output) {
		return output.append("pointcut");
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jdt.internal.compiler.ast.AbstractMethodDeclaration#printBody(int, java.lang.StringBuffer)
	 */
	public StringBuffer printBody(int indent, StringBuffer output) {
		output.append(": ");
		output.append(getPointcut());
		output.append(";");
		return output;
	}

}
