/* *******************************************************************
 * Copyright (c) 2002 Palo Alto Research Center, Incorporated (PARC).
 * All rights reserved. 
 * This program and the accompanying materials are made available 
 * under the terms of the Common Public License v1.0 
 * which accompanies this distribution and is available at 
 * http://www.eclipse.org/legal/cpl-v10.html 
 *  
 * Contributors: 
 *     PARC     initial implementation 
 * ******************************************************************/


package org.aspectj.weaver;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import org.aspectj.bridge.IMessage;
import org.aspectj.bridge.ISourceLocation;
import org.aspectj.bridge.Message;
import org.aspectj.bridge.MessageUtil;

public class Lint {
	/* private */ Map kinds = new HashMap();
	/* private */ World world;

	public final Kind invalidAbsoluteTypeName =
		new Kind("invalidAbsoluteTypeName", "no match for this type name: {0}");

	public final Kind invalidWildcardTypeName = 
		new Kind("invalidWildcardTypeName", "no match for this type pattern: {0}");
	
	public final Kind unresolvableMember = 
		new Kind("unresolvableMember", "can not resolve this member: {0}");
	
	public final Kind typeNotExposedToWeaver = 
		new Kind("typeNotExposedToWeaver", "this affected type is not exposed to the weaver: {0}");
		
	public final Kind shadowNotInStructure = 
		new Kind("shadowNotInStructure", "the shadow for this join point is not exposed in the structure model: {0}");

	public final Kind unmatchedSuperTypeInCall = 
		new Kind("unmatchedSuperTypeInCall", "does not match because declaring type is {0}, if match desired use target({1})");

	public final Kind canNotImplementLazyTjp = 
		new Kind("canNotImplementLazyTjp", "can not implement lazyTjp on this joinpoint {0} because around advice is used");

	public final Kind needsSerialVersionUIDField = 
		new Kind("needsSerialVersionUIDField", "serialVersionUID of type {0} needs to be set because of {1}");

	public final Kind serialVersionUIDBroken = 
		new Kind("brokeSerialVersionCompatibility", "serialVersionUID of type {0} is broken because of added field {1}");
		
	public final Kind noInterfaceCtorJoinpoint = 
		new Kind("noInterfaceCtorJoinpoint","no interface constructor-execution join point - use {0}+ for implementing classes");
        
	public final Kind noJoinpointsForBridgeMethods =
		new Kind("noJoinpointsForBridgeMethods","pointcut did not match on the method call to a bridge method.  Bridge methods are generated by the compiler and have no join points");
	
	public final Kind enumAsTargetForDecpIgnored =
		new Kind("enumAsTargetForDecpIgnored","enum type {0} matches a declare parents type pattern but is being ignored");

	public final Kind annotationAsTargetForDecpIgnored =
		new Kind("annotationAsTargetForDecpIgnored","annotation type {0} matches a declare parents type pattern but is being ignored");
	
	public final Kind cantMatchArrayTypeOnVarargs =
		new Kind("cantMatchArrayTypeOnVarargs","an array type as the last parameter in a signature does not match on the varargs declared method: {0}");
	
    public final Kind adviceDidNotMatch =
        new Kind("adviceDidNotMatch","advice defined in {0} has not been applied");
    
    public final Kind invalidTargetForAnnotation =
    	new Kind("invalidTargetForAnnotation","{0} is not a valid target for annotation {1}, this annotation can only be applied to {2}");

    public final Kind elementAlreadyAnnotated =
    	new Kind("elementAlreadyAnnotated","{0} - already has an annotation of type {1}, cannot add a second instance");
    	
	public final Kind runtimeExceptionNotSoftened = 
		new Kind("runtimeExceptionNotSoftened","{0} will not be softened as it is already a RuntimeException");
	
    public Lint(World world) {
		this.world = world;
	}
	
	
	public void setAll(String messageKind) {
		setAll(getMessageKind(messageKind));
	}
	
	private void setAll(IMessage.Kind messageKind) {
		for (Iterator i = kinds.values().iterator(); i.hasNext(); ) {
			Kind kind = (Kind)i.next();
			kind.setKind(messageKind);
		}
	}
	
	public void setFromProperties(File file) {
		try {
			InputStream s = new FileInputStream(file);
			setFromProperties(s);
		} catch (IOException ioe) {
			MessageUtil.error(world.getMessageHandler(),
					WeaverMessages.format(WeaverMessages.XLINT_LOAD_ERROR,file.getPath(),ioe.getMessage()));
		}
	}

	public void loadDefaultProperties() {
		InputStream s = getClass().getResourceAsStream("XlintDefault.properties");
		if (s == null) {
			MessageUtil.warn(world.getMessageHandler(), 
					WeaverMessages.format(WeaverMessages.XLINTDEFAULT_LOAD_ERROR));
			return;
		}
		try {
			setFromProperties(s);
		} catch (IOException ioe) {
			MessageUtil.error(world.getMessageHandler(),
					WeaverMessages.format(WeaverMessages.XLINTDEFAULT_LOAD_PROBLEM,ioe.getMessage()));
		}

	}


	private void setFromProperties(InputStream s) throws IOException {
		Properties p = new Properties();
		p.load(s);
		setFromProperties(p);
	}
	
	
	public void setFromProperties(Properties properties) {
		for (Iterator i = properties.entrySet().iterator(); i.hasNext(); ) {
			Map.Entry entry = (Map.Entry)i.next();
			Kind kind = (Kind)kinds.get(entry.getKey());
			if (kind == null) {
				MessageUtil.error(world.getMessageHandler(),
						WeaverMessages.format(WeaverMessages.XLINT_KEY_ERROR,entry.getKey()));
			} else {
				kind.setKind(getMessageKind((String)entry.getValue()));
			}
		}
	}

	private IMessage.Kind getMessageKind(String v) {
		if (v.equals("ignore")) return null;
		else if (v.equals("warning")) return IMessage.WARNING;
		else if (v.equals("error")) return IMessage.ERROR;
		
		MessageUtil.error(world.getMessageHandler(), 
				WeaverMessages.format(WeaverMessages.XLINT_VALUE_ERROR,v));
		return null;
	}
	
	
	
	public class Kind {
		private String name;
		private String message;
		private IMessage.Kind kind = IMessage.WARNING;
		public Kind(String name, String message) {
			this.name = name;
			this.message = message;
			kinds.put(this.name, this);
		}
		
		public boolean isEnabled() {
			return kind != null;
		}
		
		public String getName() {
			return name;
		}
		
		public IMessage.Kind getKind() {
			return kind;
		}

		public void setKind(IMessage.Kind kind) {
			this.kind = kind;
		}
		
		public void signal(String info, ISourceLocation location) {
			if (kind == null) return;
			
			String text = MessageFormat.format(message, new Object[] {info} );
			text += " [Xlint:" + name + "]";
			world.getMessageHandler().handleMessage(new Message(text, kind, null, location));
		}

		public void signal(String[] infos, ISourceLocation location, ISourceLocation[] extraLocations) {
            if (kind == null) return;
            
            String text = MessageFormat.format(message, infos );
            text += " [Xlint:" + name + "]";
            world.getMessageHandler().handleMessage(
                new Message(text, "", kind, location, null, extraLocations));
		}
	}
}
