/*
 * (c) Copyright QNX Software Systems Ltd. 2002.
 * All Rights Reserved.
 *
 */
package org.eclipse.cdt.debug.mi.core.cdi;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.cdt.debug.core.cdi.CDIException;
import org.eclipse.cdt.debug.core.cdi.ICDILocation;
import org.eclipse.cdt.debug.core.cdi.model.ICDIArgument;
import org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame;
import org.eclipse.cdt.debug.core.cdi.model.ICDIVariable;
import org.eclipse.cdt.debug.mi.core.MIException;
import org.eclipse.cdt.debug.mi.core.MISession;
import org.eclipse.cdt.debug.mi.core.command.CommandFactory;
import org.eclipse.cdt.debug.mi.core.command.MIStackListArguments;
import org.eclipse.cdt.debug.mi.core.command.MIStackListLocals;
import org.eclipse.cdt.debug.mi.core.output.MIArg;
import org.eclipse.cdt.debug.mi.core.output.MIFrame;
import org.eclipse.cdt.debug.mi.core.output.MIStackListArgumentsInfo;
import org.eclipse.cdt.debug.mi.core.output.MIStackListLocalsInfo;

/**
 */
public class StackFrame extends CObject implements ICDIStackFrame {

	MIFrame frame;
	CThread cthread;

	public StackFrame(CThread thread, MIFrame f) {
		super(thread.getCTarget());
		cthread = thread;
		frame = f;
	}

	MIFrame getMIFrame() {
		return frame;
	}

	CThread getCThread() {
		return cthread;
	}

	/**
	 * @see org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame#getArguments()
	 */
	public ICDIArgument[] getArguments() throws CDIException {
		List cdiList = new ArrayList();
		if (frame != null) {
			CSession session = getCTarget().getCSession();
			VariableManager mgr = (VariableManager)session.getVariableManager();
			MISession mi = session.getMISession();
			CommandFactory factory = mi.getCommandFactory();
			int level = frame.getLevel();
			MIStackListArguments listArgs =
				factory.createMIStackListArguments(false, level, level);
			try {
				cthread.setCurrentStackFrame(this);
				MIArg[] args = null;
				mi.postCommand(listArgs);
				MIStackListArgumentsInfo info =
					listArgs.getMIStackListArgumentsInfo();
				if (info == null) {
					throw new CDIException("No answer");
				}
				MIFrame[] miFrames = info.getMIFrames();
				if (miFrames != null && miFrames.length == 1) {
					args = miFrames[0].getArgs();
				}
				if (args != null) {
					for (int i = 0; i < args.length; i++) {
						try {
							cdiList.add(mgr.createArgument(this, args[i].getName()));
						} catch (CDIException e) {
						}
					}
				}
			} catch (MIException e) {
				//throw new CDIException(e.getMessage());
				//System.err.println(e);
			} catch (CDIException e) {
				//throw e;
				//System.err.println(e);
			}
		}
		return (ICDIArgument[])cdiList.toArray(new ICDIArgument[0]);
	}

	/**
	 * @see org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame#getLocalVariables()
	 */
	public ICDIVariable[] getLocalVariables() throws CDIException {
		List cdiList = new ArrayList();
		CSession session = getCTarget().getCSession();
		VariableManager mgr = (VariableManager)session.getVariableManager();
		MISession mi = session.getMISession();
		CommandFactory factory = mi.getCommandFactory();
		MIStackListLocals locals = factory.createMIStackListLocals(false);
		try {
			cthread.setCurrentStackFrame(this);
			MIArg[] args = null;
			mi.postCommand(locals);
			MIStackListLocalsInfo info = locals.getMIStackListLocalsInfo();
			if (info == null) {
				throw new CDIException("No answer");
			}
			args = info.getLocals();
			if (args != null) {
				for (int i = 0; i < args.length; i++) {
					try {
						cdiList.add(mgr.createVariable(this, args[i].getName()));
					} catch (CDIException e) {
					}
				}
			}
		} catch (MIException e) {
			//throw new CDIException(e.getMessage());
			//System.err.println(e);
		} catch (CDIException e) {
			//throw e;
			//System.err.println(e);
		}
		return (ICDIVariable[])cdiList.toArray(new ICDIVariable[0]);
	}

	/**
	 * @see org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame#getLocation()
	 */
	public ICDILocation getLocation() {
		if (frame != null) {
			return new Location(frame.getFile(), frame.getFunction(),
					frame.getLine(), frame.getAddress());
		}
		return new Location("", "", 0, 0);
	}

	/**
	 * @see org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame#getLevel()
	 */
	public int getLevel() {
		if (frame != null) {
			return frame.getLevel();
		}
		return 0;
	}
	/**
	 * @see org.eclipse.cdt.debug.core.cdi.model.ICDIStackFrame#equals(ICDIStackFrame)
	 */
	public boolean equals(ICDIStackFrame stackframe) {
		if (stackframe instanceof StackFrame) {
			StackFrame stack = (StackFrame)stackframe;
			return  cthread != null &&
				cthread.equals(stack.getCThread()) &&
				frame != null &&
				frame.getLevel() == stack.getMIFrame().getLevel() &&
				frame.getFile().equals(stack.getMIFrame().getFile()) &&
				frame.getFunction().equals(stack.getMIFrame().getFunction());
		}
		return super.equals(stackframe);
	}

}
