/**********************************************************************
 * Copyright (c) 2004 QNX Software Systems and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 * QNX Software Systems - Initial API and implementation
***********************************************************************/
package org.eclipse.cdt.debug.internal.core.model;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.LineNumberReader;

import org.eclipse.cdt.debug.core.cdi.model.ICDIInstruction;
import org.eclipse.cdt.debug.core.cdi.model.ICDIMixedInstruction;
import org.eclipse.cdt.debug.core.model.IAsmInstruction;
import org.eclipse.cdt.debug.core.model.IAsmSourceLine;
import org.eclipse.cdt.debug.core.model.ICStackFrame;
import org.eclipse.cdt.debug.core.model.IDisassembly;
import org.eclipse.cdt.debug.core.model.IDisassemblyBlock;
import org.eclipse.cdt.debug.core.model.IExecFileInfo;
import org.eclipse.cdt.debug.core.sourcelookup.ICSourceLocator;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.debug.core.model.ISourceLocator;

/**
 * CDI-based implementation of <code>IDisassemblyBlock</code>.
 */
public class DisassemblyBlock implements IDisassemblyBlock, IAdaptable {

	private IDisassembly fDisassembly;
	
	private IAsmSourceLine[] fSourceLines;

	private long fStartAddress = 0;

	private long fEndAddress = 0;
	
	private boolean fMixedMode = false;

	/**
	 * Constructor for DisassemblyBlock.
	 */
	private DisassemblyBlock( IDisassembly disassembly ) {
		fDisassembly = disassembly;
	}

	public static DisassemblyBlock create( IDisassembly disassembly, ICDIMixedInstruction[] instructions ) {
		DisassemblyBlock block = new DisassemblyBlock( disassembly );
		block.setMixedMode( true );
		ISourceLocator adapter = disassembly.getDebugTarget().getLaunch().getSourceLocator();
		ICSourceLocator locator = null;
		if ( adapter instanceof IAdaptable ) {
			locator = (ICSourceLocator)((IAdaptable)adapter).getAdapter( ICSourceLocator.class );
		}
		block.setSourceLines( createSourceLines( locator, instructions ) );
		block.initializeAddresses();
		return block;
	}

	public static DisassemblyBlock create( IDisassembly disassembly, ICDIInstruction[] instructions ) {
		DisassemblyBlock block = new DisassemblyBlock( disassembly );
		block.setMixedMode( false );
		block.setSourceLines( createSourceLines( instructions ) );
		block.initializeAddresses();
		return block;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cdt.debug.core.model.IDisassemblyBlock#getDisassembly()
	 */
	public IDisassembly getDisassembly() {
		return fDisassembly;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cdt.debug.core.model.IDisassemblyBlock#getModuleFile()
	 */
	public String getModuleFile() {
		IDisassembly d = getDisassembly();
		if ( d != null ) {
			IExecFileInfo info = (IExecFileInfo)d.getAdapter( IExecFileInfo.class );
			if ( info != null && info.getExecFile() != null ) {
				return info.getExecFile().getLocation().toOSString();
			}
		}
		return ""; //$NON-NLS-1$
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cdt.debug.core.model.IDisassemblyBlock#contains(org.eclipse.cdt.debug.core.model.ICStackFrame)
	 */
	public boolean contains( ICStackFrame frame ) {
		if ( !getDisassembly().getDebugTarget().equals( frame.getDebugTarget() ) )
			return false;
		long address = frame.getAddress();
		return (address >= fStartAddress && address <= fEndAddress);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cdt.debug.core.model.IDisassemblyBlock#getSourceLines()
	 */
	public IAsmSourceLine[] getSourceLines() {
		return fSourceLines;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter( Class adapter ) {
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.cdt.debug.core.model.IDisassemblyBlock#isMixedMode()
	 */
	public boolean isMixedMode() {
		return fMixedMode;
	}

	public void dispose() {
	}

	private static IAsmSourceLine[] createSourceLines( ICSourceLocator locator, ICDIMixedInstruction[] mi ) {
		IAsmSourceLine[] result = new IAsmSourceLine[mi.length];
		LineNumberReader reader = null;
		if ( result.length > 0 && locator != null ) {
			String fileName = mi[0].getFileName();
			Object element = locator.findSourceElement( fileName );
			File file= null;
			if ( element instanceof IFile ) {
				file = ((IFile)element).getLocation().toFile();
			}
			else if ( element instanceof IStorage ) {
				file = ((IStorage)element).getFullPath().toFile();
			}
			if ( file != null ) {
				try {
					reader = new LineNumberReader( new FileReader( file ) );
				}
				catch( FileNotFoundException e ) {
				}				
			}
		}
		for ( int i = 0; i < result.length; ++i ) {
			String text = null;
			if ( reader != null ) {
				int lineNumber = mi[i].getLineNumber();
				while( reader.getLineNumber() + 1 < lineNumber ) {
					try {
						reader.readLine();
					}
					catch( IOException e ) {
					}
				}
				if ( reader.getLineNumber() + 1 == lineNumber ) {
					try {
						text = reader.readLine() + '\n';
					}
					catch( IOException e ) {
					}
				}
			}
			result[i] = new AsmSourceLine( text, mi[i].getInstructions() );
		}
		return result;
	}

	private static IAsmSourceLine[] createSourceLines( ICDIInstruction[] instructions ) {
		return new IAsmSourceLine[] { new AsmSourceLine( "", instructions ) }; //$NON-NLS-1$
	}

	private void initializeAddresses() {
		if ( fSourceLines.length > 0 ) {
			IAsmInstruction[] instr = fSourceLines[0].getInstructions();
			fStartAddress = instr[0].getAdress();
			instr = fSourceLines[fSourceLines.length - 1].getInstructions();
			fEndAddress = instr[instr.length - 1].getAdress();
		}
	}

	private void setMixedMode( boolean mixedMode ) {
		this.fMixedMode = mixedMode;
	}

	private void setSourceLines( IAsmSourceLine[] sourceLines ) {
		this.fSourceLines = sourceLines;
	}
}
