/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.cdt.internal.core.search.indexing;

import java.io.IOException;
import java.util.HashSet;

import org.eclipse.cdt.internal.core.Util;
import org.eclipse.cdt.internal.core.index.IIndex;
import org.eclipse.cdt.internal.core.index.IQueryResult;
import org.eclipse.cdt.internal.core.index.impl.IFileDocument;
import org.eclipse.cdt.internal.core.search.SimpleLookupTable;
import org.eclipse.cdt.internal.core.search.processing.JobManager;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceProxy;
import org.eclipse.core.resources.IResourceProxyVisitor;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;


public class IndexAllProject extends IndexRequest {
	IProject project;

	public IndexAllProject(IProject project, IndexManager manager) {
		super(project.getFullPath(), manager);
		this.project = project;
	}
	
	public boolean equals(Object o) {
		if (o instanceof IndexAllProject)
			return this.project.equals(((IndexAllProject) o).project);
		return false;
	}
	/**
	 * Ensure consistency of a project index. Need to walk all nested resources,
	 * and discover resources which have either been changed, added or deleted
	 * since the index was produced.
	 */
	public boolean execute(IProgressMonitor progressMonitor) {

		if (progressMonitor != null && progressMonitor.isCanceled()) return true;
		if (!project.isAccessible()) return true; // nothing to do

		IIndex index = this.manager.getIndex(this.indexPath, true, /*reuse index file*/ true /*create if none*/);
		if (index == null) return true;
		ReadWriteMonitor monitor = this.manager.getMonitorFor(index);
		if (monitor == null) return true; // index got deleted since acquired

		try {
			monitor.enterRead(); // ask permission to read
			saveIfNecessary(index, monitor);

			IQueryResult[] results = index.queryInDocumentNames(""); // all file names //$NON-NLS-1$
			int max = results == null ? 0 : results.length;
			final SimpleLookupTable indexedFileNames = new SimpleLookupTable(max == 0 ? 33 : max + 11);
			final String OK = "OK"; //$NON-NLS-1$
			final String DELETED = "DELETED"; //$NON-NLS-1$
			for (int i = 0; i < max; i++)
				indexedFileNames.put(results[i].getPath(), DELETED);
			final long indexLastModified = max == 0 ? 0L : index.getIndexFile().lastModified();

//			CModel model = (CModel) CModelManager.getDefault().getCModel();
//			ICProject cProject = model.getCProject(project.getName());
//			ICElement[] kids = cProject.getChildren();
			IPath cProjectPath = project.getFullPath(); 
			
			IWorkspaceRoot root = this.project.getWorkspace().getRoot();
			IResource sourceFolder = root.findMember(cProjectPath);
			
			if (this.isCancelled) return false;

			if (sourceFolder != null) {
						
				// collect output locations if source is project (see http://bugs.eclipse.org/bugs/show_bug.cgi?id=32041)
				final HashSet outputs = new HashSet();
			
				final boolean hasOutputs = !outputs.isEmpty();
						
				final char[][] patterns = null;
				if (max == 0) {
					sourceFolder.accept(
							new IResourceProxyVisitor() {
								public boolean visit(IResourceProxy proxy) {
									if (isCancelled) return false;
									switch(proxy.getType()) {
										case IResource.FILE :
										//TODO: BOG Put the file name checking back
										//if (Util.isCCFileName(proxy.getName())) {
											IResource resource = proxy.requestResource();
											if (resource.getLocation() != null && (patterns == null || !Util.isExcluded(resource, patterns))) {
												String name = new IFileDocument((IFile) resource).getName();
												indexedFileNames.put(name, resource);
											}
										//}
										//	return false;
			
										case IResource.FOLDER :
												if (patterns != null && Util.isExcluded(proxy.requestResource(), patterns))
													return false;
												if (hasOutputs && outputs.contains(proxy.requestFullPath())) {
													return false;
												}
										}
										return true;
								 }
								},
								IResource.NONE
							);
						} else {
							sourceFolder.accept(
								new IResourceProxyVisitor() {
									public boolean visit(IResourceProxy proxy) {
										if (isCancelled) return false;
										switch(proxy.getType()) {
											case IResource.FILE :
//											TODO: BOG Put the file name checking back
												//if (Util.isCCFileName(proxy.getName())) {
													IResource resource = proxy.requestResource();
													IPath path = resource.getLocation();
													if (path != null && (patterns == null || !Util.isExcluded(resource, patterns))) {
														String name = new IFileDocument((IFile) resource).getName();
														indexedFileNames.put(name,
															indexedFileNames.get(name) == null || indexLastModified < path.toFile().lastModified()
																? (Object) resource
																: (Object) OK);
													}
												//}
												//return false;
											case IResource.FOLDER :
												if (patterns != null && Util.isExcluded(proxy.requestResource(), patterns))
													return false;
												if (hasOutputs && outputs.contains(proxy.requestFullPath())) {
													return false;
												}
										}
										return true;
									}
								},
								IResource.NONE
							);
						}
					}
					
			Object[] names = indexedFileNames.keyTable;
			Object[] values = indexedFileNames.valueTable;
			boolean shouldSave = false;
			for (int i = 0, length = names.length; i < length; i++) {
				String name = (String) names[i];
				if (name != null) {
					if (this.isCancelled) return false;

					Object value = values[i];
					if (value != OK) {
						shouldSave = true;
						if (value == DELETED)
							this.manager.remove(name, this.indexPath);
						else
							this.manager.addSource((IFile) value, this.indexPath);
					}
				}
			}

			// request to save index when all cus have been indexed
			if (shouldSave)
				this.manager.request(new SaveIndex(this.indexPath, this.manager));

		} catch (CoreException e) {
			if (IndexManager.VERBOSE) {
				JobManager.verbose("-> failed to index " + this.project + " because of the following exception:"); //$NON-NLS-1$ //$NON-NLS-2$
				e.printStackTrace();
			}
			this.manager.removeIndex(this.indexPath);
			return false;
		} catch (IOException e) {
			if (IndexManager.VERBOSE) {
				JobManager.verbose("-> failed to index " + this.project + " because of the following exception:"); //$NON-NLS-1$ //$NON-NLS-2$
				e.printStackTrace();
			}
			this.manager.removeIndex(this.indexPath);
			return false;
		} finally {
			monitor.exitRead(); // free read lock
		}
		return true;
	}
	
	public int hashCode() {
		return this.project.hashCode();
	}
	
	protected Integer updatedIndexState() {
		return IndexManager.REBUILDING_STATE;
	}
	
	public String toString() {
		return "indexing project " + this.project.getFullPath(); //$NON-NLS-1$
	}
}
