/**
 * <copyright> 
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.common.ui/src/org/eclipse/emf/common/ui/viewer/ExtendedTableTreeViewer.java, emf.common.ui, org.eclipse.dev, 20030620_1105VL
 * @version 1.5 6/20/03
 */
package org.eclipse.emf.common.ui.viewer;


import java.util.Iterator;
import java.util.LinkedList;

import org.eclipse.jface.viewers.TableTreeViewer;

import org.eclipse.swt.SWT;

import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.TableTreeItem;

import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;

import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Item;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Widget;


/**
 * This class extends a TableTreeViewer to draw images and tree lines in the tree column.
 */
public class ExtendedTableTreeViewer extends TableTreeViewer
{
  public static final String ITEM_ID = "TableTreeItemID"; 

  public ExtendedTableTreeViewer(TableTree tableTree)
  {
    super(tableTree);
  }

  public ExtendedTableTreeViewer(Composite parent)
  {
    super(parent);
  }

  public ExtendedTableTreeViewer(Composite parent, int style)
  {
    super(parent, style);
  }

  protected Item newItem(Widget parent, int flags, int index) 
  {
    TableTreeItem item = 
      index >= 0 ?
        parent instanceof TableTreeItem ?
          new ExtendedTableTreeItem((TableTreeItem) parent, flags, index) :
          new ExtendedTableTreeItem((TableTree) parent, flags, index) :
        parent instanceof TableTreeItem ?
          new ExtendedTableTreeItem((TableTreeItem) parent, flags) :
          new ExtendedTableTreeItem((TableTree) parent, flags);

    return item;
  }

  protected int offset;
  protected int indent;

  protected void hookControl(Control control)
  {
    super.hookControl(control);

/*
    getTableTree().getTable().addPaintListener
      (new PaintListener()
       {
         public void paintControl(PaintEvent event)
         {
           if (event.count > 0)
           {
             Thread.dumpStack();
           }
         }
       });
*/
    getTableTree().getTable().addPaintListener
      (new PaintListener()
       {
         protected boolean isStarted;
         protected TableTreeItem firstTableTreeItem;
         protected TableTreeItem lastTableTreeItem;
         protected LinkedList chain; 
         protected int scrollX;

         public void paintControl(PaintEvent event)
         {
           // System.out.println("Painting....." + event + " x=" + event.x + " y=" + event.y + " width=" + event.width + " height=" + event.height);
           // if (true) return;

           Table table = (Table)event.getSource();
           TableItem[] items = table.getItems();

           firstTableTreeItem = null;
           lastTableTreeItem = null;

           // Reset each time?
           //
           offset = 0;

           for (int i = table.getTopIndex(); i < items.length; i++)
           {
             TableItem tableItem = items[i];
             ExtendedTableTreeItem tableTreeItem = (ExtendedTableTreeItem)tableItem.getData(ITEM_ID);
             if (!tableTreeItem.isDisposed())
             {
               if (firstTableTreeItem == null)
               {
                 firstTableTreeItem = tableTreeItem;
               }
               lastTableTreeItem = tableTreeItem;
  
               if (offset == 0)
               {
                 Rectangle interactorBounds = tableItem.getImageBounds(0);
                 offset = interactorBounds.width;
                 indent = Math.min(6, (offset - 8) / 2);
               }
               Rectangle bounds = tableTreeItem.getBounds(0);
               if (bounds != null)
               {
                 Image image = tableTreeItem.getFirstImage();
                 if (image != null)
                 {
                   Rectangle imageBounds = image.getBounds();
                   imageBounds.x = bounds.x + offset;
                   imageBounds.y = bounds.y;
                   imageBounds.width = imagePaddingWidth;
                   event.gc.fillRectangle(imageBounds);
                   event.gc.drawImage(image, imageBounds.x + 5, imageBounds.y);
                   if (imageBounds.y + imageBounds.height > event.y + event.height)
                   {
                     break;
                   }
                 }
               }
             }
           }

           if (firstTableTreeItem != null)
           {
             isStarted = false;
             chain = new LinkedList();
             event.gc.setForeground(table.getDisplay().getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
             scrollX = items[0].getBounds(0).x;
             paintLines(event.gc, getTableTree().getItems());
           }
         }

         protected boolean paintLines(GC gc, TableTreeItem [] tableTreeItems)
         {
           if (tableTreeItems != null)
           {
             for (int i = 0; i < tableTreeItems.length; ++i)
             {
               TableTreeItem tableTreeItem = tableTreeItems[i];
               if (!isStarted && tableTreeItem == firstTableTreeItem)
               {
                 isStarted = true;
               }
               if (isStarted)
               {
                 Rectangle bounds = tableTreeItem.getBounds(0);
                 int x = 1 + scrollX;

                 for (Iterator j = chain.iterator(); j.hasNext(); )
                 {
                   TableTreeItem ancestor = (TableTreeItem)j.next();
                   if (ancestor != null)
                   {
                     gc.drawLine(x + offset/2, bounds.y, x + offset/2, bounds.y + bounds.height);
                   }
                   x += offset;
                 }

                 if (i + 1 == tableTreeItems.length)
                 {
                   if (i != 0 || !chain.isEmpty())
                   {
                     gc.drawLine
                       (x + offset/2, bounds.y, 
                        x + offset/2, bounds.y + (tableTreeItem.getItemCount() > 0 ? indent - 1 : bounds.height/2));
                   }
                 }
                 else
                 {
                   if (tableTreeItem.getItemCount() > 0)
                   {
                     gc.drawLine
                       (x + offset/2, bounds.y, x + offset/2, bounds.y + indent - 1);
                     gc.drawLine
                       (x + offset/2, bounds.y + bounds.height - indent + 2, x + offset/2, bounds.y + bounds.height);
                   }
                   else
                   {
                     gc.drawLine(x + offset/2, bounds.y, x + offset/2, bounds.y + bounds.height);
                   }
                 }

                 gc.drawLine
                   (x + (tableTreeItem.getItemCount() > 0 ? offset - indent + 1 : offset/2), bounds.y + (bounds.height + 1)/2, 
                    x + offset + 2, bounds.y + (bounds.height + 1)/2);

               }
               if (tableTreeItem.getExpanded())
               {
                 chain.addLast(i + 1 == tableTreeItems.length ? null : tableTreeItem);
                 if (!paintLines(gc, tableTreeItem.getItems()))
                 {
                   return false;
                 }
                 chain.removeLast();
               }
               if (isStarted && tableTreeItem == lastTableTreeItem)
               {
                 return false;
               }
             }
           }

           return true;
         }
       });
  }

  protected String imagePadding;
  protected int imagePaddingWidth;

  protected void createImagePadding(int width)
  {
    GC gc = new GC(getTableTree().getTable());
    imagePadding = " ";
    while ((imagePaddingWidth = gc.stringExtent(imagePadding).x) < width + 6)
    {
      imagePadding += " ";
    }
    gc.dispose();

    TableItem [] tableItems = getTableTree().getTable().getItems();
    for (int i = 0; i < tableItems.length; ++i)
    {
      TableTreeItem tableTreeItem = (TableTreeItem)tableItems[i].getData(ITEM_ID);
      tableTreeItem.setText(0, tableTreeItem.getText(0));
    }
  }

  public class ExtendedTableTreeItem extends TableTreeItem
  {
    protected Image firstImage;

    public ExtendedTableTreeItem(TableTree parent, int style)
    {
      super(parent, style);
    }

    public ExtendedTableTreeItem(TableTree parent, int style, int index)
    {
      super(parent, style, index);
    }

    public ExtendedTableTreeItem(TableTreeItem parent, int style)
    {
      super(parent, style);
    }

    public ExtendedTableTreeItem(TableTreeItem parent, int style, int index)
    {
      super(parent, style, index);
    }

    public void setText(int index, String text)
    {
      // System.out.println("setting the text " + index + " " + text + " " + getImage(index));
      if (index == 0 && imagePadding != null)
      {
        if (text != null && text.indexOf(imagePadding) == 0)
        {
          super.setText(0, text);
        }
        else
        {
          super.setText(0, imagePadding + text);
        }
      }
      else
      {
        super.setText(index, text);
      }
    }

    public String getText(int index)
    {
      String result = super.getText(index);
      if (index == 0 && result != null && imagePadding != null && result.indexOf(imagePadding) == 0)
      {
        result = result.substring(imagePadding.length());
      }

      return result;
    }

    public void setImage(int index, Image image)
    {
      if (index == 0)
      {
        firstImage = image;
        if (image != null && imagePadding == null) 
        {
          createImagePadding(image.getBounds().width);
        }
      }
      else
      {
        super.setImage(index, image);
      }
    }

    public Image getFirstImage()
    {
      return firstImage;
    }

    public int getImagePaddingWidth()
    {
      return imagePaddingWidth;
    }
  }
}
