/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.ecore.xmi/src/org/eclipse/emf/ecore/xmi/impl/XMLHelperImpl.java, emf.ecore.xmi, org.eclipse.dev, 20030620_1105VL
 * @version 1.22 6/20/03
 */
package org.eclipse.emf.ecore.xmi.impl;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.InternalEList;
import org.eclipse.emf.ecore.xmi.DanglingHREFException;
import org.eclipse.emf.ecore.xmi.IllegalValueException;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLHelper;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.XMLHelper.ManyReference;

/**
 * This class handles the package to use when there is no XML
 * namespace in an XML file.
 */
public class XMLHelperImpl implements XMLHelper
{
  protected static final Integer INTEGER_DATATYPE_IS_MANY = new Integer(DATATYPE_IS_MANY);
  protected static final Integer INTEGER_DATATYPE_SINGLE  = new Integer(DATATYPE_SINGLE);
  protected static final Integer INTEGER_IS_MANY_ADD      = new Integer(IS_MANY_ADD);
  protected static final Integer INTEGER_IS_MANY_MOVE     = new Integer(IS_MANY_MOVE);

  protected EPackage noNamespacePackage;
  protected XMLResource.XMLMap xmlMap;
  protected XMLResource resource;
  protected Map packages;
  protected Map featuresToKinds;
  protected String processDanglingHREF;
  protected DanglingHREFException danglingHREFException;
  protected Map prefixesToURIs;

  public static String saveString(Map options, List contents, String encoding, XMLHelper helper) throws Exception
  {
    if (helper == null)
      helper = new XMIHelperImpl();
    if (!options.containsKey(XMIResource.OPTION_DECLARE_XML))
    {
      options = new HashMap(options);
      options.put(XMIResource.OPTION_DECLARE_XML, Boolean.FALSE);
    }
    XMLSaveImpl save = new XMISaveImpl(options, helper, encoding);
    ((XMLHelperImpl)helper).processDanglingHREF = (String)options.get(XMIResource.OPTION_PROCESS_DANGLING_HREF);
    save.traverse(contents);
    char[] chars = save.toChar();
    return new String(chars);
  }

  public XMLHelperImpl()
  {
    super();
    packages = new HashMap();
    featuresToKinds = new HashMap();
    prefixesToURIs = new HashMap();
  }

  public XMLHelperImpl(XMLResource resource)
  {
    this();
    this.resource = resource;
  }

  public void setNoNamespacePackage(EPackage pkg)
  {
    noNamespacePackage = pkg;
  }

  public EPackage getNoNamespacePackage()
  {
    return noNamespacePackage;
  }

  public void setXMLMap(XMLResource.XMLMap map)
  {
    xmlMap = map;
    if (map != null && map.getNoNamespacePackage() != null)
    {
      setNoNamespacePackage(map.getNoNamespacePackage());
    }
  }

  public XMLResource.XMLMap getXMLMap()
  {
    return xmlMap;
  }

  public XMLResource getResource()
  {
    return resource;
  }

  public void setResource(XMLResource resource)
  {
    this.resource = resource;
  }

  public Object getValue(EObject obj, EStructuralFeature f)
  {
    return obj.eGet(f, false);
  }

  public String getQName(EClass c)
  {
    String name = getName(c);

    if (xmlMap != null)
    {
      XMLResource.XMLInfo clsInfo = xmlMap.getInfo(c);

      if (clsInfo != null)
      {
        String targetNamespace = clsInfo.getTargetNamespace();
        return getQName(targetNamespace, name);
      }
    }

    EPackage p = c.getEPackage();
    packages.put(p, null);

    if (p.getNsPrefix().equals(""))
    {
      return name;
    }
    else
    {
      return p.getNsPrefix() + ":" + name;
    }
  }

  protected String getQName(String uri, String name)
  {
    if (uri == null)
    {
      return name;
    }

    EPackage pkg = EPackage.Registry.INSTANCE.getEPackage(uri);

    if (pkg == null || pkg.getNsPrefix().equals(""))
    {
      return name;
    }
    else
    {
      packages.put(pkg, null);
      return pkg.getNsPrefix() + ":" + name;
    }
  }

  public String getName(ENamedElement obj)
  {
    XMLResource.XMLInfo info = null;

    if (xmlMap != null)
    {
      info = xmlMap.getInfo(obj);
    }

    if (info != null && info.getName() != null)
    {
      return info.getName();
    }
    else
    {
      return obj.getName();
    }
  }

  public String getQName(EStructuralFeature feature)
  {
    String name = getName(feature);

    if (xmlMap != null)
    {
      XMLResource.XMLInfo info = xmlMap.getInfo(feature);

      if (info != null)
      {
        return getQName(info.getTargetNamespace(), name);
      }
    }

    return name;
  }

  public String getID(EObject obj)
  {
    return resource == null ? null : resource.getID(obj);
  }

  public String getIDREF(EObject obj)
  {
    return resource == null ? null : resource.getURIFragment(obj);
  }

  protected URI handleDanglingHREF(EObject object)
  {
    if (!XMIResource.OPTION_PROCESS_DANGLING_HREF_DISCARD.equals(processDanglingHREF))
    {
      DanglingHREFException exception = new DanglingHREFException("The object '" + object + "' is not contained in a resource.", resource.getURI().toString(), 0, 0);
 
      if (danglingHREFException == null)
      {
        danglingHREFException = exception;
      }
   
      resource.getErrors().add(exception);
    }

    return null;
  }

  public String getHREF(EObject obj)
  {
    InternalEObject o = (InternalEObject) obj;

    URI objectURI = o.eProxyURI();
    if (objectURI == null)
    {
      Resource otherResource = obj.eResource();
      if (otherResource == null)
      {
        objectURI = handleDanglingHREF(obj);
        if (objectURI == null)
        {
          return null;
        }
      }
      else
      {
        objectURI = otherResource.getURI().appendFragment(otherResource.getURIFragment(obj));
      }
    }

    if (!objectURI.isRelative())
    {
      URI uri = resource.getURI();
      if (!uri.isRelative() && uri.isHierarchical())
      {
        URI deresolvedURI = objectURI.deresolve(uri, true, true, false);
        if (deresolvedURI.hasRelativePath())
        {
          objectURI = deresolvedURI;
        }
      }
    }

    return objectURI.toString();
  }

  public int getFeatureKind(EStructuralFeature feature)
  {
    Integer kind = (Integer) featuresToKinds.get(feature);
    if (kind != null)
    {
      return kind.intValue();
    }
    else
    {
      computeFeatureKind(feature);
      kind = (Integer) featuresToKinds.get(feature);
      if (kind != null)
      {
        return kind.intValue();
      }
      else
      {
        return OTHER;
      }
    }
  }

  public EObject createObject(EFactory eFactory, String classXMIName)
  {
    EPackage ePackage = eFactory.getEPackage();
    EClass eClass = (EClass)ePackage.getEClassifier(classXMIName);

    if (eClass == null && xmlMap != null)
    {
      eClass = (EClass) xmlMap.getClassifier(ePackage.getNsURI(), classXMIName);
    }

    if (eClass != null)
    {
      return (EObject) eFactory.create(eClass);
    }
    else
    {
      return null;
    }
  }

  public EStructuralFeature getFeature(EClass eClass, String namespaceURI, String name)
  {
    EStructuralFeature feature = getFeatureWithoutMap(eClass, name);

    if (feature == null && xmlMap != null)
    {
      feature = xmlMap.getFeature(eClass, namespaceURI, name);
      if (feature != null)
      {
        computeFeatureKind(feature);
      }
    }

    return feature;
  }

  protected EStructuralFeature getFeatureWithoutMap(EClass eClass, String name)
  {
    EStructuralFeature feature = (EStructuralFeature) eClass.getEStructuralFeature(name);

    if (feature != null)
      computeFeatureKind(feature);

    return feature;
  }

  protected void computeFeatureKind(EStructuralFeature feature)
  {
    EClassifier eMetaObject = feature.getEType();

    if (eMetaObject instanceof EDataType)
    {
      if (feature.isMany())
        featuresToKinds.put(feature, INTEGER_DATATYPE_IS_MANY);
      else
        featuresToKinds.put(feature, INTEGER_DATATYPE_SINGLE);
    }
    else
    {
      if (feature.isMany())
      {
        EReference reference = (EReference) feature;
        EReference opposite  = reference.getEOpposite();

        if (opposite == null || opposite.isTransient() || !opposite.isMany())
          featuresToKinds.put(feature, INTEGER_IS_MANY_ADD);
        else
          featuresToKinds.put(feature, INTEGER_IS_MANY_MOVE);
      }
    }
  }

  public String getJavaEncoding(String xmlEncoding)
  {
    return xmlEncoding;
  }

  public String getXMLEncoding(String javaEncoding)
  {
    return javaEncoding;
  }

  public EPackage[] packages()
  {
    Set pkgs = packages.keySet();
    EPackage[] packages = new EPackage[pkgs.size()];
    pkgs.toArray(packages);
    Comparator comparator =
      new Comparator()
      {
        public int compare(Object o1, Object o2)
        {
          return ((EPackage) o1).getNsPrefix().compareTo(((EPackage) o2).getNsPrefix());
        }
      };
    Arrays.sort(packages, comparator);
    return packages;
  }

  public void setValue(EObject object, EStructuralFeature feature, Object value, int position)
  {
    int kind = getFeatureKind(feature);

    switch (kind)
    {
      case DATATYPE_SINGLE:
      case DATATYPE_IS_MANY:
        EClassifier eMetaObject = feature.getEType();
        EDataType eDataType = (EDataType) eMetaObject;
        EFactory eFactory = eDataType.getEPackage().getEFactoryInstance();

        if (kind == DATATYPE_IS_MANY)
        {
          InternalEList list = (InternalEList) object.eGet(feature);
          if (position == -2)
          {
            for (StringTokenizer stringTokenizer = new StringTokenizer((String)value, " "); stringTokenizer.hasMoreTokens(); )
            {
              String token = stringTokenizer.nextToken();
              list.addUnique(eFactory.createFromString(eDataType, token));
            }

            // Make sure that the list will appear to be set to be empty.
            //
            if (list.isEmpty())
            {
              list.clear();
            }
          }
          else if (value == null)
          {
            list.addUnique(null);
          }
          else
          {
            list.addUnique(eFactory.createFromString(eDataType, (String) value));
          }
        }
        else if (value == null)
        {
          object.eSet(feature, null);
        }
        else
        {
          object.eSet(feature, eFactory.createFromString(eDataType, (String) value));
        }
        break;
      case IS_MANY_ADD:
      case IS_MANY_MOVE:
        InternalEList list = (InternalEList) object.eGet(feature);

        if (position == -1)
        {
          list.addUnique(value);
        }
        else if (position == -2)
        {
          list.clear();
        }
        else if (kind == IS_MANY_ADD)
        {
          list.addUnique(position, value);
        }
        else
        {
          list.move(position, value);
        }
        break;
      default:
        object.eSet(feature, value);
    }
  }


  public List setManyReference(ManyReference reference, String location)
  {
    EStructuralFeature feature = reference.getFeature();
    int kind = getFeatureKind(feature);
    InternalEList list = (InternalEList) reference.getObject().eGet(feature);
    List xmiExceptions = new BasicEList();
    Object[] values = reference.getValues();
    int[] positions = reference.getPositions();

    if (kind == IS_MANY_ADD)
      for (int i = 0, l = values.length; i < l; i++)
      {
        if (values[i] != null)
          try
          {
            list.addUnique(positions[i], values[i]);
          }
          catch (RuntimeException e)
          {
            xmiExceptions.add(new IllegalValueException
                                    (reference.getObject(),
                                     feature,
                                     values[i],
                                     e,
                                     location,
                                     reference.getLineNumber(),
                                     reference.getColumnNumber()
                                    ));
          }
      }
    else
      for (int i = 0, l = values.length; i < l; i++)
      {
        if (values[i] != null)
          try
          {
            list.move(positions[i], values[i]);
          }
          catch (RuntimeException e)
          {
            xmiExceptions.add(new IllegalValueException
                                    (reference.getObject(),
                                     feature,
                                     values[i],
                                     e,
                                     location,
                                     reference.getLineNumber(),
                                     reference.getColumnNumber()
                                    ));
          }
      }

    if (xmiExceptions.isEmpty())
      return null;
    else
      return xmiExceptions;
  }

  public void setProcessDanglingHREF(String value)
  {
    processDanglingHREF = value;
  }

  public DanglingHREFException getDanglingHREFException()
  {
    return danglingHREFException;
  }

  public URI resolve(URI relative, URI base) 
  {
    return relative.resolve(base);
  }

  public void addPrefix(String prefix, String uri) 
  {
    prefixesToURIs.put(prefix, uri);
  }

  public String getURI(String prefix) 
  {
    return (String) prefixesToURIs.get(prefix);
  }
}
