/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.mapping/src/org/eclipse/emf/mapping/provider/MappingItemProviderAdapterFactory.java, emf.mapping, org.eclipse.dev, 20030620_1105VL
 * @version 1.1 6/20/03
 */
package org.eclipse.emf.mapping.provider;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;

import org.eclipse.emf.edit.provider.ChangeNotifier;
import org.eclipse.emf.edit.provider.ComposeableAdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;
import org.eclipse.emf.edit.provider.Disposable;
import org.eclipse.emf.edit.provider.IChangeNotifier;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.edit.provider.IEditingDomainItemProvider;
import org.eclipse.emf.edit.provider.IItemLabelProvider;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.INotifyChangedListener;
import org.eclipse.emf.edit.provider.IStructuredItemContentProvider;
import org.eclipse.emf.edit.provider.ITableItemLabelProvider;
import org.eclipse.emf.edit.provider.ITreeItemContentProvider;
import org.eclipse.emf.edit.provider.IUpdateableItemText;

import org.eclipse.emf.mapping.util.MappingAdapterFactory;


/**
 * This is the factory that is used to provide the interfaces needed to support {@link org.eclipse.jface.viewers.Viewer}s.
 * The adapters generated by this factory convert EMF adapter notifications into calls to {@link #fireNotifyChanged fireNotifyChanged}.
 */
public class MappingItemProviderAdapterFactory 
  extends 
    MappingAdapterFactory 
  implements 
    ComposeableAdapterFactory, 
    IChangeNotifier,
    IDisposable
{
  /**
   *  This keeps track of the root adapter factory that delegates to this adapter factory
   */
  protected ComposedAdapterFactory parentAdapterFactory;

  /**
   * This is used to implement {@link #com.ibm.etools.emf.edit.provider.IChangeNotifier}.
   */
  protected IChangeNotifier changeNotifier = new ChangeNotifier();

  /**
   * This is used to implement {@link #com.ibm.etools.emf.edit.provider.IDisposable}.
   */
  protected Disposable disposable = new Disposable();

  /**
   * This keeps track of all the supported types checked by {@link #isFactoryForType isFactoryForType}.
   */
  protected Collection supportedTypes = new ArrayList();

  /**
   * This constructs an instance from a domain notifier.
   */
  public MappingItemProviderAdapterFactory()
  {
    supportedTypes.add(IStructuredItemContentProvider.class);
    supportedTypes.add(ITreeItemContentProvider.class);
    supportedTypes.add(IItemPropertySource.class);
    supportedTypes.add(IEditingDomainItemProvider.class);
    supportedTypes.add(IItemLabelProvider.class);
    supportedTypes.add(ITableItemLabelProvider.class);

    // This is here only to support deprecated adapt(object) rather than adapt(object, Interface.class).
    //
    // supportedTypes.add(null);

    // supportedTypes.add(IUpdateableItemText.class);
  }

  /**
   * This returns the root adapter factory that contains this factory.
   */
  public ComposeableAdapterFactory getRootAdapterFactory()
  {
    return parentAdapterFactory == null ? this : parentAdapterFactory.getRootAdapterFactory();
  }

  /**
   * This sets the composed adapter factory that contains this factory.
   */
  public void setParentAdapterFactory(ComposedAdapterFactory parentAdapterFactory)
  {
    this.parentAdapterFactory = parentAdapterFactory;
  }

  public boolean isFactoryForType(Object type)
  {
    return super.isFactoryForType(type) || supportedTypes.contains(type);
  }

  public Adapter adapt(Notifier notifier, Object type)
  {
    return super.adapt(notifier, this);
  }

  public Object adapt(Object object, Object type)
  {
    if (isFactoryForType(type))
    {
      Object adapter = super.adapt(object, type);
      if (!(type instanceof Class) || (((Class)type).isInstance(adapter)))
      {
        return adapter;
      }
    }

    return null;
  }

  public Adapter adaptNew(Notifier object, Object type)
  {
    Adapter result = super.adaptNew(object, type);
    disposable.add(result);
    return result;
  }

  /**
   * This adds a listener.
   */
  public void addListener(INotifyChangedListener notifyChangedListener)
  {
    changeNotifier.addListener(notifyChangedListener);
  }

  /**
   * This removes a listener.
   */
  public void removeListener(INotifyChangedListener notifyChangedListener)
  {
    changeNotifier.removeListener(notifyChangedListener);
  }

  /**
   * This delegates to {@link #changeNotifier} and to {@link #parentAdapterFactory}.
   */
  public void fireNotifyChanged(Notification msg)
  {
     changeNotifier.fireNotifyChanged(msg);

    if (parentAdapterFactory != null)
    {
      parentAdapterFactory.fireNotifyChanged(msg);
    }
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.MappingRoot} instances.
   */
  protected MappingRootItemProvider mappingRootItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.MappingRoot}.
   */
  public Adapter createMappingRootAdapter()
  {
    // EATM
    // We want stateful ones.
    // if (mappingRootItemProvider == null)
    // {
    //   mappingRootItemProvider = new MappingRootItemProvider(this, domainNotifier);
    // }

    mappingRootItemProvider = new MappingRootItemProvider(this);

    return mappingRootItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.Mapping} instances.
   */
  protected MappingItemProvider mappingItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.Mapping}.
   */
  public Adapter createMappingAdapter()
  {
    // EATM 
    // We want stateful ones.
    // if (mappingItemProvider == null)
    // {
    //   mappingItemProvider = new MappingItemProvider(this);
    // }

    mappingItemProvider = new MappingItemProvider(this);

    return mappingItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.FunctionPair} instances.
   */
  protected FunctionPairItemProvider functionPairItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.FunctionPair}.
   */
  public Adapter createFunctionPairAdapter()
  {
    // if (functionPairItemProvider == null)
    {
      functionPairItemProvider = new FunctionPairItemProvider(this);
    }

    return functionPairItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.TypeConverter} instances.
   */
  protected TypeConverterItemProvider typeConverterItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.TypeConverter}.
   */
  public Adapter createTypeConverterAdapter()
  {
    // if (typeConverterItemProvider == null)
    {
      typeConverterItemProvider = new TypeConverterItemProvider(this);
    }

    return typeConverterItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.MappingHelper} instances.
   */
  protected MappingHelperItemProvider mappingHelperItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.MappingHelper}.
   */
  public Adapter createMappingHelperAdapter()
  {
    // if (mappingHelperItemProvider == null)
    {
      mappingHelperItemProvider = new MappingHelperItemProvider(this);
    }

    return mappingHelperItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.MappingStrategy} instances.
   */
  protected MappingStrategyItemProvider mappingStrategyItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.MappingStrategy}.
   */
  public Adapter createMappingStrategyAdapter()
  {
    // if (mappingStrategyItemProvider == null)
    {
      mappingStrategyItemProvider = new MappingStrategyItemProvider(this);
    }

    return mappingStrategyItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.FunctionNamePair} instances.
   */
  protected FunctionNamePairItemProvider functionNamePairItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.FunctionNamePair}.
   */
  public Adapter createFunctionNamePairAdapter()
  {
    // if (functionNamePairItemProvider == null)
    {
      functionNamePairItemProvider = new FunctionNamePairItemProvider(this);
    }

    return functionNamePairItemProvider;
  }

  /**
   * This keeps track of the one adapter used for all {@link org.eclipse.emf.mapping.ComplexTypeConverter} instances.
   */
  protected ComplexTypeConverterItemProvider complexTypeConverterItemProvider;

  /**
   * This creates an adapter for a {@link org.eclipse.emf.mapping.ComplexTypeConverter}.
   */
  public Adapter createComplexTypeConverterAdapter()
  {
    // if (complexTypeConverterItemProvider == null)
    {
      complexTypeConverterItemProvider = new ComplexTypeConverterItemProvider(this);
    }

    return complexTypeConverterItemProvider;
  }

  public void dispose()
  {
    disposable.dispose();
  }
}
