/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.common/src/org/eclipse/emf/common/notify/impl/NotifierImpl.java, emf.common, org.eclipse.111, 20031020_1612WL
 * @version 1.18 10/20/03
 */
package org.eclipse.emf.common.notify.impl;


import java.util.Collection;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;


/**
 * An extensible notifier implementation.
 */
public class NotifierImpl implements Notifier
{

  /**
   * The bit of {@link #eFlags} that is used to represent {@link #eDeliver}.
   */
  protected static final int EDELIVER = 0x0001;

  /**
   * The last bit used by this class; derived classes may use bit values higher than this.
   */
  protected static final int ELAST_NOTIFIER_FLAG = EDELIVER;

  /**
   * An extensible set of bit flags;
   * the first bit is used for {@link #EDELIVER} to implement {@link #eDeliver}.
   */
  protected int eFlags = EDELIVER;

  /**
   * The list of {@link Adapter}s associated with the notifier.
   */
  protected BasicEList eAdapters;

  /**
   * Creates a blank new instance.
   */
  public NotifierImpl()
  {
  }

  /*
   * Javadoc copied from interface.
   */
  public EList eAdapters()
  {
    if (eAdapters == null)
    {
      eAdapters =  
       new BasicEList()
       {
         protected boolean safe;

         protected boolean canContainNull()
         {
           return false;
         }

         protected boolean useEquals()
         {
           return false;
         }

         protected Object [] newData(int capacity)
         {
           return new Adapter [capacity];
         }

         protected void didAdd(int index, Object newObject)
         {
           Adapter adapter = (Adapter)newObject;
           adapter.setTarget(NotifierImpl.this);
         }

         protected void didRemove(int index, Object oldObject)
         {
           Adapter adapter = (Adapter)oldObject;
           if ((eFlags & EDELIVER) != 0)
           {
             Notification notification = 
               new NotificationImpl(Notification.REMOVING_ADAPTER, oldObject, null, index)
               {
                 public Object getNotifier()
                 {
                   return NotifierImpl.this;
                 }
               };
             adapter.notifyChanged(notification);
           }
           if (adapter.getTarget() == NotifierImpl.this) 
           {
             adapter.setTarget(null);
           }
         }

         public Object [] data()
         {
           safe = true;
           return (Adapter [])data;
         }

         protected void ensureSafety()
         {
           if (safe && data != null)
           {
             Object [] oldData = data;
             data = newData(data.length);
             System.arraycopy(oldData, 0, data, 0, size);
             safe = false;
           }
         }

         public boolean add(Object object)
         {
           ensureSafety();
           return super.add(object);
         }

         public void add(int index, Object object)
         {
           ensureSafety();
           super.add(index, object);
         }

         public boolean addAll(Collection collection)
         {
           ensureSafety();
           return super.addAll(collection);
         }

         public boolean remove(Object object)
         {
           ensureSafety();
           return super.remove(object);
         }

         public Object remove(int index)
         {
           ensureSafety();
           return super.remove(index);
         }

         public boolean removeAll(Collection collection)
         {
           ensureSafety();
           return super.removeAll(collection);
         }

         public void clear()
         {
           ensureSafety();
           super.clear();
         }

         public boolean retainAll(Collection collection)
         {
           ensureSafety();
           return super.retainAll(collection);
         }

         public Object set(int index, Object object)
         {
           ensureSafety();
           return super.set(index, object);
         }

         public void move(int newPosition, Object object)
         {
           ensureSafety();
           super.move(newPosition, object);
         }

         public Object move(int newPosition, int oldPosition)
         {
           ensureSafety();
           return super.move(newPosition, oldPosition);
         }
       };
    }
    return eAdapters;
  }

  /*
   * Javadoc copied from interface.
   */
  public boolean eDeliver()
  {
    return (eFlags & EDELIVER) != 0;
  }

  /*
   * Javadoc copied from interface.
   */
  public void eSetDeliver(boolean deliver)
  {
    if (deliver)
    {
      this.eFlags |= EDELIVER;
    }
    else
    {
      this.eFlags &= ~EDELIVER;
    }
  }

  /*
   * Javadoc copied from interface.
   */
  public void eNotify(Notification notification)
  {
    if (eDeliver() && eAdapters != null)
    {
      int size = eAdapters.size();
      if (size > 0)
      {
        Adapter [] adapters = (Adapter [])eAdapters.data();
        for (int i = 0; i < size; ++i)
        {
          adapters[i].notifyChanged(notification);
        }
      }
    }
  }

  /**
   * Returns whether {@link #eNotify eNotify} needs to be called.
   * This may return <code>true</code> even when {@link #eDeliver eDeliver} is <code>false</code> 
   * or when {@link #eAdapters eAdapters} is empty.
   * @return whether {@link #eNotify eNotify} needs to be called.
   */
  public boolean eNotificationRequired()
  {
    return eAdapters != null && eDeliver() && !eAdapters.isEmpty();
  }
}
