/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.ecore/src/org/eclipse/emf/ecore/util/EcoreDynamicEList.java, emf.ecore, org.eclipse.102, 20030326_0335VL
 * @version 1.33 3/26/03
 */
package org.eclipse.emf.ecore.util;


import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.NotificationChain;

import org.eclipse.emf.common.util.EList;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;

import org.eclipse.emf.ecore.impl.ENotificationImpl;


public class EcoreDynamicEList extends EcoreEList 
{
  public static Class wrapperClassFor(Class javaClass)
  {
    if (javaClass == null)
    {
      return Object.class;
    }
    else if (javaClass.isPrimitive())
    {
      if (javaClass == Boolean.TYPE)
      {
        return Boolean.class;
      }
      else if (javaClass == Integer.TYPE)
      {
        return Integer.class;
      }
      else if (javaClass == Float.TYPE)
      {
        return Float.class;
      }
      else if (javaClass == Double.TYPE)
      {
        return Double.class;
      }
      else if (javaClass == Long.TYPE)
      {
        return Long.class;
      }
      else if (javaClass == Short.TYPE)
      {
        return Short.class;
      }
      else if (javaClass == Byte.TYPE)
      {
        return Byte.class;
      }
      else // if (javaClass == Character.TYPE)
      {
        return Character.class;
      }
    }
    else 
    {
      return javaClass;
    }
  }

  protected EClassifier eClassifier;
  protected EStructuralFeature eStructuralFeature;

  public static final int HAS_MANY_INVERSE = 0x0001;
  public static final int HAS_NAVIGABLE_INVERSE = 0x0002;
  public static final int IS_EOBJECT = 0x0004;
  public static final int IS_CONTAINMENT = 0x0008;
  public static final int IS_UNIQUE = 0x0010;
  public static final int HAS_PROXIES = 0x0020;
  public static final int IS_MANY = 0x0040;
  public static final int HAS_INSTANCE_CLASS = 0x0080;
  public static final int IS_CONTAINER = 0x0100;
  public static final int IS_UNSETTABLE = 0x0200;
  public static final int IS_PRIMITIVE = 0x0400;
  public static final int IS_ENUM = 0x0800;
  public static final int IS_SET = 0x1000;

  protected int kind;

  public EcoreDynamicEList(EStructuralFeature eStructuralFeature, InternalEObject owner)
  {
    super(wrapperClassFor(eStructuralFeature.getEType().getInstanceClass()), owner);

    this.eStructuralFeature = eStructuralFeature;
    this.eClassifier = eStructuralFeature.getEType();

    // This forces the feature ids to be assigned.
    //
    owner.eClass().getEAllStructuralFeatures();

    if (eClassifier.getInstanceClass() != null) 
    {
      kind |= HAS_INSTANCE_CLASS;
    }

    if (eStructuralFeature.isUnsettable())
    {
      kind |= IS_UNSETTABLE;
    }

    if (eStructuralFeature instanceof EReference)
    {
      EReference eReference = (EReference)eStructuralFeature;
      EReference inverseEStructuralFeature = eReference.getEOpposite();
      if (eReference.isContainment())
      {
        kind |= IS_CONTAINMENT;
      }

      if (inverseEStructuralFeature != null)
      {
        // This forces the feature ids to be assigned.
        //
        inverseEStructuralFeature.getEContainingClass().getEAllStructuralFeatures();
        kind |= HAS_NAVIGABLE_INVERSE;
        if (inverseEStructuralFeature.isMany())
        {
          kind |= HAS_MANY_INVERSE;
        }
        if (inverseEStructuralFeature.isContainment())
        {
          kind |= IS_CONTAINER;
        }
      }

      kind |= IS_EOBJECT;
    }
    else // if (eStructuralFeature instanceof EAttribute
    {
      if (eClassifier instanceof EEnum)
      {
        kind |= IS_ENUM;
      }
      else
      {
        Class instanceClass = eClassifier.getInstanceClass();
        if (instanceClass != null && instanceClass.isPrimitive())
        {
          kind |= IS_PRIMITIVE;
        }
      }
    }

    if (eStructuralFeature.isMany())
    {
      kind |= IS_MANY;
    }

    if (eStructuralFeature.isUnique())
    {
      kind |= IS_UNIQUE;
    }
  }

  public EcoreDynamicEList(int kind, EStructuralFeature eStructuralFeature, InternalEObject owner)
  {
    super(wrapperClassFor(eStructuralFeature.getEType().getInstanceClass()), owner);
    this.kind = kind;
  }

  protected EcoreDynamicEList(Class instanceClass, InternalEObject owner)
  {
    super(instanceClass, owner);
  }

  public Object getDefaultValue()
  {
    return eStructuralFeature.getDefaultValue();
  }

  public EStructuralFeature getEStructuralFeature()
  {
    return eStructuralFeature;
  }

  protected Object validate(int index, Object object)
  {
    if (object != null && !hasInstanceClass() && !eClassifier.isInstance(object))
    {
      throw new ArrayStoreException();
    }
    return super.validate(index, object);
  }

  protected boolean hasInverse()
  {
    return (kind & (HAS_NAVIGABLE_INVERSE | IS_CONTAINMENT)) != 0;
  }

  protected boolean hasManyInverse()
  {
    return (kind & HAS_MANY_INVERSE) != 0;
  }

  protected boolean hasNavigableInverse()
  {
    return (kind & HAS_NAVIGABLE_INVERSE) != 0;
  }

  protected boolean isEObject()
  {
    return (kind & IS_EOBJECT) != 0;
  }

  protected boolean isContainment()
  {
    return (kind & IS_CONTAINMENT) != 0;
  }

  protected boolean isUnique()
  {
    return (kind & IS_UNIQUE) != 0;
  }

  protected boolean hasProxies()
  {
    return (kind & HAS_PROXIES) != 0;
  }

  protected boolean isMany()
  {
    return (kind & IS_MANY) != 0;
  }

  protected boolean hasInstanceClass()
  {
    return (kind & HAS_INSTANCE_CLASS) != 0;
  }

  protected boolean isContainer()
  {
    return (kind & IS_CONTAINER) != 0;
  }

  protected boolean isUnsettable()
  {
    return (kind & IS_UNSETTABLE) != 0;
  }

  protected boolean isPrimitive()
  {
    return (kind & IS_PRIMITIVE) != 0;
  }

  protected boolean isEnum()
  {
    return (kind & IS_ENUM) != 0;
  }

  public Object get(boolean resolve)
  {
    if (isMany())
    {
      return super.get(resolve);
    }
    else if (isContainer())
    {
      return owner.eContainer();
    }
    else if (isEmpty())
    {
      return getDefaultValue();
    }
    else if (resolve)
    {
      return get(0);
    }
    else
    {
      return data[0];
    }
  }

  public void set(Object newValue)
  {
    if (isMany())
    {
      super.set(newValue);
    }
    else if (isContainer())
    {
      EObject eContainer = owner.eContainer();
      if (newValue != eContainer || (owner.eContainmentFeature() != getInverseEStructuralFeature() && newValue != null))
      {
        if (EcoreUtil.isAncestor(owner, (EObject)newValue))
          throw new IllegalArgumentException("Recursive containment not allowed for " + toString() + ".");

        NotificationChain notifications = null;
        if (newValue != null)
        {
          notifications = ((InternalEObject)newValue).eInverseAdd(owner, getInverseFeatureID(), getInverseFeatureClass(), notifications);
        }

        if (eContainer != null)
        {
          notifications = owner.eBasicRemoveFromContainer(notifications);
        }
        notifications = owner.eBasicSetContainer((InternalEObject)newValue, getFeatureID(), notifications);
        if (notifications != null) notifications.dispatch();
      }
      else
      {
        if (owner.eNotificationRequired())
          owner.eNotify(new ENotificationImpl(owner, Notification.SET, getFeatureID(), newValue, newValue));
      }
    }
    else if (newValue == null && !canContainNull())
    {
      if (!isEmpty())
      {
        remove(0);
      }
      else
      {
        if (isNotificationRequired())
        {
          Object defaultValue = getDefaultValue();
          owner.eNotify(createNotification(Notification.SET, defaultValue, defaultValue, Notification.NO_INDEX));
        }
      }
    }
    else if (isEmpty())
    {
      add(newValue);
    }
    else
    {
      set(0, newValue);
    }
  }

  public boolean isSet()
  {
    if (isMany())
    {
      return 
        isUnsettable() ?
          (kind & IS_SET) != 0 :
          super.isSet();
    }
    else if (isContainer())
    {
      return owner.eContainer() != null;
    }
    else if (!isUnsettable())
    {
      if (isEmpty()) return false;
      Object defaultValue = getDefaultValue();
      Object currentValue = data[0];
      return defaultValue == null ? currentValue != null : !defaultValue.equals(currentValue);
    }
    else
    {
      return !isEmpty();
    }
  }

  public void unset()
  {
    if (isMany())
    {
      super.unset();
      if (isUnsettable())
      {
        if (isNotificationRequired())
        {
          boolean oldIsSet = (kind & IS_SET) != 0;
          kind &= ~IS_SET;
          owner.eNotify(createNotification(Notification.UNSET, oldIsSet, false));
        }
        else
        {
          kind &= ~IS_SET;
        }
      }
    }
    else if (isContainer())
    {
      set(null);
    }
    else if (!isEmpty())
    {
      remove(0);
    }
    else
    {
      if (isNotificationRequired())
      {
        Object defaultValue = getDefaultValue();
        owner.eNotify
          (createNotification(isUnsettable() ? Notification.UNSET : Notification.SET,
                              defaultValue,
                              defaultValue,
                              Notification.NO_INDEX));
      }
    }
  }

  protected Notification createNotification(int eventType, Object oldObject, Object newObject, int index)
  {
    if (!isMany())
    {
      switch (eventType)
      {
        case Notification.ADD:
          return
            new ENotificationImpl
              (owner,
               Notification.SET,
               getEStructuralFeature(),
               getDefaultValue(),
               newObject,
               isUnsettable());
        case Notification.REMOVE:
          return
            new ENotificationImpl
              (owner,
               isUnsettable() ? Notification.UNSET : Notification.SET,
               getEStructuralFeature(),
               oldObject,
               getDefaultValue(),
               isUnsettable());
        case Notification.SET:
          return
            new ENotificationImpl
              (owner,
               Notification.SET,
               getEStructuralFeature(),
               oldObject,
               newObject,
               Notification.NO_INDEX);
      }
    }

    return new ENotificationImpl(owner, eventType, getEStructuralFeature(), oldObject, newObject, index);
  }

  public NotificationChain basicAdd(Object object, NotificationChain notifications)
  {
    if (isContainer())
    {
      if (owner.eContainer() != null)
      {
        notifications = owner.eBasicRemoveFromContainer(notifications);
      }
      return owner.eBasicSetContainer((InternalEObject)object, getFeatureID(), notifications);
    }
    else if (!isMany() && !isEmpty())
    {
      if (hasInverse())
      {
        Object oldObject = data[0];
        if (oldObject != object)
        {
          notifications = inverseRemove(oldObject, notifications);
        }
      }
      return basicSet(0, object, notifications);
    }
    else
    {
      return super.basicAdd(object, notifications);
    }
  }

  public NotificationChain basicRemove(Object object, NotificationChain notifications)
  {
    if (isContainer())
    {
      return owner.eBasicSetContainer(null, getFeatureID(), notifications);
    }
    else
    {
      return super.basicRemove(object, notifications);
    }
  }

  protected boolean canContainNull()
  {
    return (!isEObject() || !isMany() && isUnsettable()) && !isPrimitive() && !isEnum();
  }

  public NotificationChain inverseAdd(Object object, NotificationChain notifications)
  {
    return 
      object == null ?
        notifications :
        super.inverseAdd(object, notifications);
  }

  public NotificationChain inverseRemove(Object object, NotificationChain notifications)
  {
    return 
      object == null ?
        notifications :
        super.inverseRemove(object, notifications);
  }

  protected EObject resolveProxy(EObject eObject)
  {
    return eObject == null ? null : super.resolveProxy(eObject);
  }
}
