/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.ecore.xmi/src/org/eclipse/emf/ecore/xmi/impl/XMLString.java, emf.ecore.xmi, org.eclipse.102, 20030326_0335VL
 * @version 1.8 3/26/03
 */
package org.eclipse.emf.ecore.xmi.impl;


import java.util.List;

import org.eclipse.emf.common.util.BasicEList;


/*
 * Calling sequence is:
 * startElement()
 *   addAttrribute() 0 or more times
 *   startAttribute(), addAttributeContent(), and endAttribute(), 0 or more times
 *   endEmptyElement() or endContentElement()
 *   startElement()  0 or more times
 * endElement()
 */
public class XMLString extends StringSegment
{
  protected List elementNames;
  protected List indents;
  protected int lineWidth;
  protected int markedLineWidth;
  protected int currentLineWidth;
  protected boolean lastElementIsStart;

  public XMLString(int lineWidth) 
  {
    super();

    this.lineWidth = lineWidth;

    elementNames = new BasicEList();
    indents = new BasicEList();
    indents.add("");
  }

  public void startElement(String name) 
  {
  	if (lastElementIsStart)
  	{
  		closeStartElement();
  	}
    elementNames.add(name);
    add(getElementIndent());
    add("<");
    add(name);
    lastElementIsStart = true;
  }

  public void addAttribute(String name, String value) 
  {
    if (currentLineWidth > lineWidth)
    {
      addLine();
      add(getAttributeIndent());
    }
    else
    {
      add(" ");
    }
    add(name);
    add("=\"");
    add(value);
    add("\"");
  }

  public void addAttributeNS(String prefix, String localName, String value) 
  {
    if (currentLineWidth > lineWidth)
    {
      addLine();
      add(getAttributeIndent());
    }
    else
    {
      add(" ");
    }
    add(prefix);
    add(":");
    add(localName);
    add("=\"");
    add(value);
    add("\"");
  }

  public void startAttribute(String name)
  {
    if (currentLineWidth > lineWidth)
    {
      addLine();
      add(getAttributeIndent());
    }
    else
    {
      add(" ");
    }
    add(name);
    add("=\"");
  }

  public void addAttributeContent(String content)
  {
    add(content);
  }

  public void endAttribute()
  {
    add("\"");
  }

  protected void closeStartElement() 
  {
    add(">");
    addLine();
  }

  public void endEmptyElement() 
  {
    removeLast();
    add("/>");
    addLine();
    lastElementIsStart = false;
  }

  public void endContentElement(String content) 
  {
    add(">");
    add(content);
    add("</");
    String name = removeLast();
    add(name);
    add(">");
    addLine();
    lastElementIsStart = false;
  }
  
  public void endElement() 
  {
  	if (lastElementIsStart)
  	{
  		endEmptyElement();
  	}
  	else
  	{
	    add(getElementIndent());
	    add("</");
	    String name = removeLast();
	    add(name);
	    add(">");
	    addLine();
	    lastElementIsStart = false;
  	}
  }

  protected String removeLast() 
  {
    int end = elementNames.size();
    return (String) elementNames.remove(end-1);
  }

  protected String getElementIndent() 
  {
    int depth = elementNames.size() - 1; 
    for (int i = indents.size() - 1; i < depth; ++i)
    {
      indents.add(indents.get(i) + "  ");
    }
    return (String)indents.get(depth);
  }

  protected String getAttributeIndent() 
  {
    int depth = elementNames.size() + 1; 
    for (int i = indents.size() - 1; i < depth; ++i)
    {
      indents.add(indents.get(i) + "  ");
    }
    return (String)indents.get(depth);
  }


  public void add(String newString)
  {
    // Avoid a function call...
    //
    if (lineWidth != Integer.MAX_VALUE)
    {
      currentLineWidth += newString.length();
    }
    super.add(newString);
  }

  public void addLine()
  {
    super.addLine();
    currentLineWidth = 0;
  }

  public Object mark()
  {
    markedLineWidth = currentLineWidth;
    currentLineWidth = lineWidth - 2;
    return super.mark();
  }

  public void resetToMark(Object mark)
  {
    super.resetToMark(mark);

    // Ensure correct indentation for a top level attribute.
    //
    elementNames.add("mark");

    currentLineWidth = markedLineWidth;
  }
}
