/**
 * <copyright> 
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.edit.ui/src/org/eclipse/emf/edit/ui/action/EditingDomainActionBarContributor.java, emf.edit.ui, org.eclipse.102, 20030326_0335VL
 * @version 1.2 3/26/03
 */
package org.eclipse.emf.edit.ui.action;


import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;

import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;

import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;

import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPropertyListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchActionConstants;

import org.eclipse.ui.part.IPage;
import org.eclipse.ui.part.MultiPageEditorActionBarContributor;
import org.eclipse.ui.part.MultiPageEditorPart;

import org.eclipse.ui.views.properties.PropertySheet;
import org.eclipse.ui.views.properties.IPropertySheetPage;


/**
 * This is a contributor for an editor, multipage or otherwise,
 * that implements {@link IEditingDomainProvider}.
 * It automatically hooks up the Undo, Redo, Cut, Copy, Paste, and Delete actions on the Edit menu
 * to the corresponding commands supported by the {@link EditingDomain}.
 * The editor site'selection provider is used to keep the Cut, Copy, Paste, and Delete actions up-to-date.
 * The actions are also refreshed everytime the editor fires to its {@link IPropertyListener}s.
 * <p>
 * Another very useful feature of this contributor is that it can be used as follows:
 * <pre>
 *   ((IMenuListener)((IEditorSite)getSite()).getActionBarContributor()).menuAboutToShow(menuManager);
 * </pre>
 * to contribute the Edit menu actions to a popup menu.
 */
public class EditingDomainActionBarContributor 
  extends 
    MultiPageEditorActionBarContributor 
  implements 
    IMenuListener,
    IPropertyListener
{
  /**
   * This keeps track of the current editor part.
   */
  protected IEditorPart activeEditor;

  /**
   * This is the action used to implement delete.
   */
  protected DeleteAction deleteAction;

  /**
   * This is the action used to implement cut.
   */
  protected CutAction cutAction;

  /**
   * This is the action used to implement copy.
   */
  protected CopyAction copyAction;

  /**
   * This is the action used to implement paste.
   */
  protected PasteAction pasteAction;

  /**
   * This is the action used to implement undo.
   */
  protected UndoAction undoAction;

  /**
   * This is the action used to implement redo.
   */
  protected RedoAction redoAction;


  /**
   * This creates an instance the contributor.
   */
  public EditingDomainActionBarContributor()
  {
    super();
  }

  public void init(IActionBars actionBars)
  {
    super.init(actionBars);

    deleteAction = new DeleteAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.DELETE, deleteAction);

    cutAction = new CutAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.CUT, cutAction);

    copyAction = new CopyAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.COPY, copyAction);

    pasteAction = new PasteAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.PASTE, pasteAction);

    undoAction = new UndoAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.UNDO, undoAction);

    redoAction = new RedoAction();
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.REDO, redoAction);
  }

  public void contributeToMenu(IMenuManager menuManager) 
  {
    super.contributeToMenu(menuManager);
  }

  public void contributeToStatusLine(IStatusLineManager statusLineManager) 
  {
    super.contributeToStatusLine(statusLineManager);
  }

  public void contributeToToolBar(IToolBarManager toolBarManager)
  {
    super.contributeToToolBar(toolBarManager);
  }

  public void shareGlobalActions(IPage page, IActionBars actionBars)
  {
    if (!(page instanceof IPropertySheetPage))
    {
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.DELETE, deleteAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.CUT, cutAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.COPY, copyAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.PASTE, pasteAction);
    }
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.UNDO, undoAction);
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.REDO, redoAction);
  }

  /**
   * @deprecated
   */
  public void setActiveView(IViewPart part)
  {
    IActionBars actionBars = part.getViewSite().getActionBars();
    if (!(part instanceof PropertySheet))
    {
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.DELETE, deleteAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.CUT, cutAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.COPY, copyAction);
      actionBars.setGlobalActionHandler(IWorkbenchActionConstants.PASTE, pasteAction);
    }
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.UNDO, undoAction);
    actionBars.setGlobalActionHandler(IWorkbenchActionConstants.REDO, redoAction);

    actionBars.updateActionBars();
  }

  public void setActiveEditor(IEditorPart part) 
  {
    super.setActiveEditor(part);

    if (part != activeEditor)
    {
      if (activeEditor != null)
      {
        deactivate();
      }

      if (part instanceof IEditingDomainProvider)
      {
        activeEditor = part;
        activate();

      }
    }
  }

  public void setActivePage(IEditorPart part) 
  {
  }

  public void deactivate()
  {
    activeEditor.removePropertyListener(this);

    deleteAction.setActiveEditor(null);
    cutAction.setActiveEditor(null);
    copyAction.setActiveEditor(null);
    pasteAction.setActiveEditor(null);
    undoAction.setActiveEditor(null);
    redoAction.setActiveEditor(null);

    ISelectionProvider selectionProvider = 
      activeEditor instanceof ISelectionProvider ?
        (ISelectionProvider)activeEditor :
        activeEditor.getEditorSite().getSelectionProvider();
    selectionProvider.removeSelectionChangedListener(deleteAction);
    selectionProvider.removeSelectionChangedListener(cutAction);
    selectionProvider.removeSelectionChangedListener(copyAction);
    selectionProvider.removeSelectionChangedListener(pasteAction);
  }

  public void activate()
  {
    activeEditor.addPropertyListener(this);

    deleteAction.setActiveEditor(activeEditor);
    cutAction.setActiveEditor(activeEditor);
    copyAction.setActiveEditor(activeEditor);
    pasteAction.setActiveEditor(activeEditor);
    undoAction.setActiveEditor(activeEditor);
    redoAction.setActiveEditor(activeEditor);

    ISelectionProvider selectionProvider = 
      activeEditor instanceof ISelectionProvider ?
        (ISelectionProvider)activeEditor :
        activeEditor.getEditorSite().getSelectionProvider();
    selectionProvider.addSelectionChangedListener(deleteAction);
    selectionProvider.addSelectionChangedListener(cutAction);
    selectionProvider.addSelectionChangedListener(copyAction);
    selectionProvider.addSelectionChangedListener(pasteAction);

    update();
  }

  public void update()
  {
    ISelectionProvider selectionProvider = 
      activeEditor instanceof ISelectionProvider ?
        (ISelectionProvider)activeEditor :
        activeEditor.getEditorSite().getSelectionProvider();
    ISelection selection = selectionProvider.getSelection();
    IStructuredSelection structuredSelection =
      selection instanceof IStructuredSelection ?  (IStructuredSelection)selection : StructuredSelection.EMPTY;

    deleteAction.updateSelection(structuredSelection);
    cutAction.updateSelection(structuredSelection);
    copyAction.updateSelection(structuredSelection);
    pasteAction.updateSelection(structuredSelection);
    undoAction.update();
    redoAction.update();
  }

  /**
   * This implements {@link org.eclipse.jface.action.IMenuListener} to help fill the context menus with contributions from the Edit menu.
   */
  public void menuAboutToShow(IMenuManager menuManager)
  {
    // Add our standard marker.
    //
    menuManager.add(new Separator("additions"));
    menuManager.add(new Separator("edit"));

    // Add the edit menu actions.
    //
    menuManager.add(new ActionContributionItem(undoAction));
    menuManager.add(new ActionContributionItem(redoAction));
    menuManager.add(new Separator());
    menuManager.add(new ActionContributionItem(cutAction));
    menuManager.add(new ActionContributionItem(copyAction));
    menuManager.add(new ActionContributionItem(pasteAction));
    menuManager.add(new Separator());
    menuManager.add(new ActionContributionItem(deleteAction));

    // Add our other standard marker.
    //
    menuManager.add(new Separator("additions-end"));
  }

  public void propertyChanged(Object source, int id)
  {
    update();
  }
}
