/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.ecore/src/org/eclipse/emf/ecore/InternalEObject.java, emf.ecore, org.eclipse.dev, 20030620_1105VL
 * @version 1.30 6/20/03
 */
package org.eclipse.emf.ecore;


import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.NotificationChain;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;

import org.eclipse.emf.ecore.resource.Resource;


/**
 * An internal interface implemented by all {@link EObject}s.
 */
public interface InternalEObject extends EObject
{
  /**
   * The base value for negative, i.e., opposite-end, eContainerFeatureID values.
   */
  static final int EOPPOSITE_FEATURE_BASE = -1;

  /**
   * Returns whether {@link #eNotify eNotify} needs to be called.
   * <p>
   * This may return <code>true</code> even when {@link #eDeliver eDeliver} is <code>false</code>
   * or when {@link #eAdapters eAdapters} is empty.
   * <p>
   * @return whether {@link #eNotify eNotify} needs to be called.
   */
  boolean eNotificationRequired();

  /**
   * Returns the fragment segment that, 
   * when passed to {@link #eObjectForURIFragmentSegment eObjectForURIFragmentSegment},
   * will resolve to the given object in this object's given feature.
   * <p>
   * The feature argument may be <code>null</code> in which case it will be deduced, if possible.
   * The default result will be of the form:
   *<pre>
   *  "@&lt;feature-name>[.&lt;index>]"
   *</pre>
   * The index is used only for {@link EStructuralFeature#isMany many-valued} features;
   * it represents the position within the list.
   * </p>
   * @param eFeature the feature relating the given object to this object, or <code>null</code>.
   * @param eObject the object to be identified.
   * @return the fragment segment that resolves to the given object in this object's given feature.
   */
  String eURIFragmentSegment(EStructuralFeature eFeature, EObject eObject);

  /**
   * Returns the object resolved by the fragment segment.
   * <p>
   * The fragment segment encoding will typically be of the form returned by {@link #eURIFragmentSegment eURIFragmentSegment}.
   * @param uriFragment a fragment segment.
   * @return the fragment segment that resolves to the given object in this object's given feature.
   */
  EObject eObjectForURIFragmentSegment(String uriFragmentSegment);

  /**
   * Sets the meta class.
   * An object with a set meta class is, by definition, a dynamic instance;
   * it will support dynamic {@link #eSetting settings}.
   * an.
   * @param eClass the dynamic meta class.
   * @see #eSetting(EStructuralFeature)
   */
  void eSetClass(EClass eClass);

  /**
   * Returns a setting that can be used to access this object's feature.
   * @param feature the feature of the desired setting.
   * @return a setting that can be used to access this object's feature.
   */
  EStructuralFeature.Setting eSetting(EStructuralFeature feature);

  /**
   * Returns the feature ID relative to the base class, given a feature ID relative to this derived object's actual class.
   * @param derivedFeatureID the ID in the actual derived class
   * @param baseClass the base class for which a relative ID is desired.
   * @return the up-cast feature ID.
   */
  int eBaseStructuralFeatureID(int derivedFeatureID, Class baseClass);

  /**
   * Returns the container feature ID.
   * If the container isn't a navigable feature, this will be a negative ID indicating the inverse of the containment feature's ID.
   * @return the container feature ID.
   * @see #eContainmentFeature()
   * @see #EOPPOSITE_FEATURE_BASE
   */
  int eContainerFeatureID();

  /**
   * Returns the feature ID relative to this derived object's actual class, given a feature ID relative to the given base class.
   * @param baseFeatureID the ID relative to a base class.
   * @param baseClass the base class to which the ID is relative.
   * @return the down-cast feature ID.
   */
  int eDerivedStructuralFeatureID(int baseFeatureID, Class baseClass);

  /**
   * Sets this object to be directly contained by the resource 
   * and returns accumulated notifications.
   * This is only done as the inverse of {@link Resource#getContents()}<code>.add(this)</code>.
   * @return accumulated notifications.
   */
  NotificationChain eSetResource(Resource.Internal resource, NotificationChain notifications);

  /**
   * Adds the object at the other end of a bidirectional reference to the appropriate feature
   * and returns accumulated notifications.
   * @return accumulated notifications.
   */
  NotificationChain eInverseAdd(InternalEObject otherEnd, int featureID, Class baseClass, NotificationChain notifications);

  /**
   * Removes the object at the other end of a bidirectional reference from the appropriate feature
   * and returns accumulated notifications.
   * @return accumulated notifications.
   */
  NotificationChain eInverseRemove(InternalEObject otherEnd, int featureID, Class baseClass, NotificationChain notifications);

  /**
   * Sets the container to be new container and appropriate feature.
   * and returns accumulated notifications.
   * Since the container feature may not be navigable, i.e., may not exist, the containment feature may be encoded instead.
   * @return accumulated notifications.
   */
  NotificationChain eBasicSetContainer(InternalEObject newContainer, int newContainerFeatureID, NotificationChain notifications);

  /**
   * Removes this object from whatever container holds it,
   * and returns accumulated notifications.
   * @return accumulated notifications.
   */
  NotificationChain eBasicRemoveFromContainer(NotificationChain notifications);

  /**
   * Returns the proxy URI of this object.
   * It can be used to {@link org.eclipse.emf.ecore.util.EcoreUtil#resolve resolve} to the actual object.
   * @return the proxy URI of this object.
   * @see org.eclipse.emf.ecore.util.EcoreUtil#resolve(EObject, ResourceSet)
   * @see Resource#unload
   */
  URI eProxyURI();

  /**
   * Set the proxy URI of this object.
   * It will be used to {@link org.eclipse.emf.ecore.util.EcoreUtil#resolve resolve} to the actual object.
   * @param uri the URI.
   * @see org.eclipse.emf.ecore.util.EcoreUtil#resolve(EObject, ResourceSet)
   * @see Resource#unload
   */
  void eSetProxyURI(URI uri);

  /**
   * Returns the containing internal resource, or <code>null</code>.
   * @return the containing internal resource.
   * @see #eResource
   */
  Resource.Internal eInternalResource();
}
