/**
 * <copyright> 
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.codegen.ecore/src/org/eclipse/emf/codegen/ecore/rose2ecore/parser/RoseNode.java, emf.codegen.ecore, org.eclipse.111, 20031120_1149WL
 * @version 1.18 11/20/03
 */
package org.eclipse.emf.codegen.ecore.rose2ecore.parser;


import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.emf.codegen.ecore.rose2ecore.RoseStrings;


/**
 * Is used to build a tree for Rose file.
 */
public class RoseNode 
{
  public final static int STRING       = 0;
  public final static int STRING_SEQ   = 1;
  public final static int OBJECT       = 2;
  public final static int LIST         = 3;
  public final static int VALUE        = 4;

  protected String key;
  protected String value;
  protected RoseNode parent;
  protected List nodes = new ArrayList();
  protected int type;
  protected String id;
  protected boolean commit = true;
  protected Object node;

  public RoseNode(String key, String value, int type)
  {
    this.key = key;
    this.value = value;
    this.type = type;
  }

  public void setParent(RoseNode parent)
  {
    this.parent = parent;
  }

  public RoseNode getParent()
  {
    return parent;
  }

  public RoseNode getRoot()
  {
    RoseNode result = this;
    for (RoseNode root = this.getParent(); root != null; root = root.getParent())
    {
      result = root;
    }
    return result;
  }

  public String getKey()
  {
    return key;
  }

  public String getValue()
  {
    return value;
  }

  public String getAllValues()
  {
    return getAllValues(false);
  }

  public String getAllValues(boolean preserveSpace)
  {
    if (type == STRING_SEQ) 
    {
      StringBuffer temp = new StringBuffer();
      for (int i = 0; i < nodes.size(); i++) 
      {
        RoseNode n = (RoseNode) nodes.get(i);
        temp.append(n.getValue());
        if (preserveSpace && i < nodes.size() - 1)
        {
          temp.append('\n');
        }
      }
      return temp.toString();
    } 
    else 
    {
      return preserveSpace ? Util.trimQuotes(value) : value;
    }
  }

  public String getFilteredValues(String filter)
  {
    if (type == STRING_SEQ) 
    {
      StringBuffer temp = new StringBuffer();
      for (int i = 0; i < nodes.size(); i++) 
      {
        RoseNode n = (RoseNode) nodes.get(i);
        if (i == 0 && Util.getWord(n.getValue(), 1).equals(filter))
        {
          continue;
        }
        temp.append(n.getValue());
        if (i < nodes.size() - 1)
        {
          temp.append('\n');
        }
      }
      return temp.toString();
    } 
    else 
    {
      String val = Util.trimQuotes(value);
      return Util.getWord(val, 1).equals(filter) ? "" : val;
    }
  }

  public void setValue(String value)
  {
    this.value = value;
  }

  public boolean getCommit()
  {
    return commit;
  }

  public void setCommit(boolean b)
  {
    commit = b;
  }

  public String getId()
  {
    return id;
  }

  public String getAtId()
  {
    int ind = value.lastIndexOf("@");
    return ind != -1 ?  value.substring(ind+1) : null;
  }

  public void setId(String id)
  {
    this.id = id;
  }

  public int getRoseNodeType()
  {
    return type;
  }

  public List getNodes()
  {
    return nodes;
  }

  public void addNode(RoseNode n)
  {
    n.parent = this;
    nodes.add(n);
  }

  public void deleteNode(RoseNode n)
  {
    if (n != null) 
    {
      nodes.remove(n);
    }
  }


  public RoseNode findNodeWithKey(String key)
  {
    for (int i = 0; i < nodes.size(); i++) 
    {
      RoseNode node = (RoseNode) nodes.get(i);
      if (key.equals(node.getKey()))
      {
        return node;
      }
    }
    return null;
  }

  public RoseNode findNodeWithValue(String value)
  {
    for (int i = 0; i < nodes.size(); i++) 
    {
      RoseNode node = (RoseNode) nodes.get(i);
      if (value.equals(node.getValue()))
      {
        return node;
      }
    }
    return null;
  }

  public RoseNode findNodeWithWord(String w)
  {
    for (int i = 0; i < nodes.size(); i++) 
    {
      RoseNode node = (RoseNode) nodes.get(i);
      String value = node.getValue();
      StringTokenizer st = new StringTokenizer(value);
      while (st.hasMoreTokens()) 
      {
        String tok = st.nextToken();
        if (tok.equals(w))
        {
          return node;
        }
      }
    }
    return null;
  }

  public String getRoseId()
  {
    RoseNode node = findNodeWithKey("quid");
    return node != null ? node.getValue() : null;
  }

  public String getRoseRefId()
  {
    RoseNode node = findNodeWithKey("quidu");
    return node != null ? node.getValue() : null;
  }

  public String getRoseSupplier()
  {
    RoseNode node = findNodeWithKey("supplier");
    return node != null ? Util.trimQuotes(node.getValue()) : null;
  }

  public void setNode(Object node)
  {
    this.node = node;
  }

  public Object getNode()
  {
    return node;
  }

  public String getType()
  {
    String attributeValue = getAttributeValue(RoseStrings.TYPE);
    return attributeValue;
  }

  public String getStereotype()
  {
    String attributeValue = getAttributeValue(RoseStrings.STEREOTYPE);
    return attributeValue;
  }

  public boolean isDerived()
  {
    String attributeValue = getAttributeValue(RoseStrings.DERIVED);
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isAbstract()
  {
    String attributeValue = getAttributeValue(RoseStrings.ABSTRACT);
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isNavigable()
  {
    String attributeValue = getAttributeValue(RoseStrings.IS_NAVIGABLE);
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isAggregate()
  {
    String attributeValue = getAttributeValue(RoseStrings.IS_AGGREGATE);
    return "true".equalsIgnoreCase(attributeValue);
  }

  public String getContainment()
  {
    String attributeValue = getAttributeValue(RoseStrings.CONTAINMENT);
    return attributeValue;
  }

  public String getConstraints()
  {
    String attributeValue = getAttributeValue(RoseStrings.CONSTRAINTS);
    return attributeValue;
  }

  public String getResult()
  {
    String attributeValue = getAttributeValue(RoseStrings.RESULT);
    return attributeValue;
  }

  public String getInitV()
  {
    String attributeValue = getAttributeValue(RoseStrings.INITV);
    return attributeValue;
  }

  public String getDocumentation()
  {
    String attributeValue = getAttributeValue(RoseStrings.DOCUMENTATION);
    return attributeValue;
  }

  public boolean isLoaded()
  {
    String attributeValue = getAttributeValue(RoseStrings.IS_LOADED);
    return attributeValue == null || "true".equalsIgnoreCase(attributeValue);
  }

  protected String getAttributeValue(String key)
  {
    for (Iterator i = getNodes().iterator(); i.hasNext(); )
    {
      RoseNode roseNode = (RoseNode)i.next();
      if (roseNode.getRoseNodeType() == RoseNode.STRING) 
      {
        String nodeKey = roseNode.getKey();
        String nodeValue = roseNode.getValue();
        if (nodeKey.equals(key)) 
        {
          nodeValue = dequote(nodeValue);
          return nodeValue;
        }
      }
      else if (roseNode.getRoseNodeType() == RoseNode.STRING_SEQ && roseNode.getKey().equals(key)) 
      {
        List subNodes = roseNode.getNodes();
        StringBuffer result = new StringBuffer();
        for (Iterator j = subNodes.iterator(); j.hasNext(); )
        {
          RoseNode subNode = (RoseNode)j.next();
          if (subNode.getRoseNodeType() == RoseNode.STRING) 
          {
            if (subNode.getValue().equals(""))
            {
              result.append(" ");
            }
            else 
            {
              if (result.length() != 0)
              {
                result.append(" ");
              }
              result.append(subNode.getValue());
            }
          }
        }
        return result.toString();
      }
    }

    return null;
  }

  public String getUML2MOFCorbaType()
  {
    String attributeValue = getAttributeValue("MOF", "uml2mof.corbaType");
    return attributeValue;
  }

  public boolean isTransient()
  {
    String attributeValue = getAttributeValue("Ecore", "isTransient");
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isVolatile()
  {
    String attributeValue = getAttributeValue("Ecore", "isVolatile");
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isChangeable()
  {
    String attributeValue = getAttributeValue("Ecore", "isChangeable");
    return !"false".equalsIgnoreCase(attributeValue);
  }

  public boolean isResolveProxies()
  {
    String attributeValue = getAttributeValue("Ecore", "isResolveProxies");
    return !"false".equalsIgnoreCase(attributeValue);
  }

  public boolean isUnsettable()
  {
    String attributeValue = getAttributeValue("Ecore", "isUnsettable");
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isID()
  {
    String attributeValue = getAttributeValue("Ecore", "isID");
    return "true".equalsIgnoreCase(attributeValue);
  }

  public boolean isUnique()
  {
    String attributeValue = getAttributeValue("Ecore", "isUnique");
    return !"false".equalsIgnoreCase(attributeValue);
  }

  public String getBasePackage()
  {
    String attributeValue = getAttributeValue("Ecore", "basePackage");
    return attributeValue;
  }

  public String getPrefix()
  {
    String attributeValue = getAttributeValue("Ecore", "prefix");
    return attributeValue;
  }

  public String getPackageName()
  {
    String attributeValue = getAttributeValue("Ecore", "packageName");
    return attributeValue;
  }

  public String getClassifierName()
  {
    String attributeValue = getAttributeValue("Ecore", "classifierName");
    return attributeValue;
  }

  public String getReferenceName()
  {
    String attributeValue = getAttributeValue("Ecore", "referenceName");
    return attributeValue;
  }

  public String getAttributeName()
  {
    String attributeValue = getAttributeValue("Ecore", "attributeName");
    return attributeValue;
  }

  public String getOperationName()
  {
    String attributeValue = getAttributeValue("Ecore", "operationName");
    return attributeValue;
  }

  public String getNsPrefix()
  {
    String attributeValue = getAttributeValue("Ecore", "nsPrefix");
    if (attributeValue == null || attributeValue.length() == 0)
    {
      attributeValue = getAttributeValue("Ecore", "nsName");
    }
    return attributeValue;
  }

  public String getNsURI()
  {
    String attributeValue = getAttributeValue("Ecore", "nsURI");
    return attributeValue;
  }

  /**
   * This provides backwards compatibility for the renaming of "eCore" to
   * "Ecore".
   */
  protected String getAttributeValue(String tool, String name)
  {
    String result = basicGetAttributeValue(tool, name);
    if (result == null && "Ecore".equals(tool))
    {
      result = basicGetAttributeValue("eCore", name);
    }
    return result;
  }

  protected String basicGetAttributeValue(String tool, String name)
  {
    RoseNode attributeListNode = findNodeWithKey(RoseStrings.ATTRIBUTES);

    String value = getAttributeValueInAttributeList(attributeListNode, tool, name);
    
    if (value == null) 
    {
      value = getAttributeValueInAttributeList(getDefaultAttributeList(tool), tool, name);
    }

    return value;
  }

  private String getAttributeValueInAttributeList(RoseNode attributeListNode, String tool, String name)
  {
    if (attributeListNode != null) 
    {
      List attributeNodes = attributeListNode.getNodes();
      if (attributeNodes != null) 
      {
        for (Iterator i = attributeNodes.iterator(); i.hasNext(); )
        {
          RoseNode attributeNode = (RoseNode)i.next();
          List nodes = attributeNode.getNodes();
          if (nodes != null) 
          {
            String setName = "";
            String tagName = "";
            String valueName = "";
            for (Iterator j = nodes.iterator(); j.hasNext(); )
            {
              RoseNode node = (RoseNode)j.next();
              if (node.getRoseNodeType() == RoseNode.STRING) 
              {
                String stringV = dequote(node.getValue());
                if (node.getKey().equals(RoseStrings.TOOL)) 
                {
                  setName = stringV;
                } 
                else if (node.getKey().equals(RoseStrings.NAME)) 
                {
                  tagName = stringV;
                } 
                else if (node.getKey().equals(RoseStrings.VALUE)) 
                {
                  valueName = stringV;
                }
              } 
              else if (node.getRoseNodeType() == RoseNode.VALUE) 
              {
                List stringNodes = node.getNodes();
                if (stringNodes != null && stringNodes.size() == 1) 
                {
                  RoseNode stringNode = (RoseNode)stringNodes.get(0);
                  String stringV = dequote(stringNode.getValue());
                  if (node.getKey().equals(RoseStrings.TOOL)) 
                  {
                    setName = stringV;
                  } 
                  else if (node.getKey().equals(RoseStrings.NAME)) 
                  {
                    tagName = stringV;
                  } 
                  else if (node.getKey().equals(RoseStrings.VALUE)) 
                  {
                    valueName = stringV;
                  }
                }
              }
            }
            if (setName.equals(tool) && tagName.equals(name))
            {
              return valueName;
            }
          } 
        }
      }
    }
    return null;
  }

  private RoseNode getDefaultAttributeList(String tool)
  {
    RoseNode attributeListNode = null;
    RoseNode rootNode = getRoot();
    String rootNodeValue = rootNode.getValue();
    String rootObjectKey  = rootNode.getKey();
    String rootObjectType = Util.getType(rootNodeValue);

    if (rootObjectType.equals(RoseStrings.DESIGN))
    {
      String objectType = Util.getType(getValue());
      RoseNode defaultProperties = rootNode.findNodeWithKey(RoseStrings.PROPERTIES);
      if (defaultProperties != null)
      {
        RoseNode attributes = defaultProperties.findNodeWithKey(RoseStrings.ATTRIBUTES);
        for (Iterator i = attributes.getNodes().iterator(); i.hasNext(); )
        {
          RoseNode attribute = (RoseNode)i.next();
          RoseNode toolNode = attribute.findNodeWithKey(RoseStrings.TOOL);
          if (toolNode != null && Util.trimQuotes(toolNode.getValue()).equals(tool))
          {
            RoseNode nameNode = attribute.findNodeWithKey(RoseStrings.NAME);
            String theName =  Util.trimQuotes(nameNode.getValue());
            if (objectType.equals(RoseStrings.CLASSATTRIBUTE) && theName.equals(RoseStrings.DEFAULT_ATTRIBUTE))
            {
              attributeListNode = attribute.findNodeWithKey(RoseStrings.VALUE);
              break;
            }
            else if (objectType.equals(RoseStrings.CLASS) && theName.equals(RoseStrings.DEFAULT_CLASS))
            {
              attributeListNode = attribute.findNodeWithKey(RoseStrings.VALUE);
              break;
            }
            else if (objectType.equals(RoseStrings.CLASS_CATEGORY) && theName.equals(RoseStrings.DEFAULT_CATEGORY))
            {
              attributeListNode = attribute.findNodeWithKey(RoseStrings.VALUE);
              break;
            }
            else if (objectType.equals(RoseStrings.OPERATION) && theName.equals(RoseStrings.DEFAULT_OPERATION))
            {
              attributeListNode = attribute.findNodeWithKey(RoseStrings.VALUE);
              break;
            }
            else if (objectType.equals(RoseStrings.ROLE) && theName.equals(RoseStrings.DEFAULT_ROLE))
            {
              attributeListNode = attribute.findNodeWithKey(RoseStrings.VALUE);
              break;
            }
          }
        }
      }
    }
    return attributeListNode;
  }




  public String getRoleMultiplicity()
  {
    List nodes = getNodes();
    for (int i=0; i<nodes.size(); i++) 
    {
      RoseNode node = (RoseNode)nodes.get(i);
      if (node.getRoseNodeType() == RoseNode.VALUE) 
      {
        String objKey  = node.getKey();
        String objType = Util.getType(node.getValue());
        String objName = Util.getName(node.getValue());
        if (objKey.equals(RoseStrings.CLIENT_CARDINALITY) && objType.equals(RoseStrings.CARDINALITY)) 
        {
          List subNodes = node.getNodes();
          String value = "";
          for (int j=0; j<subNodes.size(); j++) 
          {
            // size should be 1
            RoseNode subNode = (RoseNode)subNodes.get(j);
            if (subNode.getRoseNodeType() == RoseNode.STRING) 
            {
              value = subNode.getValue();
              value = value.substring(1, value.length()-1);
            } 
            else if (subNode.getRoseNodeType() == RoseNode.STRING_SEQ) 
            {
              // could this happen?
              List subSubNodes = subNode.getNodes();
              for (int k=0; k<subSubNodes.size(); k++) 
              {
                RoseNode subSubNode = (RoseNode)subSubNodes.get(k);
                if (subSubNode.getRoseNodeType() == RoseNode.STRING)
                  value = subSubNode.getValue() + "|#" + value;
              }
            }
          }
          return value;
        }
      }
    }
    return null;
  }

  protected static String dequote(String s)
  {
    if (s != null && s.length() >= 2 && s.charAt(0) == '\"' && s.charAt(s.length() - 1) == '\"')
    {
      s = s.substring(1, s.length() - 1);
      for (int i = s.indexOf("\\\\"); i != -1; i = s.indexOf("\\\\"))
      {
        s = s.substring(0, i) + s.substring(i + 1, s.length());
      }
    }
    return s;
  }
}
