/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.emf.ecore.xmi/src/org/eclipse/emf/ecore/xmi/impl/XMLHandler.java, emf.ecore.xmi, org.eclipse.111, 20031120_1149WL
 * @version 1.30.1.1 11/20/03
 */
package org.eclipse.emf.ecore.xmi.impl;

import java.io.InputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;

import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.InternalEObject;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.emf.ecore.util.InternalEList;

import org.eclipse.emf.ecore.xmi.ClassNotFoundException;
import org.eclipse.emf.ecore.xmi.FeatureNotFoundException;
import org.eclipse.emf.ecore.xmi.IllegalValueException;
import org.eclipse.emf.ecore.xmi.PackageNotFoundException;
import org.eclipse.emf.ecore.xmi.UnresolvedReferenceException;
import org.eclipse.emf.ecore.xmi.XMIException;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.XMLHelper;
import org.eclipse.emf.ecore.xmi.XMLResource;

/**
 * This class is a generic interface for loading XML files and
 * creating EObjects from them. Its subclasses include the SAXXMLHandler
 * class, which wraps this class in a SAX default handler.
 */
public abstract class XMLHandler
{
  protected static final String ERROR_TYPE = "error";
  protected static final String OBJECT_TYPE = "object";

  protected final static String TYPE_ATTRIB = XMIResource.XSI_NS + ":" + XMIResource.TYPE;
  protected final static String NIL_ATTRIB = XMIResource.XSI_NS + ":" + XMIResource.NIL;
  protected final static String SCHEMA_LOCATION_ATTRIB = XMIResource.XSI_NS + ":" + XMIResource.SCHEMA_LOCATION;
  protected final static String NO_NAMESPACE_SCHEMA_LOCATION_ATTRIB = XMIResource.XSI_NS + ":" + XMIResource.NO_NAMESPACE_SCHEMA_LOCATION;

  protected static class MyStack extends ArrayList
  {
    public MyStack()
    {
    }

    public Object peek()
    {
      int size = size();
      return size == 0 ? null : get(size - 1);
    }

    public void push(Object o)
    {
      add(o);
    }

    public Object pop()
    {
      int size = size();
      return size == 0 ?  null : remove(size - 1);
    }
  }

  /**
   * For unresolved forward references, the line number where the incorrect id
   * appeared in an XML resource is needed, so the Value for the forward reference
   * and the line number where the forward reference occurred must be saved until
   * the end of the XML resource is encountered.
   */
  protected static class SingleReference
  {
    private EObject object;
    private EStructuralFeature feature;
    private Object value;
    private int position;
    private int lineNumber;
    private int columnNumber;

    public SingleReference(EObject object,
                    EStructuralFeature feature,
                    Object value,
                    int position,
                    int lineNumber,
                    int columnNumber)
    {
      this.object       = object;
      this.feature      = feature;
      this.value        = value;
      this.position     = position;
      this.lineNumber   = lineNumber;
      this.columnNumber = columnNumber;
    }

    public EObject getObject()
    {
      return object;
    }

    public EStructuralFeature getFeature()
    {
      return feature;
    }

    public Object getValue()
    {
      return value;
    }

    public int getPosition()
    {
      return position;
    }

    public int getLineNumber()
    {
      return lineNumber;
    }

    public int getColumnNumber()
    {
      return columnNumber;
    }
  }

  protected static class ManyReference implements XMLHelper.ManyReference
  {
    private EObject object;
    private EStructuralFeature feature;
    private Object[] values;
    private int[] positions;
    private int lineNumber;
    private int columnNumber;

    ManyReference(EObject object,
                  EStructuralFeature feature,
                  Object[] values,
                  int[] positions,
                  int lineNumber,
                  int columnNumber)
    {
      this.object       = object;
      this.feature      = feature;
      this.values       = values;
      this.positions    = positions;
      this.lineNumber   = lineNumber;
      this.columnNumber = columnNumber;
    }

    public EObject getObject()
    {
      return object;
    }

    public EStructuralFeature getFeature()
    {
      return feature;
    }

    public Object[] getValues()
    {
      return values;
    }

    public int[] getPositions()
    {
      return positions;
    }

    public int getLineNumber()
    {
      return lineNumber;
    }

    public int getColumnNumber()
    {
      return columnNumber;
    }
  }

  protected XMLResource xmlResource;
  protected XMLHelper helper;
  protected MyStack elements;
  protected MyStack objects;
  protected MyStack types;
  protected Map prefixesToFactories;
  protected Map urisToLocations;
  protected InternalEList extent;
  protected ResourceSet resourceSet;
  protected URI resourceURI;
  protected boolean resolve;
  protected boolean oldStyleProxyURIs;
  protected boolean disableNotify;
  protected StringBuffer text;
  protected List sameDocumentProxies;
  protected List forwardSingleReferences;
  protected List forwardManyReferences;
  protected Object[] identifiers;
  protected int[] positions;
  protected static final int ARRAY_SIZE = 64;
  protected static final int REFERENCE_THRESHOLD = 5;
  protected int capacity;
  protected Set notFeatures;
  protected String idAttribute;
  protected XMLResource.XMLMap xmlMap;

  /**
   */
  public XMLHandler(XMLResource xmlResource, XMLHelper helper, Map options)
  {
    this.xmlResource = xmlResource;
    this.helper = helper;
    elements = new MyStack();
    objects  = new MyStack();
    types    = new MyStack();
    prefixesToFactories = new HashMap();
    forwardSingleReferences = new LinkedList();
    forwardManyReferences   = new LinkedList();
    sameDocumentProxies = new LinkedList();
    identifiers = new Object[ARRAY_SIZE];
    positions   = new int[ARRAY_SIZE];
    capacity    = ARRAY_SIZE;
    resourceSet  = xmlResource.getResourceSet();
    resourceURI  = xmlResource.getURI();
    extent       = (InternalEList) xmlResource.getContents();
    resolve      = resourceURI.isHierarchical() && !resourceURI.isRelative();

    if (Boolean.TRUE.equals(options.get(XMLResource.OPTION_DISABLE_NOTIFY)))
      disableNotify = true;

    notFeatures = new HashSet();
    notFeatures.add(TYPE_ATTRIB);
    notFeatures.add(SCHEMA_LOCATION_ATTRIB);
    notFeatures.add(NO_NAMESPACE_SCHEMA_LOCATION_ATTRIB);

    xmlMap = (XMLResource.XMLMap) options.get(XMLResource.OPTION_XML_MAP);
    helper.setXMLMap(xmlMap);

    if (xmlMap != null)
    {
      EPackage pkg = xmlMap.getNoNamespacePackage();

      if (pkg != null)
      {
        helper.setNoNamespacePackage(pkg);
      }

      idAttribute = xmlMap.getIDAttributeName();
    }
  }

  /**
   * Process the XML attributes for the newly created object.
   */
  protected abstract void handleObjectAttribs(EObject obj);

  /**
   * Process the XML namespace declarations.
   */
  protected abstract void handleNamespaceAttribs();

  /**
   * Returns true if the xsi:nil attribute is in the list of attributes.
   */
  protected abstract boolean isNull();

  /**
   * Sets the current attributes and returns the old ones.
   */
  protected abstract Object setAttributes(Object attributes);

  /**
   * Sets the object that might be used for determining the line and
   * column number.
   */
  protected abstract void setLocator(Object locator);

  /**
   * Returns the xsi type attribute's value.
   */
  protected abstract String getXSIType();

  public void startDocument()
  {
  }

  /**
   * This method determines whether to make an object or not, then makes an
   * object based on the XML attributes and the metamodel.
   */
  public void startElement(String uri, String localName, String name)
  {
    elements.push(name);

    handleNamespaceAttribs();

    // Turning on namespace awareness is very expensive so this simulates the effect.
    //
    int index = name.indexOf(":");
    String prefix = "";
    localName = name;
    if (index != -1)
    {
      prefix    = name.substring(0, index);
      localName = name.substring(index + 1);
    }

    processElement(name, prefix, localName);
  }

  protected void processElement(String name, String prefix, String localName)
  {
    if (isError())
    {
      types.push(ERROR_TYPE);
    }
    else
    {
      if (objects.isEmpty())
      {
        createTopObject(prefix, localName);
      }
      else
      {
        handleFeature(prefix, localName);
      }
    }
  }

  /**
   * Check if the values of the forward references have been set (they may
   * have been set due to a bi-directional reference being set).  If not,
   * set them.
   */
  public void endDocument()
  {
    // Handle the same document proxies, which may have problems resulting from the
    // other end of a bidirectional reference being handled as an IDREF rather than as a proxy.
    // When we are done with these, we know that funny proxies are now resolved as if they were handled as IDREFs.
    //
    for (Iterator i = sameDocumentProxies.iterator(); i.hasNext(); )
    {
      InternalEObject proxy = (InternalEObject)i.next();

      // Look through all the references...
      //
      LOOP:
      for (Iterator j = proxy.eClass().getEAllReferences().iterator(); j.hasNext(); )
      {
        // And find the one that holds this proxy.
        //
        EReference eReference = (EReference)j.next();
        EReference oppositeEReference = eReference.getEOpposite();
        if (oppositeEReference != null && proxy.eIsSet(eReference))
        {
          // Try to resolve the proxy locally.
          //
          EObject resolvedEObject = xmlResource.getEObject(proxy.eProxyURI().fragment());
          if (resolvedEObject != null)
          {
            // Compute the holder of the proxy
            //
            EObject proxyHolder = (EObject)(eReference.isMany() ? ((List)proxy.eGet(eReference)).get(0) : proxy.eGet(eReference));

            // If the proxy holder can hold many values,
            // it may contain a duplicate that resulted when the other end was processed as an IDREF
            // and hence did both sides of the bidirectional relation.
            //
            if (oppositeEReference.isMany())
            {
              // So if the resolved object is also present...
              //
              InternalEList holderContents = (InternalEList)proxyHolder.eGet(oppositeEReference);
              List basicHolderContents = holderContents.basicList();
              int resolvedEObjectIndex = basicHolderContents.indexOf(resolvedEObject);
              if (resolvedEObjectIndex != -1)
              {
                // Move the resolved object to the right place, remove the proxy, and we're done.
                //
                int proxyIndex = basicHolderContents.indexOf(proxy);
                holderContents.move(proxyIndex, resolvedEObjectIndex);
                holderContents.remove(proxyIndex > resolvedEObjectIndex ? proxyIndex  - 1 : proxyIndex + 1);
                break LOOP;
              }
            }

            // If the resolved object doesn't contain a reference to the proxy holder as it should.
            //
            if (eReference.isMany() ?
                  !((InternalEList)resolvedEObject.eGet(eReference)).basicList().contains(proxyHolder) :
                  resolvedEObject.eGet(eReference) != proxyHolder)
            {
              // The proxy needs to be replaced in a way that updates both ends of the reference.
              //
              if (oppositeEReference.isMany())
              {
                InternalEList proxyHolderList = (InternalEList)proxyHolder.eGet(oppositeEReference);
                proxyHolderList.setUnique(proxyHolderList.indexOf(proxy), resolvedEObject);
              }
              else
              {
                proxyHolder.eSet(oppositeEReference, resolvedEObject);
              }
            }
          }

          break;
        }
      }
    }

    for (Iterator i = forwardSingleReferences.iterator(); i.hasNext(); )
    {
      SingleReference ref = (SingleReference) i.next();
      EObject obj = (EObject) xmlResource.getEObject((String) ref.getValue());

      if (obj != null)
      {
        EStructuralFeature feature = ref.getFeature();
        setFeatureValue(ref.getObject(), feature, obj, ref.getPosition());
      }
      else
      {
        error
          (new UnresolvedReferenceException
            ((String) ref.getValue(),
             xmlResource.getURI().toString(),
             ref.getLineNumber(),
             ref.getColumnNumber()));
      }
    }
    for (Iterator i = forwardManyReferences.iterator(); i.hasNext(); )
    {
      ManyReference ref = (ManyReference) i.next();
      Object[] values = ref.getValues();

      for (int j = 0, l = values.length; j < l; j++)
      {
        String id = (String) values[j];
        EObject obj = (EObject) xmlResource.getEObject(id);
        values[j] = obj;

        if (obj == null)
          error
            (new UnresolvedReferenceException
              ((String) id,
               xmlResource.getURI().toString(),
               ref.getLineNumber(),
               ref.getColumnNumber()));
      }

      setFeatureValues(ref);
    }

    if (disableNotify) {
      for (Iterator i = xmlResource.getAllContents(); i.hasNext(); )
      {
        EObject eObject = (EObject)i.next();
        eObject.eSetDeliver(true);
      }
    }
  }

  /**
   * Create a top object based on the prefix and name.
   */
  protected void createTopObject(String prefix, String name)
  {
    EFactory eFactory = getFactoryForPrefix(prefix);

    if (eFactory == null && prefix.equals("") && helper.getURI(prefix) == null)
    {
      error
        (new PackageNotFoundException
           (null,
            getLocation(),
            getLineNumber(),
            getColumnNumber()));
    }

    EObject newObject = createObjectFromFactory(eFactory, name);
    processTopObject(newObject);
  }

  /**
   * Add object to extent and call processObject.
   */
  protected void processTopObject(EObject object)
  {
    if (object != null)
    {
      extent.addUnique(object);
    }
    processObject(object);
  }

  /**
   * Pop the appropriate stacks and set features whose values are in
   * the content of XML elements.
   */
  public void endElement(String uri, String localName, String name)
  {
    elements.pop();
    Object type = types.pop();

    if (type == OBJECT_TYPE)
    {
      if (text != null)
      {
        handleProxy((InternalEObject)objects.pop(), text.toString().trim());
        text = null;
      }
      else
      {
        objects.pop();
      }
    }
    else if (isTextFeatureValue(type))
    {
      EObject eObject = (EObject)objects.pop();
      if (eObject == null)
      {
        eObject = (EObject)objects.peek();
      }
      setFeatureValue(eObject, (EStructuralFeature) type, text == null ? null : text.toString());
      text = null;
    }
  }

  protected boolean isTextFeatureValue(Object type)
  {
    return type != ERROR_TYPE;
  }

  public void startPrefixMapping(String prefix, String uri)
  {
    helper.addPrefix(prefix, uri);
  }

  public void endPrefixMapping(String prefix)
  {
  }

  public void characters(char [] ch, int start, int length)
  {
    if (text != null)
    {
      text.append(ch, start, length);
    }
  }

  protected void handleXMLNSAttribute(String attrib, String value)
  {
    // Handle namespaces
    int index = attrib.indexOf(":");
    String prefix = index == -1 ? "" : attrib.substring(index + 1);
    helper.addPrefix(prefix, value);
  }

  protected void handleXSISchemaLocation(String schemaLocations)
  {
    if (urisToLocations == null)
    {
      urisToLocations = new HashMap();
      if (xmlResource != null)
      {
        xmlResource.getDefaultSaveOptions().put(XMLResource.OPTION_SCHEMA_LOCATION, Boolean.TRUE);
      }
    }

    for (StringTokenizer stringTokenizer = new StringTokenizer(schemaLocations, " "); stringTokenizer.hasMoreTokens(); )
    {
      String key = stringTokenizer.nextToken();
      if (stringTokenizer.hasMoreTokens())
      {
        String value = stringTokenizer.nextToken();
        URI uri = URI.createURI(value);
        if (resolve && uri.isRelative() && uri.hasRelativePath())
        {
          uri = helper.resolve(uri, resourceURI);
        }
        urisToLocations.put(key, uri);
      }
    }
  }

  protected void handleXSINoNamespaceSchemaLocation(String noNamespaceSchemaLocation)
  {
    if (urisToLocations == null)
    {
      urisToLocations = new HashMap();
      if (xmlResource != null)
      {
        xmlResource.getDefaultSaveOptions().put(XMLResource.OPTION_SCHEMA_LOCATION, Boolean.TRUE);
      }
    }

    if (helper.getNoNamespacePackage() == null)
    {
      URI uri = URI.createURI(noNamespaceSchemaLocation);
      if (resolve && uri.isRelative() && uri.hasRelativePath())
      {
        uri = helper.resolve(uri, resourceURI);
      }

      helper.setNoNamespacePackage(getPackageForURI(uri.toString()));
    }
  }

  /**
   * The XML element represents a feature. There are two
   * cases to handle:
   *   1. The feature has a type that is a datatype.
   *   2. The feature has a type that is a class.
   */
  protected void handleFeature(String prefix, String name)
  {
    EObject peekObject = (EObject) objects.peek();
    EStructuralFeature feature = getFeature(peekObject, prefix, name);
    if (feature != null)
    {
      int kind = helper.getFeatureKind(feature);

      if (kind == XMLHelper.DATATYPE_SINGLE || kind == XMLHelper.DATATYPE_IS_MANY)
      {
        objects.push(null);
        types.push(feature);
        if (!isNull())
        {
          text = new StringBuffer();
        }
      }
      else
      {
        createObject(peekObject, feature);
      }
    }
    else
    {
      // Try to get a general-content feature.
      // Use a pattern that's not possible any other way.
      //
      if ((feature = getFeature(peekObject, null, "")) != null)
      {
        EFactory eFactory = getFactoryForPrefix(prefix);

        // This is for the case for a local unqualified element that has been bound.
        //
        if (eFactory == null)
        {
          eFactory = feature.getEContainingClass().getEPackage().getEFactoryInstance();
        }

        EObject newObject = createObjectFromFactory(eFactory, name);
        if (newObject != null)
        {
          setFeatureValue(peekObject, feature, newObject);
        }
        processObject(newObject);
      }
      else
      {
        // This handles the case of a substitution group.
        //
        if (xmlMap != null)
        {
          EFactory eFactory = getFactoryForPrefix(prefix);
          EObject newObject = createObjectFromFactory(eFactory, name);
          if (newObject != null)
          {
            for (Iterator i = peekObject.eClass().getEAllReferences().iterator(); i.hasNext(); )
            {
              EReference eReference = (EReference)i.next();
              if (eReference.getEType().isInstance(newObject))
              {
                setFeatureValue(peekObject, eReference, newObject);
                processObject(newObject);
                return;
              }
            }
          }
        }

        error
          (new FeatureNotFoundException
            (name,
             peekObject,
             getLocation(),
             getLineNumber(),
             getColumnNumber()));
      }
    }
  }

  protected int getLineNumber()
  {
    return -1;
  }

  protected int getColumnNumber()
  {
    return -1;
  }

  protected String getLocation()
  {
    return xmlResource.getURI().toString();
  }

  public void error(XMIException e)
  {
    xmlResource.getErrors().add(e);
  }

  public void warning(XMIException e)
  {
    xmlResource.getWarnings().add(e);
  }

  public void fatalError(XMIException e)
  {
    xmlResource.getErrors().add(e);
  }

  /**
   * Create an object based on the given feature and attributes.
   */
  protected void createObject(EObject peekObject, EStructuralFeature feature)
  {
    if (isNull())
    {
      setFeatureValue(peekObject, feature, null);
      objects.push(null);
      types.push(OBJECT_TYPE);
    }
    else
    {
      String xsiType = getXSIType();
      if (xsiType != null)
      {
        createObjectFromTypeName(peekObject, xsiType, feature);
      }
      else
      {
        EObject object = createObjectFromFeatureType(peekObject, feature);
        if (xmlMap != null && !((EReference)feature).isContainment())
        {
          XMLResource.XMLInfo info = xmlMap.getInfo(feature);
          if (info != null && info.getXMLRepresentation() == XMLResource.XMLInfo.ELEMENT)
          {
            text = new StringBuffer();
          }
        }
      }
    }
  }

  /**
   * Create an object from the given qualified type name.
   */
  protected EObject createObjectFromTypeName(EObject peekObject, String typeQName, EStructuralFeature feature)
  {
    String typeName = null;
    String prefix = "";
    int index = typeQName.indexOf(":");
    if (index > 0)
    {
      prefix = typeQName.substring(0, index);
      typeName = typeQName.substring(index + 1);
    }
    else
    {
      typeName = typeQName;
    }

    EFactory eFactory = getFactoryForPrefix(prefix);

    if (eFactory == null && prefix.equals("") && helper.getURI(prefix) == null)
    {
      error
        (new PackageNotFoundException
           (null,
            getLocation(),
            getLineNumber(),
            getColumnNumber()));
    }

    EObject obj = createObjectFromFactory(eFactory, typeName);

    if (obj != null)
    {
      setFeatureValue(peekObject, feature, obj);
    }

    processObject(obj);

    return obj;
  }

  /**
   * Create an object based on the type of the given feature.
   */
  protected EObject createObjectFromFeatureType(EObject peekObject, EStructuralFeature feature)
  {
    String typeName = null;
    EFactory factory = null;
    EClassifier eType = null;

    if (feature != null && (eType = feature.getEType()) != null)
    {
      EClass eClass = (EClass) eType;
      typeName = eClass.getName();
      factory = eClass.getEPackage().getEFactoryInstance();
    }

    EObject obj = createObjectFromFactory(factory, typeName);
    if (obj != null)
    {
      setFeatureValue(peekObject, feature, obj);
    }

    processObject(obj);
    return obj;
  }

  /**
   * Create an object given a content helper, a factory, and a type name,
   * and process the XML attributes.
   */
  protected EObject createObjectFromFactory(EFactory factory, String typeName)
  {
    EObject newObject = null;

    if (factory != null)
    {
      newObject = helper.createObject(factory, typeName);

      if (newObject != null)
      {
        if (disableNotify)
          newObject.eSetDeliver(false);

        handleObjectAttribs(newObject);
      }
      else
      {
        error
          (new ClassNotFoundException
            (typeName,
             factory,
             getLocation(),
             getLineNumber(),
             getColumnNumber()));
      }
    }

    return newObject;
  }

  /**
   * Add object to appropriate stacks.
   */
  protected void processObject(EObject object)
  {
    if (object != null)
    {
      objects.push(object);
      types.push(OBJECT_TYPE);
    }
    else
    {
      types.push(ERROR_TYPE);
    }
  }

  protected EFactory getFactoryForPrefix(String prefix)
  {
    EFactory factory = (EFactory) prefixesToFactories.get(prefix);

    if (factory == null)
    {
      String uriString = helper.getURI(prefix);
      EPackage ePackage = getPackageForURI(uriString);

      if (ePackage == null && uriString == null && prefix.equals(""))
      {
        ePackage = helper.getNoNamespacePackage();
      }


      if (ePackage != null)
      {
        factory = ePackage.getEFactoryInstance();
        prefixesToFactories.put(prefix, factory);
      }
    }

    return factory;
  }

  /**
   * Attempt to get the namespace for the given prefix, then return
   * ERegister.getPackage() or null.
   */
  protected EPackage getPackageForURI(String uriString)
  {
    if (uriString == null)
    {
      return null;
    }

    EPackage ePackage = EPackage.Registry.INSTANCE.getEPackage(uriString);

    if (ePackage == null)
    {
      URI uri = URI.createURI(uriString);
      if (uri.scheme() == null)
      {
        for (Iterator entries = EPackage.Registry.INSTANCE.entrySet().iterator(); entries.hasNext(); )
        {
          Map.Entry entry = (Map.Entry)entries.next();
          String nsURI = (String)entry.getKey();
          if (nsURI.endsWith(uriString) &&
                nsURI.charAt(nsURI.length() - uriString.length() - 1) == '/')
          {
            oldStyleProxyURIs = true;
            return (EPackage)entry.getValue();
          }
        }
      }

      if (urisToLocations != null)
      {
        URI locationURI = (URI)urisToLocations.get(uriString);
        if (locationURI != null)
        {
          uri = locationURI;
        }
      }

      String fragment = uri.fragment();
      Resource resource = null;

      if (resourceSet != null)
      {
        URI trimmedURI = uri.trimFragment();
        resource = resourceSet.getResource(trimmedURI, false);
        if (resource == null)
        {
          try
          {
            InputStream inputStream = resourceSet.getURIConverter().createInputStream(trimmedURI);
            resource = resourceSet.createResource(trimmedURI);
            if (resource == null)
            {
              inputStream.close();
            }
            else
            {
              resource.load(inputStream, resourceSet.getLoadOptions());
            }
          }
          catch (IOException exception)
          {
          }
        }
      }

      if (resource != null)
      {
        Object content = null;
        if (fragment != null)
        {
          content = resource.getEObject(fragment);
        }
        else
        {
          List contents = resource.getContents();
          if (!contents.isEmpty())
          {
            content = contents.get(0);
          }
        }

        if (content instanceof EPackage)
        {
          ePackage = (EPackage)content;
        }
      }
    }

    if (ePackage == null)
    {
      error
        (new PackageNotFoundException
           (uriString,
            getLocation(),
            getLineNumber(),
            getColumnNumber()));
    }

    return ePackage;
  }

  protected void setFeatureValue(EObject object, EStructuralFeature feature, Object value)
  {
    setFeatureValue(object, feature, value, -1);
  }

  /**
   * Set the given feature of the given object to the given value.
   */
  protected void setFeatureValue(EObject object, EStructuralFeature feature, Object value, int position)
  {
    try
    {
      helper.setValue(object, feature, value, position);
    }
    catch (RuntimeException e)
    {
      error
        (new IllegalValueException
           (object,
            feature,
            value,
            e,
            getLocation(),
            getLineNumber(),
            getColumnNumber()));
    }
  }

  /**
   * Set the values for the given multi-valued forward reference.
   */
  protected void setFeatureValues(ManyReference reference)
  {
    List xmiExceptions = helper.setManyReference(reference, getLocation());

    if (xmiExceptions != null)
      for (Iterator i = xmiExceptions.iterator(); i.hasNext(); )
      {
        XMIException exception = (XMIException) i.next();
        error(exception);
      }
  }

  /**
   * Create a feature with the given name for the given object with the
   * given values.
   */
  protected void setAttribValue(EObject object, String name, String value)
  {
    int index = name.indexOf(":");

    // We use null here instead of "" because an attribute without a prefix is considered to have the null target namespace...
    String prefix = null;
    String localName = name;
    if (index != -1)
    {
      prefix    = name.substring(0, index);
      localName = name.substring(index + 1);
    }

    EStructuralFeature feature = getFeature(object, prefix, localName);

    if (feature == null)
    {
      error
        (new FeatureNotFoundException
           (name,
            object,
            getLocation(),
            getLineNumber(),
            getColumnNumber()));
    }
    else
    {
      int kind = helper.getFeatureKind(feature);

      if (kind == XMLHelper.DATATYPE_SINGLE || kind == XMLHelper.DATATYPE_IS_MANY)
      {
        setFeatureValue(object, feature, value, -2);
      }
      else
      {
        setValueFromId(object, (EReference)feature, value);
      }
    }
  }

  /**
   * Create a ValueLine object and put it in the list
   * of references to resolve at the end of the document.
   */
  protected void setValueFromId(EObject object, EReference eReference, String ids)
  {
    StringTokenizer st = new StringTokenizer(ids);

    boolean isFirstID = true;
    boolean mustAdd = false;
    boolean mustAddOrNotOppositeIsMany = false;

    int size = 0;
    String qName = null;
    int position = 0;
    while (st.hasMoreTokens())
    {
      String id = st.nextToken();
      int index = id.indexOf("#");
      if (index != -1)
      {
        if (index == 0)
        {
          id = id.substring(1);
        }
        else
        {
          Object oldAttributes = setAttributes(null);
          // Create a proxy in the correct way and pop it.
          //
          InternalEObject proxy =
            (InternalEObject)
              (qName == null ?
                 createObjectFromFeatureType(object, eReference) :
                 createObjectFromTypeName(object, qName, eReference));
          setAttributes(oldAttributes);
          handleProxy(proxy, id);
          objects.pop();
          qName = null;
          ++position;
          continue;
        }
      }
      else if (id.indexOf(":") != -1)
      {
        qName = id;
        continue;
      }

      if (isFirstID)
      {
        EReference eOpposite = eReference.getEOpposite();
        mustAdd = eOpposite == null || eOpposite.isTransient() || eReference.isMany();
        mustAddOrNotOppositeIsMany = mustAdd || !eOpposite.isMany();
        isFirstID = false;
      }

      if (mustAddOrNotOppositeIsMany)
      {
        EObject resolvedEObject = xmlResource.getEObject(id);
        if (resolvedEObject != null)
        {
          setFeatureValue(object, eReference, resolvedEObject);
          qName = null;
          ++position;
          continue;
        }
      }

      if (mustAdd)
      {
        if (size == capacity)
          growArrays();

        identifiers[size] = id;
        positions[size]   = position;
        ++size;
      }
      qName = null;
      ++position;
    }

    if (position == 0)
    {
      setFeatureValue(object, eReference, null, -2);
    }
    else if (size <= REFERENCE_THRESHOLD)
    {
      for (int i = 0; i < size; i++)
      {
        SingleReference ref = new SingleReference
                                   (object,
                                    eReference,
                                    identifiers[i],
                                    positions[i],
                                    getLineNumber(),
                                    getColumnNumber());
        forwardSingleReferences.add(ref);
      }
    }
    else
    {
      Object[] values = new Object[size];
      int[] currentPositions = new int[size];
      System.arraycopy(identifiers, 0, values, 0, size);
      System.arraycopy(positions, 0, currentPositions, 0, size);

      ManyReference ref = new ManyReference
                                 (object,
                                  eReference,
                                  values,
                                  currentPositions,
                                  getLineNumber(),
                                  getColumnNumber());
      forwardManyReferences.add(ref);
    }
  }

  protected void handleProxy(InternalEObject proxy, String uriLiteral)
  {
    URI proxyURI;
    if (oldStyleProxyURIs)
    {
      proxy.eSetProxyURI(proxyURI = URI.createURI(uriLiteral.startsWith("/") ? uriLiteral : "/" + uriLiteral));
    }
    else
    {
      URI uri = URI.createURI(uriLiteral);
      if (resolve && uri.isRelative() && uri.hasRelativePath() && !EPackage.Registry.INSTANCE.containsKey(uri.trimFragment().toString()))
      {
        uri = helper.resolve(uri, resourceURI);
      }
      proxy.eSetProxyURI(proxyURI = uri);
    }

    // Test for a same document reference that would usually be handled as an IDREF.
    //
    if (proxyURI.trimFragment().equals(resourceURI))
    {
      sameDocumentProxies.add(proxy);
    }
  }

  protected void growArrays() {
    int oldCapacity = capacity;
    capacity = capacity * 2;
    Object[] newIdentifiers = new Object[capacity];
    int[] newPositions = new int[capacity];
    System.arraycopy(identifiers, 0, newIdentifiers, 0, oldCapacity);
    System.arraycopy(positions, 0, newPositions, 0, oldCapacity);
    identifiers = newIdentifiers;
    positions = newPositions;
  }

  /**
   * Returns true if there was an error in the last XML element; false otherwise.
   */
  protected boolean isError()
  {
    return types.peek() == ERROR_TYPE;
  }

  static class EClassFeatureNamePair
  {
    public EClass eClass;
    public String featureName;
    public boolean equals(Object that)
    {
      EClassFeatureNamePair typedThat = (EClassFeatureNamePair)that;
      return  typedThat.eClass == eClass && typedThat.featureName == featureName;
    }
    public int hashCode()
    {
      return eClass.hashCode() ^ featureName.hashCode();
    }
  }

  Map eClassFeatureNamePairToEStructuralFeatureMap = new HashMap();
  EClassFeatureNamePair eClassFeatureNamePair = new  EClassFeatureNamePair();

  /**
   * Get the EStructuralFeature from the metaObject for the given object
   * and feature name.
   */
  protected EStructuralFeature getFeature(EObject object, String prefix, String name)
  {
    EClass eClass = object.eClass();
    eClassFeatureNamePair.eClass = eClass;
    eClassFeatureNamePair.featureName = name;
    EStructuralFeature result = (EStructuralFeature)eClassFeatureNamePairToEStructuralFeatureMap.get(eClassFeatureNamePair);
    if (result == null)
    {
      result = helper.getFeature(eClass, helper.getURI(prefix), name);
      EClassFeatureNamePair entry = new EClassFeatureNamePair();
      entry.eClass = eClass;
      entry.featureName = name;
      eClassFeatureNamePairToEStructuralFeatureMap.put(entry, result);
    }
    return result;
  }


  /**
   * Searches the array of bytes to determine the XML
   * encoding.
   */
  public static String getXMLEncoding(byte[] bytes)
  {
    String javaEncoding = null;

    if (bytes.length >= 4)
    {
      if (((bytes[0] == -2) && (bytes[1] == -1))  ||
          ((bytes[0] == 0) && (bytes[1] == 60)))
        javaEncoding = "UnicodeBig";
      else if (((bytes[0] == -1) && (bytes[1] == -2)) ||
                ((bytes[0] == 60) && (bytes[1] == 0)))
        javaEncoding = "UnicodeLittle";
      else if ((bytes[0] == -17) && (bytes[1] == -69) && (bytes[2] == -65))
        javaEncoding = "UTF8";
    }

    String header = null;

    try
    {
      if (javaEncoding != null)
        header = new String(bytes, 0, bytes.length, javaEncoding);
      else
        header = new String(bytes, 0, bytes.length);
    }
    catch (UnsupportedEncodingException e)
    {
      return null;
    }

    if (!header.startsWith("<?xml"))
      return "UTF-8";

    int endOfXMLPI = header.indexOf("?>");
    int encodingIndex = header.indexOf("encoding", 6);

    if ((encodingIndex == -1) || (encodingIndex > endOfXMLPI))
      return "UTF-8";

    int firstQuoteIndex = header.indexOf("\"", encodingIndex);
    int lastQuoteIndex;

    if ((firstQuoteIndex == -1) || (firstQuoteIndex > endOfXMLPI))
    {
      firstQuoteIndex = header.indexOf("'", encodingIndex);
      lastQuoteIndex = header.indexOf("'", firstQuoteIndex + 1);
    }
    else
      lastQuoteIndex = header.indexOf("\"", firstQuoteIndex + 1);

    return header.substring(firstQuoteIndex + 1, lastQuoteIndex);
  }
}
