/**
 * <copyright>
 *
 * Copyright (c) 2002-2003 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.xsd/src/org/eclipse/xsd/ecore/XSDEcoreBuilder.java, xsd, org.eclipse.111, 20031020_1612WL
 * @version 1.17.1.1 10/20/03
 */
package org.eclipse.xsd.ecore;


import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.BasicEMap;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.common.util.URI;

import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

import org.eclipse.xsd.XSDAttributeDeclaration;
import org.eclipse.xsd.XSDAttributeUse;
import org.eclipse.xsd.XSDComplexTypeDefinition;
import org.eclipse.xsd.XSDComponent;
import org.eclipse.xsd.XSDCompositor;
import org.eclipse.xsd.XSDContentTypeCategory;
import org.eclipse.xsd.XSDDerivationMethod;
import org.eclipse.xsd.XSDElementDeclaration;
import org.eclipse.xsd.XSDEnumerationFacet;
import org.eclipse.xsd.XSDNamedComponent;
import org.eclipse.xsd.XSDModelGroup;
import org.eclipse.xsd.XSDParticle;
import org.eclipse.xsd.XSDSchema;
import org.eclipse.xsd.XSDSimpleTypeDefinition;
import org.eclipse.xsd.XSDTerm;
import org.eclipse.xsd.XSDTypeDefinition;
import org.eclipse.xsd.XSDWildcard;

import org.eclipse.xsd.util.XSDConstants;


public class XSDEcoreBuilder extends MapBuilder
{
  public static final String XSD2ECORE_URI = "http:///org/eclipse/emf/mapping/xsd2ecore/XSD2Ecore";

  public static final int TRIVIAL = 0x0001;
  protected int style;
  protected XSDSchema xsdSchema;
  protected Set xsdSchemas = new HashSet();
  protected Map targetNamespaceToEPackageMap = new HashMap();
  protected Map eClassToOrderedFeaturesMap = new HashMap();

  public XSDEcoreBuilder()
  {
    // style = TRIVIAL;
  }

  public XSDEcoreBuilder(int style)
  {
    this.style = style;
  }

  public XSDSchema getSchema()
  {
    return xsdSchema;
  }

  public Map getTargetNamespaceToEPackageMap()
  {
     return targetNamespaceToEPackageMap;
  }

  public Map getXSDComponentToEModelElementMap()
  {
    return xsdComponentToEModelElementMap;
  }

  protected static final List DOMAINS = Arrays.asList(new String [] {"COM", "com", "ORG", "org" });

  public EPackage getEPackage(XSDNamedComponent xsdNamedComponent)
  {
    XSDSchema containingXSDSchema = xsdNamedComponent.getSchema();
    if (containingXSDSchema != null && !xsdSchemas.contains(containingXSDSchema))
    {
      xsdSchemas.add(containingXSDSchema);
      addInput(containingXSDSchema);
    }

    String targetNamespace = 
      containingXSDSchema == null ? 
        xsdNamedComponent.getTargetNamespace() : 
        containingXSDSchema.getTargetNamespace();
    EPackage ePackage = (EPackage)targetNamespaceToEPackageMap.get(targetNamespace);
    if (ePackage == null)
    {
      ePackage = EcoreFactory.eINSTANCE.createEPackage();
      addOutput(ePackage);
      if (targetNamespace == null)
      {
        if (containingXSDSchema == null)
        {
          containingXSDSchema = xsdSchema;
        }
        ePackage.setName(validName(containingXSDSchema.eResource().getURI().trimFileExtension().lastSegment(), false));
        ePackage.setNsURI(containingXSDSchema.eResource().getURI().toString());
      }
      else
      {
        URI uri = URI.createURI(targetNamespace);
        List parsedName;
        if (uri.isHierarchical())
        {
          String host = uri.host();
          if (host != null && host.startsWith("www."))
          {
            host = host.substring(4);
          }
          parsedName = parseName(host, '.');
          Collections.reverse(parsedName);
          if (!parsedName.isEmpty())
          {
            parsedName.set(0, ((String)parsedName.get(0)).toLowerCase());
          }
  
          parsedName.addAll(parseName(uri.trimFileExtension().path(), '/'));
        }
        else
        {
          String opaquePart = uri.opaquePart();
          int index = opaquePart.indexOf(":");
          if (index != -1 && "urn".equalsIgnoreCase(uri.scheme()))
          {
            parsedName = parseName(opaquePart.substring(0, index), '-');
            if (parsedName.size() > 0 && DOMAINS.contains(parsedName.get(parsedName.size() - 1))) 
            {
              Collections.reverse(parsedName);
              parsedName.set(0, ((String)parsedName.get(0)).toLowerCase());
            }
            parsedName.addAll(parseName(opaquePart.substring(index + 1), '/'));
          }
          else
          {
            parsedName = parseName(opaquePart, '/');
          }
        }

        StringBuffer qualifiedPackageName = new StringBuffer();
        for (Iterator i = parsedName.iterator(); i.hasNext(); )
        {
          String packageName = (String)i.next();
          if (packageName.length() > 0)
          {
            if (qualifiedPackageName.length() > 0)
            {
              qualifiedPackageName.append('.');
            }
            qualifiedPackageName.append(validName(packageName, false));
          }
        }
        ePackage.setName(qualifiedPackageName.toString());
        ePackage.setNsURI(targetNamespace);
      }

      ePackage.setNsPrefix(ePackage.getName());

      EAnnotation eAnnotation = createAnnotation();
      addAnnotationDetail(eAnnotation, "representation", "schema");
      addAnnotationDetail(eAnnotation, "targetNamespace", targetNamespace);
      ePackage.getEAnnotations().add(eAnnotation);

      targetNamespaceToEPackageMap.put(targetNamespace, ePackage);
    }

    return ePackage;
  }

  public EClassifier getEClassifier(XSDTypeDefinition xsdTypeDefinition)
  {
    return getEClassifier(xsdTypeDefinition, false);
  }

  public EClassifier getEClassifier(XSDTypeDefinition xsdTypeDefinition, boolean supportNull)
  {
    EClassifier eClassifier = (EClassifier)xsdComponentToEModelElementMap.get(xsdTypeDefinition);
    if (eClassifier == null)
    {
      if (xsdTypeDefinition == null)
      {
        eClassifier = EcorePackage.eINSTANCE.getEString();
      }
      else if (xsdTypeDefinition instanceof XSDSimpleTypeDefinition)
      {
        XSDSimpleTypeDefinition xsdSimpleTypeDefinition = (XSDSimpleTypeDefinition)xsdTypeDefinition;
        if (XSDConstants.isSchemaForSchemaNamespace(xsdSimpleTypeDefinition.getTargetNamespace()))
        {
          String name = xsdSimpleTypeDefinition.getName();
          if ((style & TRIVIAL) == 0 &&
                     ("anyURI".equals(name) || "QName".equals(name) || "IDREF".equals(name) || "IDREFS".equals(name)))
          {
            eClassifier = EcorePackage.eINSTANCE.getEObject();
          }
          else if ("string".equals(name) || "anySimpleType".equals(name))
          {
            eClassifier = EcorePackage.eINSTANCE.getEString();
          }
          else if ("boolean".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEBooleanObject() : EcorePackage.eINSTANCE.getEBoolean();
          }
          else if ("float".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEFloatObject() : EcorePackage.eINSTANCE.getEFloat();
          }
          else if ("double".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEDoubleObject() : EcorePackage.eINSTANCE.getEDouble();
          }
          else if ("long".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getELongObject() : EcorePackage.eINSTANCE.getELong();
          }
          else if ("int".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEIntegerObject() : EcorePackage.eINSTANCE.getEInt();
          }
          else if ("short".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEShortObject() : EcorePackage.eINSTANCE.getEShort();
          }
          else if ("byte".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEByteObject() : EcorePackage.eINSTANCE.getEByte();
          }
          else if ("unsignedByte".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEShortObject() : EcorePackage.eINSTANCE.getEShort();
          }
          else if ("unsignedShort".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getEIntegerObject() : EcorePackage.eINSTANCE.getEInt();
          }
          else if ("unsignedInt".equals(name))
          {
            eClassifier = supportNull ? EcorePackage.eINSTANCE.getELongObject() : EcorePackage.eINSTANCE.getELong();
          }
          else 
          {
            XSDSimpleTypeDefinition baseTypeDefinition = xsdSimpleTypeDefinition.getBaseTypeDefinition();
            if (baseTypeDefinition != null && !XSDConstants.isURType(baseTypeDefinition)) 
            {
              eClassifier = getEClassifier(baseTypeDefinition, supportNull);
            }
          }
        }
        else 
        {
          XSDSimpleTypeDefinition baseTypeDefinition = xsdSimpleTypeDefinition.getBaseTypeDefinition();
          if (baseTypeDefinition != null && !supportNull)
          {
            if (!xsdSimpleTypeDefinition.getEnumerationFacets().isEmpty() &&
                  XSDConstants.isSchemaForSchemaNamespace(baseTypeDefinition.getTargetNamespace()) &&
                  "NCName".equals(baseTypeDefinition.getName()))
            {
              List enumerators = new ArrayList();
              for (Iterator i = xsdSimpleTypeDefinition.getEnumerationFacets().iterator(); 
                   i.hasNext(); )
              {
                XSDEnumerationFacet xsdEnumerationFacet = (XSDEnumerationFacet)i.next();
                String enumerator = xsdEnumerationFacet.getLexicalValue();
                if (enumerator != null && enumerator.length() != 0)
                {
                  if (Character.isJavaIdentifierStart(enumerator.charAt(0)))
                  {
                    for (int j = enumerator.length() - 1; j > 0; --j)
                    {
                      if (!Character.isJavaIdentifierPart(enumerator.charAt(j)))
                      {
                        enumerators.clear();
                        break;
                      }
                    }
                  }
                  enumerators.add(xsdEnumerationFacet);
                  continue;
                }

                enumerators.clear();
                break;
              }

              if (!enumerators.isEmpty())
              {
                EEnum eEnum = EcoreFactory.eINSTANCE.createEEnum();
                for (ListIterator i = enumerators.listIterator(); i.hasNext(); )
                {
                  XSDEnumerationFacet xsdEnumerationFacet = (XSDEnumerationFacet)i.next();
                  String enumerator = xsdEnumerationFacet.getLexicalValue();
                  EEnumLiteral eEnumLiteral = EcoreFactory.eINSTANCE.createEEnumLiteral();
                  eEnumLiteral.setName(enumerator);
                  eEnumLiteral.setValue(i.previousIndex());
                  eEnum.getELiterals().add(eEnumLiteral);
                  map(xsdEnumerationFacet, eEnumLiteral);
                }
                eClassifier = eEnum;
              }
            }

            if (eClassifier == null)
            {
              eClassifier = getEClassifier(baseTypeDefinition, supportNull);
            }
          }
        }

        if (eClassifier == null)
        {
          eClassifier = EcorePackage.eINSTANCE.getEString();
        }
      }
      else if (XSDConstants.isAnyType(xsdTypeDefinition))
      {
        eClassifier = EcorePackage.eINSTANCE.getEObject();
      }
      else
      {
        XSDComplexTypeDefinition xsdComplexTypeDefinition = (XSDComplexTypeDefinition)xsdTypeDefinition;
        EClass eClass = EcoreFactory.eINSTANCE.createEClass();

        if (xsdComplexTypeDefinition.isAbstract())
        {
          eClass.setAbstract(true);
        }

        // Do this early to prevent recursive loop.
        xsdComponentToEModelElementMap.put(xsdComplexTypeDefinition, eClass);

        eClassifier = eClass;

        XSDTypeDefinition baseTypeDefinition = xsdComplexTypeDefinition.getBaseTypeDefinition();
        if (!XSDConstants.isURType(baseTypeDefinition))
        {
          EClassifier baseType = getEClassifier(baseTypeDefinition, supportNull);
          if (baseType instanceof EClass && baseType != EcorePackage.eINSTANCE.getEObject())
          {
            eClass.getESuperTypes().add(baseType);
          }
        }

        if (eClass.getESuperTypes().isEmpty() ||
              xsdComplexTypeDefinition.getDerivationMethod() == XSDDerivationMethod.EXTENSION_LITERAL)
        {
          if (xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.SIMPLE_LITERAL)
          {
            if (eClass.getEAllStructuralFeatures().isEmpty())
            {
              EStructuralFeature eStructuralFeature =
                createFeature
                  (eClass,
                   "value",
                   getEClassifier(xsdComplexTypeDefinition.getSimpleType(), false),
                   null,
                   0,
                   1);
            }
          }
          else if ((style & TRIVIAL) != 0 ||
                     xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.MIXED_LITERAL)
          {
            if (eClass.getEAllReferences().isEmpty())
            {
              EStructuralFeature eStructuralFeature =
                createFeature
                  (eClass,
                   "contents",
                   EcorePackage.eINSTANCE.getEObject(),
                   xsdComplexTypeDefinition.getComplexType(),
                   0, 
                   -1);

              bindElements(xsdComplexTypeDefinition.getComplexType().getTerm());
            }
          }
          else if (xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.ELEMENT_ONLY_LITERAL &&
                     xsdComplexTypeDefinition.getContent() != null)
          {
            List orderedFeatures = new ArrayList();
            eClassToOrderedFeaturesMap.put(eClass, orderedFeatures);

            EMap xsdParticles = collectParticles((XSDParticle)xsdComplexTypeDefinition.getContent());
            for (Iterator i = xsdParticles.iterator(); i.hasNext(); )
            {
              Map.Entry entry = (Map.Entry)i.next();
              XSDParticle xsdParticle = (XSDParticle)entry.getKey();
              EffectiveOccurrence effectiveOccurrence = (EffectiveOccurrence)entry.getValue();
              XSDTerm xsdTerm = xsdParticle.getTerm();
              EStructuralFeature eStructuralFeature;
              if (xsdTerm instanceof XSDModelGroup)
              {
                eStructuralFeature =
                  createFeature
                    (eClass,
                     "contents",
                     EcorePackage.eINSTANCE.getEObject(),
                     xsdParticle,
                     0,
                     -1);

                bindElements(xsdTerm);
              }
              else if (xsdTerm instanceof XSDWildcard)
              {
                eStructuralFeature =
                  createFeature
                    (eClass,
                     "any",
                     EcorePackage.eINSTANCE.getEObject(),
                     xsdParticle,
                     effectiveOccurrence.minOccurs,
                     effectiveOccurrence.maxOccurs);
              }
              else
              {
                XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)xsdTerm;
                EClassifier type = 
                  getEClassifier(xsdElementDeclaration.getTypeDefinition(), xsdElementDeclaration.isNillable());

                if (type instanceof EDataType &&
                      xsdElementDeclaration.isGlobal() &&
                      xsdElementDeclaration.getDisallowedSubstitutions().size() != 3)
                {
                  type = getEClassifier(xsdElementDeclaration);
                }

                eStructuralFeature =
                  createFeature
                    (eClass,
                     validName(xsdElementDeclaration.getName(), true),
                     type,
                     xsdParticle,
                     effectiveOccurrence.minOccurs,
                     effectiveOccurrence.maxOccurs);
              }

              orderedFeatures.add(eStructuralFeature);
            }
          }

          Collection baseAttributeUses = new HashSet();
          if (baseTypeDefinition instanceof XSDComplexTypeDefinition)
          {
            for (Iterator i = ((XSDComplexTypeDefinition)baseTypeDefinition).getAttributeUses().iterator(); i.hasNext(); )
            {
              baseAttributeUses.add(((XSDAttributeUse)i.next()).getAttributeDeclaration().getURI());
            }
          }

          for (Iterator i = xsdComplexTypeDefinition.getAttributeUses().iterator(); i.hasNext(); )
          {
            XSDAttributeUse xsdAttributeUse = (XSDAttributeUse)i.next();
            if (!baseAttributeUses.contains(xsdAttributeUse.getAttributeDeclaration().getURI()))
            {
              EStructuralFeature eStructuralFeature =
                createFeature
                  (eClass,
                   validName(xsdAttributeUse.getAttributeDeclaration().getName(), true),
                   getEClassifier(xsdAttributeUse.getAttributeDeclaration().getTypeDefinition(), false),
                   xsdAttributeUse,
                   xsdAttributeUse.isRequired() ? 1 : 0,
                   1);
            }
          }
        }
      }

      if (eClassifier.eContainer() == null)
      {
        eClassifier.setName(validName(xsdTypeDefinition.getAliasName(), true));

        EAnnotation eAnnotation = createAnnotation();
        addAnnotationDetail(eAnnotation, "representation", "type");
        addAnnotationDetail(eAnnotation, "name", xsdTypeDefinition.getAliasName());
        addAnnotationDetail(eAnnotation, "targetNamespace", xsdTypeDefinition.getTargetNamespace());

        eClassifier.getEAnnotations().add(eAnnotation);

        EPackage ePackage = getEPackage(xsdTypeDefinition);
        ePackage.getEClassifiers().add(eClassifier);
       }

      map(xsdTypeDefinition, eClassifier);
    }

    return eClassifier;
  }

  protected EAnnotation createAnnotation()
  {
    EAnnotation eAnnotation = EcoreFactory.eINSTANCE.createEAnnotation();
    eAnnotation.setSource(XSD2ECORE_URI);
    return eAnnotation;
  }

  protected void addAnnotationDetail(EAnnotation eAnnotation, String key, String value)
  {
    eAnnotation.getDetails().put(key, value);
  }

  protected EStructuralFeature createFeature(EClass eClass, String name, EClassifier type, XSDComponent xsdComponent)
  {
    return createFeature(eClass, name, type, xsdComponent, 0, 1);
  }

  protected EStructuralFeature createFeature
    (EClass eClass, String name, EClassifier type, XSDComponent xsdComponent, int minOccurs, int maxOccurs)
  {
    if (xsdComponent != null)
    {
      XSDSchema containingXSDSchema = xsdComponent.getSchema();
      if (containingXSDSchema != null && !xsdSchemas.contains(containingXSDSchema))
      {
        xsdSchemas.add(containingXSDSchema);
        addInput(containingXSDSchema);
      }
    }

    if (type instanceof EClass)
    {
      EReference eReference = EcoreFactory.eINSTANCE.createEReference();
      eReference.setName(Character.toLowerCase(name.charAt(0)) + name.substring(1));
      eReference.setEType(type);
      eReference.setLowerBound(minOccurs);
      eReference.setUpperBound(maxOccurs);

      eClass.getEReferences().add(eReference);
      if (xsdComponent != null)
      {
        map(xsdComponent, eReference);
        if (xsdComponent instanceof XSDParticle)
        {
          eReference.setContainment(true);

          XSDParticle xsdParticle = (XSDParticle)xsdComponent;

          EAnnotation eAnnotation = createAnnotation();
          XSDTerm xsdTerm = ((XSDParticle)xsdComponent).getTerm();
          if (xsdTerm instanceof XSDElementDeclaration)
          {
            XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)xsdTerm;
            if (xsdElementDeclaration.getScope() instanceof XSDComplexTypeDefinition &&
                  xsdElementDeclaration.getTypeDefinition() instanceof XSDSimpleTypeDefinition)
            {
              eReference.setContainment(false);
            }

            addAnnotationDetail(eAnnotation, "representation", "element");
            addAnnotationDetail(eAnnotation, "name", ((XSDElementDeclaration)xsdTerm).getName());
            addAnnotationDetail(eAnnotation, "targetNamespace", ((XSDElementDeclaration)xsdTerm).getTargetNamespace());
          }
          else if (xsdTerm instanceof XSDWildcard)
          {
            addAnnotationDetail(eAnnotation, "representation", "element-wildcard");
          }
          else
          {
            addAnnotationDetail(eAnnotation, "representation", "general-content");
          }

          eReference.getEAnnotations().add(eAnnotation);
        }
      }

      return eReference;
    }
    else
    {
      EAttribute eAttribute = EcoreFactory.eINSTANCE.createEAttribute();
      eAttribute.setName(Character.toLowerCase(name.charAt(0)) + name.substring(1));
      eAttribute.setEType(type);
      eAttribute.setLowerBound(minOccurs);
      eAttribute.setUpperBound(maxOccurs);
      eClass.getEAttributes().add(eAttribute);

      EAnnotation eAnnotation = createAnnotation();
      if (xsdComponent == null)
      {
        addAnnotationDetail(eAnnotation, "representation", "simple-content");
      }
      else
      {
        map(xsdComponent, eAttribute);
        if (xsdComponent instanceof XSDAttributeUse)
        {
          XSDAttributeUse xsdAttributeUse = (XSDAttributeUse)xsdComponent;
          XSDAttributeDeclaration xsdAttributeDeclaration = xsdAttributeUse.getAttributeDeclaration();
          addAnnotationDetail(eAnnotation, "representation", "attribute");
          addAnnotationDetail(eAnnotation, "name", xsdAttributeDeclaration.getName());
          addAnnotationDetail(eAnnotation, "targetNamespace", xsdAttributeDeclaration.getTargetNamespace());
          eAttribute.setDefaultValueLiteral(xsdAttributeUse.getLexicalValue());
          if (XSDConstants.isOrIsDerivedFromID(xsdAttributeDeclaration.getTypeDefinition()))
          {
            eAttribute.setID(true);
          }
        }
        else if (xsdComponent instanceof XSDParticle)
        {
          XSDParticle xsdParticle = (XSDParticle)xsdComponent;


          XSDTerm xsdTerm = ((XSDParticle)xsdComponent).getTerm();
          if (xsdTerm instanceof XSDElementDeclaration)
          {
            XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)xsdTerm;
            addAnnotationDetail(eAnnotation, "representation", "element");
            addAnnotationDetail(eAnnotation, "name", xsdElementDeclaration.getName());
            addAnnotationDetail(eAnnotation, "targetNamespace", xsdElementDeclaration.getTargetNamespace());

            eAttribute.setDefaultValueLiteral(xsdElementDeclaration.getLexicalValue());
          }
          else if (xsdTerm instanceof XSDWildcard)
          {
            addAnnotationDetail(eAnnotation, "representation", "element-wildcard");
          }
          else
          {
            addAnnotationDetail(eAnnotation, "representation", "general-content");
          }
        }
        else if (xsdComponent instanceof XSDWildcard)
        {
          addAnnotationDetail(eAnnotation, "representation", "attribute-wildcard");
        }
      }
      eAttribute.getEAnnotations().add(eAnnotation);

      if (type.getDefaultValue() != null || eAttribute.getDefaultValueLiteral() != null)
      {
        eAttribute.setUnsettable(true);
      }

      return eAttribute;
    }
  }

  public static class EffectiveOccurrence
  {
    public EffectiveOccurrence(int minOccurs, int maxOccurs)
    {
      this.minOccurs = minOccurs;
      this.maxOccurs = maxOccurs;
    }

    public int minOccurs;
    public int maxOccurs;
  }

  public EMap collectParticles(XSDParticle xsdParticle)
  {
    EMap result = new BasicEMap();
    collectParticlesHelper(result, xsdParticle, 1, 1);
    return result;
  }

  public void collectParticlesHelper(EMap result, XSDParticle xsdParticle, int minOccurs, int maxOccurs)
  {
    int particleMaxOccurs = xsdParticle.getMaxOccurs();
    int effectiveMinOccurs = minOccurs * xsdParticle.getMinOccurs();
    int effectiveMaxOccurs =  maxOccurs == -1 || particleMaxOccurs == -1 ? -1 : maxOccurs * particleMaxOccurs;

    XSDTerm xsdTerm = xsdParticle.getTerm();
    if (xsdTerm instanceof XSDModelGroup)
    {
      XSDModelGroup xsdModelGroup = (XSDModelGroup)xsdTerm;
      List particles = xsdModelGroup.getParticles();
      if (particles.size() == 0)
      {
        return;
      }
      else
      {
        if ((style & TRIVIAL) != 0 || particleMaxOccurs == 1 || particles.size() == 1)
        {
          if (xsdModelGroup.getCompositor() == XSDCompositor.CHOICE_LITERAL)
          {
            effectiveMinOccurs = 0;
          }
          for (Iterator i = ((XSDModelGroup)xsdTerm).getParticles().iterator(); i.hasNext(); )
          {
            XSDParticle childXSDParticle = (XSDParticle)i.next();
            collectParticlesHelper(result, childXSDParticle, effectiveMinOccurs, effectiveMaxOccurs);
          }

          return;
        }
      }
    }

    EffectiveOccurrence effectiveOccurrence = (EffectiveOccurrence)result.get(xsdParticle);
    if (effectiveOccurrence == null)
    {
      effectiveOccurrence = new EffectiveOccurrence(effectiveMinOccurs, effectiveMaxOccurs);
      result.put(xsdParticle, effectiveOccurrence);
    }
    else
    {
      effectiveOccurrence.minOccurs += effectiveMinOccurs;
      effectiveOccurrence.maxOccurs += effectiveMaxOccurs;
    }
  }

  public void bindElements(XSDTerm xsdTerm)
  {
    if (xsdTerm instanceof XSDModelGroup)
    {
      for (Iterator i = ((XSDModelGroup)xsdTerm).getParticles().iterator(); i.hasNext(); )
      {
        XSDParticle childXSDParticle = (XSDParticle)i.next();
        bindElements(childXSDParticle.getTerm());
      }
    }
    else if (xsdTerm instanceof XSDElementDeclaration)
    {
      getEClassifier((XSDElementDeclaration)xsdTerm);
    }
  }

  public EClassifier getEClassifier(XSDElementDeclaration xsdElementDeclaration)
  {
    EClassifier eClassifier = (EClassifier)xsdComponentToEModelElementMap.get(xsdElementDeclaration);
    if (eClassifier == null)
    {
      EClassifier type = getEClassifier(xsdElementDeclaration.getType(), xsdElementDeclaration.isNillable());

      EClass elementEClass;
      if (xsdElementDeclaration.getAnonymousTypeDefinition() != null && type instanceof EClass)
      {
        // Fold in the interface for the anonymous type
        //
        elementEClass = (EClass)type;
        elementEClass.getEAnnotations().clear();
      }
      else
      {
        elementEClass = EcoreFactory.eINSTANCE.createEClass();
        EPackage ePackage = getEPackage(xsdElementDeclaration);
        ePackage.getEClassifiers().add(elementEClass);
        if (xsdElementDeclaration.isAbstract())
        {
          elementEClass.setAbstract(true);
        }
      }

      elementEClass.setName(validName(xsdElementDeclaration.getName(), true));

      EAnnotation eAnnotation = createAnnotation();
      addAnnotationDetail(eAnnotation, "representation", "element");
      addAnnotationDetail(eAnnotation, "name", xsdElementDeclaration.getAliasName());
      addAnnotationDetail(eAnnotation, "targetNamespace", xsdElementDeclaration.getTargetNamespace());
      elementEClass.getEAnnotations().add(eAnnotation);

      XSDElementDeclaration substitutionGroupAffiliation = xsdElementDeclaration.getSubstitutionGroupAffiliation();
      if (substitutionGroupAffiliation != null)
      {
        EClassifier substitutionGroupClassifier = getEClassifier(substitutionGroupAffiliation);
        if (!elementEClass.getESuperTypes().contains(substitutionGroupClassifier))
        {
          elementEClass.getESuperTypes().add(0, substitutionGroupClassifier);
        }
      }

      if (type instanceof EClass)
      {
        if (type != elementEClass && 
              type != EcorePackage.eINSTANCE.getEObject() &&
              !elementEClass.getEAllSuperTypes().contains(type))
        {
          elementEClass.getESuperTypes().add(type);
        }
      }
      else
      {
        if (elementEClass.getEAllStructuralFeatures().isEmpty())
        {
          EStructuralFeature eStructuralFeature =
            createFeature
              (elementEClass,
               "value",
               type,
               null);
        }
      }

      map(xsdElementDeclaration, elementEClass);

      eClassifier = elementEClass;
    }

    return eClassifier;
  }

  public Collection generate(URI uri)
  {
    ResourceSet resourceSet = new ResourceSetImpl();
    Resource resource = resourceSet.getResource(uri, true);
    if (!resource.getContents().isEmpty() && resource.getContents().get(0) instanceof XSDSchema)
    {
      generate((XSDSchema)resource.getContents().get(0));
    }

    List result = new ArrayList(targetNamespaceToEPackageMap.values());
    if (mapper != null)
    {
      result.add(mapper.getRoot());
    }
    return result;
  }

  protected void resolveNameConflicts()
  {
    for (Iterator i = targetNamespaceToEPackageMap.values().iterator(); i.hasNext(); )
    {
      EPackage ePackage = (EPackage)i.next();
      Map eClassifierMap = new HashMap();
      for (Iterator j = ePackage.getEClassifiers().iterator(); j.hasNext(); )
      {
        EClassifier eClassifier = (EClassifier)j.next();
        EClassifier otherEClassifier = (EClassifier)eClassifierMap.get(eClassifier.getName());
        if (otherEClassifier != null)
        {
          String representation = (String)eClassifier.getEAnnotation(XSD2ECORE_URI).getDetails().get("representation");
          if (eClassifier.getEAnnotation(XSD2ECORE_URI).getDetails().get("representation").equals("type"))
          {
            resolveNameConflict(eClassifierMap, eClassifier, "Type");
          }
          else if (otherEClassifier.getEAnnotation(XSD2ECORE_URI).getDetails().get("representation").equals("type"))
          {
            resolveNameConflict(eClassifierMap, otherEClassifier, "Type");
          }
          else
          {
            resolveNameConflict(eClassifierMap, eClassifier, "");
          }
        }
        eClassifierMap.put(eClassifier.getName(), eClassifier);

        if (eClassifier instanceof EClass)
        {
          Map eFeatureMap = new HashMap();
          for (Iterator k = ((EClass)eClassifier).getEAllStructuralFeatures().iterator(); k.hasNext(); )
          {
            EStructuralFeature eStructuralFeature = (EStructuralFeature)k.next();
            resolveNameConflict(eFeatureMap, eStructuralFeature, "");
            eFeatureMap.put(eStructuralFeature.getName(), eStructuralFeature);
          }
        }
      }
    }
  }

  protected void resolveNameConflict(Map map, ENamedElement eNamedElement, String suffix)
  {
    String name = eNamedElement.getName();
    if (!name.endsWith(suffix))
    {
      name += suffix;
    }
    if (map.containsKey(name))
    {
      int index = 0;
      while (map.containsKey(name + ++index))
      {
      }
      eNamedElement.setName(name + index);
    }
    else
    {
      eNamedElement.setName(name);
    }
  }

  public Collection generateResources(URI uri)
  {
    ResourceSet resourceSet = new ResourceSetImpl();
    Resource resource = resourceSet.getResource(uri, true);
    if (!resource.getContents().isEmpty() && resource.getContents().get(0) instanceof XSDSchema)
    {
      generate((XSDSchema)resource.getContents().get(0));
    }

    for (Iterator i = targetNamespaceToEPackageMap.values().iterator(); i.hasNext(); )
    {
      EPackage ePackage = (EPackage)i.next();
      Resource ecoreResource = resourceSet.createResource(URI.createURI("*.ecore"));
      ecoreResource.setURI
        (URI.createURI(ePackage.getNsURI().endsWith(".ecore") ? ePackage.getNsURI() : ePackage.getNsURI() + ".ecore"));
      ecoreResource.getContents().add(ePackage);
    }

    return resourceSet.getResources();
  }

  public void generate(XSDSchema xsdSchema)
  {
    this.xsdSchema = xsdSchema;
    xsdSchemas.add(xsdSchema);
    addInput(xsdSchema);

    for (Iterator i = xsdSchema.getElementDeclarations().iterator(); i.hasNext(); )
    {
      XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)i.next();
      EClassifier eClassifier = getEClassifier(xsdElementDeclaration);
    }

    for (Iterator i = xsdSchema.getTypeDefinitions().iterator(); i.hasNext(); )
    {
      XSDTypeDefinition xsdTypeDefinition = (XSDTypeDefinition)i.next();
      EClassifier eClassifier = getEClassifier(xsdTypeDefinition);
    }

    resolveNameConflicts();

    for (Iterator i = eClassToOrderedFeaturesMap.entrySet().iterator(); i.hasNext(); )
    {
      Map.Entry entry = (Map.Entry)i.next();
      EClass eClass = (EClass)entry.getKey();
      List orderedFeatures = (List)entry.getValue();

      boolean containsUnorderedEAttribute = false;
      boolean containsEReference = false;
      StringBuffer stringBuffer = new StringBuffer();
      for (int j = 0, size = orderedFeatures.size(); j < size; ++j)
      {
        if (j != 0)
        {
          stringBuffer.append(' ');
        }
        EStructuralFeature eStructuralFeature = (EStructuralFeature)orderedFeatures.get(j);
        if (eStructuralFeature instanceof EAttribute)
        {
          containsUnorderedEAttribute = containsEReference;
        }
        else
        {
          containsEReference = true;
        }
        stringBuffer.append(((EStructuralFeature)orderedFeatures.get(j)).getName());
      }

      if (containsUnorderedEAttribute)
      {
        EAnnotation eAnnotation = eClass.getEAnnotation(XSD2ECORE_URI);
        eAnnotation.getDetails().put("feature-order", stringBuffer.toString());
      }
    }
  }

  protected String validName(String name, boolean isUpperCase)
  {
    List parsedName = parseName(name, '_');
    StringBuffer result = new StringBuffer();
    for (Iterator i = parsedName.iterator(); i.hasNext(); )
    {
      String nameComponent = (String)i.next();
      if (nameComponent.length() > 0)
      {
        result.append(Character.toUpperCase(nameComponent.charAt(0)));
        result.append(nameComponent.substring(1));
      }
    }

    return
      result.length() == 0 ?
        "_" :
        Character.isJavaIdentifierStart(result.charAt(0)) ?
          isUpperCase ?
            result.toString() :
            uncapName(result.toString()) :
          "_" + result;
  }

  protected List parseName(String sourceName, char separator)
  {
    List result = new ArrayList();
    if (sourceName != null)
    {
      StringBuffer currentWord = new StringBuffer();
      boolean lastIsLower = false;
      for (int index = 0, length = sourceName.length(); index < length; ++index)
      {
        char curChar = sourceName.charAt(index);
        if (!Character.isJavaIdentifierPart(curChar))
        {
          curChar = separator;
        }
        if (Character.isUpperCase(curChar) || (!lastIsLower && Character.isDigit(curChar)) || curChar == separator)
        {
          if ((lastIsLower || curChar == separator) && currentWord.length() > 0)
          {
            result.add(currentWord.toString());
            currentWord = new StringBuffer();
          }
          lastIsLower = false;
        }
        else
        {
          if (!lastIsLower)
          {
            int currentWordLength = currentWord.length();
            if (currentWordLength > 1)
            {
              char lastChar = currentWord.charAt(--currentWordLength);
              currentWord.setLength(currentWordLength);
              result.add(currentWord.toString());
              currentWord = new StringBuffer();
              currentWord.append(lastChar);
            }
          }
          lastIsLower = true;
        }

        if (curChar != separator)
        {
          currentWord.append(curChar);
        }
      }

      result.add(currentWord.toString());
    }
    return result;
  }

  public String uncapName(String name)
  {
    if (name.length() == 0)
    {
      return name;
    }
    else
    {
      String lowerName = name.toLowerCase();
      int i;
      for (i = 0; i < name.length(); i++)
      {
        if (name.charAt(i) == lowerName.charAt(i))
        {
          break;
        }
      }
      if (i > 1 && i < name.length())
      {
        --i;
      }
      return name.substring(0, i).toLowerCase() + name.substring(i);
    }
  }
}
