/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.xsd.editor/src/org/eclipse/xsd/presentation/XSDEditor.java, xsd.editor, org.eclipse.102, 20030326_0335VL
 * @version 1.38 3/26/03
 */
package org.eclipse.xsd.presentation;


import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.IOException;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EventObject;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.Timer;
import java.util.TimerTask;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;

import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.command.CommandStackListener;

import org.eclipse.emf.common.notify.AdapterFactory;

import org.eclipse.emf.common.ui.ViewerPane;

import org.eclipse.emf.common.ui.celleditor.ExtendedComboBoxCellEditor;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;

import org.eclipse.emf.ecore.EAnnotation;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EEnum;
import org.eclipse.emf.ecore.EEnumLiteral;
import org.eclipse.emf.ecore.EModelElement;
import org.eclipse.emf.ecore.ENamedElement;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcoreFactory;
import org.eclipse.emf.ecore.EcorePackage;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;

import org.eclipse.emf.edit.provider.AdapterFactoryItemDelegator;
import org.eclipse.emf.edit.provider.IDisposable;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.emf.edit.provider.IItemPropertySource;
import org.eclipse.emf.edit.provider.ItemPropertyDescriptor;
import org.eclipse.emf.edit.provider.ItemProvider;

import org.eclipse.emf.edit.ui.action.CommandAction;
import org.eclipse.emf.edit.ui.action.CreateChildAction;
import org.eclipse.emf.edit.ui.action.CreateSiblingAction;
import org.eclipse.emf.edit.ui.action.DelegatingCommandAction;
import org.eclipse.emf.edit.ui.action.EditingDomainActionBarContributor;

import org.eclipse.emf.edit.ui.celleditor.AdapterFactoryTreeEditor;

import org.eclipse.emf.edit.ui.dnd.EditingDomainViewerDropAdapter;
import org.eclipse.emf.edit.ui.dnd.LocalTransfer;
import org.eclipse.emf.edit.ui.dnd.ViewerDragAdapter;

import org.eclipse.emf.edit.ui.provider.AdapterFactoryContentProvider;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.emf.edit.ui.provider.ExtendedImageRegistry;
import org.eclipse.emf.edit.ui.provider.PropertyDescriptor;
import org.eclipse.emf.edit.ui.provider.PropertySource;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IContributionManager;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.action.SubContributionItem;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;

import org.eclipse.jface.resource.ImageDescriptor;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentListener;

import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.IVerticalRuler;

import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ListViewer;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

import org.eclipse.jface.window.Window;

import org.eclipse.jface.wizard.WizardDialog;

import org.eclipse.swt.SWT;

import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.custom.ViewForm;

import org.eclipse.swt.dnd.DND;
import org.eclipse.swt.dnd.Transfer;

import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;

import org.eclipse.swt.graphics.Image;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Tree;

import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;

import org.eclipse.ui.actions.WorkspaceModifyOperation;

import org.eclipse.ui.dialogs.SaveAsDialog;

import org.eclipse.ui.editors.text.TextEditor;

import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.ui.part.MultiPageEditorPart;
import org.eclipse.ui.part.MultiPageSelectionProvider;

import org.eclipse.ui.views.contentoutline.ContentOutline;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;

import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySheetPage;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.PropertySheet;
import org.eclipse.ui.views.properties.PropertySheetPage;

import org.eclipse.xsd.XSDAttributeDeclaration;
import org.eclipse.xsd.XSDAttributeGroupDefinition;
import org.eclipse.xsd.XSDAttributeUse;
import org.eclipse.xsd.XSDAttributeUseCategory;
import org.eclipse.xsd.XSDComplexTypeDefinition;
import org.eclipse.xsd.XSDComponent;
import org.eclipse.xsd.XSDCompositor;
import org.eclipse.xsd.XSDConcreteComponent;
import org.eclipse.xsd.XSDConstraint;
import org.eclipse.xsd.XSDContentTypeCategory;
import org.eclipse.xsd.XSDDerivationMethod;
import org.eclipse.xsd.XSDDiagnostic;
import org.eclipse.xsd.XSDDiagnosticSeverity;
import org.eclipse.xsd.XSDElementDeclaration;
import org.eclipse.xsd.XSDEnumerationFacet;
import org.eclipse.xsd.XSDFactory;
import org.eclipse.xsd.XSDForm;
import org.eclipse.xsd.XSDImport;
import org.eclipse.xsd.XSDModelGroup;
import org.eclipse.xsd.XSDModelGroupDefinition;
import org.eclipse.xsd.XSDPackage;
import org.eclipse.xsd.XSDParticle;
import org.eclipse.xsd.XSDProcessContents;
import org.eclipse.xsd.XSDSchema;
import org.eclipse.xsd.XSDSimpleTypeDefinition;
import org.eclipse.xsd.XSDTerm;
import org.eclipse.xsd.XSDTypeDefinition;
import org.eclipse.xsd.XSDWildcard;

import org.eclipse.xsd.provider.XSDItemProviderAdapterFactory;
import org.eclipse.xsd.provider.XSDSemanticItemProviderAdapterFactory;

import org.eclipse.xsd.util.XSDConstants;
import org.eclipse.xsd.util.XSDParser;
import org.eclipse.xsd.util.XSDResourceFactoryImpl;
import org.eclipse.xsd.util.XSDResourceImpl;
import org.eclipse.xsd.util.XSDSwitch;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.XMIException;
import org.eclipse.emf.ecore.xmi.impl.XMLMapImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLResourceImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.XMLHelper;
import org.eclipse.emf.ecore.xmi.XMLLoad;
import org.eclipse.emf.ecore.xmi.impl.XMLHelperImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLLoadImpl;
import org.eclipse.emf.ecore.xmi.impl.SAXXMLHandler;
import org.eclipse.emf.ecore.xmi.impl.SAXWrapper;

import org.xml.sax.helpers.DefaultHandler;



/**
 * This is a an example of a xsd model editor.
 */
public class XSDEditor
  extends MultiPageEditorPart
  implements IEditingDomainProvider, ISelectionProvider, IMenuListener
{
  /**
   * This keeps track of the root object of the model.
   */
  protected XSDSchema xsdSchema;

  /**
   * This keeps track of the editing domain that is used to track all changes to the model.
   */
  protected AdapterFactoryEditingDomain editingDomain;

  /**
   * This is the adapter factory used for providing the syntactive views of the model.
   */
  protected XSDItemProviderAdapterFactory syntacticAdapterFactory;

  /**
   * This is the adapter factory used for providing the semantic views of the model.
   */
  protected XSDItemProviderAdapterFactory semanticAdapterFactory;

  /**
   * This is the content outline page.
   */
  protected IContentOutlinePage contentOutlinePage;

  /**
   * This is a kludge...
   */
  protected IStatusLineManager contentOutlineStatusLineManager;

  /**
   * This is the content outline page's viewer.
   */
  protected TreeViewer contentOutlineViewer;

  /**
   * This is the property sheet page.
   */
  protected PropertySheetPage propertySheetPage;

  /**
   * This source part of the editor.
   */
  protected TextEditor textEditor;
  protected ISourceViewer sourceViewer;

  /**
   * This is the syntactic viewer that shadows the selection in the content outline.
   * The parent relation must be correctly defined for this to work.
   */
  protected TreeViewer syntacticSelectionViewer;

  /**
   * This is the semantic viewer that shadows the selection in the content outline.
   */
  protected TreeViewer semanticSelectionViewer;

  /**
   * This keeps track of the active viewer pane, in the book.
   */
  protected ViewerPane currentViewerPane;

  /**
   * This keeps track of the active content viewer, which may be either one of the viewers in the pages or the content outline viewer.
   */
  protected Viewer currentViewer;

  /**
   * This listens to which ever viewer is active.
   */
  protected ISelectionChangedListener selectionChangedListener;

  /**
   * This keeps track of all the {@link org.eclipse.jface.viewers.ISelectionChangedListener}s that are listening to this editor.
   */
  protected Collection selectionChangedListeners = new ArrayList();

  /**
   * This keeps track of the selection of the editor as a whole.
   */
  protected ISelection editorSelection;

  /**
   * This is the outline action to select the next unresolved component.
   */
  protected SelectDiagnosticAction selectNextDiagnosticsAction;

  /**
   * This is the outline action to select the previous unresolved component.
   */
  protected SelectDiagnosticAction selectPreviousDiagnosticsAction;

  /**
   * This is the outline action to select the next use of a component.
   */
  protected SelectUseAction selectNextUseAction;

  /**
   * This is the outline action to select the previous use of a component.
   */
  protected SelectUseAction selectPreviousUseAction;

  /**
   * This listens for when things becomes active.
   */
  protected IPartListener partListener =
    new IPartListener()
    {
      public void partActivated(IWorkbenchPart p)
      {
        handlePartActivated(p);
      }
      public void partBroughtToTop(IWorkbenchPart p)
      {
      }
      public void partClosed(IWorkbenchPart p)
      {
      }
      public void partDeactivated(IWorkbenchPart p)
      {
      }
      public void partOpened(IWorkbenchPart p)
      {
      }
    };

  /**
   * This creates a model editor.
   */
  public XSDEditor()
  {
    super();

    // Create an adapter factory that yields item providers.
    //
    syntacticAdapterFactory = new XSDItemProviderAdapterFactory();
    semanticAdapterFactory = new XSDSemanticItemProviderAdapterFactory();

    // Create the command stack that will notify this editor as commands are executed.
    //
    BasicCommandStack commandStack = new BasicCommandStack();

    // Add a listener to set the most recent command's affected objects to be the selection of the viewer with focus.
    //
    commandStack.addCommandStackListener
      (new CommandStackListener()
       {
         public void commandStackChanged(final EventObject event)
         {
           getContainer().getDisplay().asyncExec
             (new Runnable()
              {
                public void run()
                {
                  firePropertyChange(IEditorPart.PROP_DIRTY);

                  // Try to select the affected objects.
                  //
                  Command mostRecentCommand = ((CommandStack)event.getSource()).getMostRecentCommand();
                  if (mostRecentCommand != null)
                  {
                    setSelectionToViewer(mostRecentCommand.getAffectedObjects());
                  }

                  handleStructuredModelChange();

                  updateActions();
                }
              });

         }
       });

    // Create the editing domain with a special command stack.
    //
    editingDomain = new AdapterFactoryEditingDomain(syntacticAdapterFactory, commandStack);

    // Register our xsd resource factory for this context.
    //
    editingDomain.getResourceSet().getResourceFactoryRegistry().getExtensionToFactoryMap().put("xsd", new XSDResourceFactoryImpl());
  }

  protected void updateActions()
  {
    if (selectNextDiagnosticsAction != null)
    {
      selectNextDiagnosticsAction.updateAction();
      selectPreviousDiagnosticsAction.updateAction();

      selectNextUseAction.updateAction();
      selectPreviousUseAction.updateAction();
    }
  }

  protected boolean handledStructuredModelChange = false;
  protected void handleStructuredModelChange()
  {
    IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
    if (xsdSchema.getElement() == null)
    {
      xsdSchema.updateElement();
    }

    ByteArrayOutputStream out = new ByteArrayOutputStream();
    XSDResourceImpl.serialize(out, xsdSchema.getElement());
    try
    {
      String encoding = null; // xsdSchema.getElement().getOwnerDocument().getEncoding();
      String newContent = encoding == null ? out.toString() : out.toString(encoding);
      String oldContent = document.get();

      int startIndex = 0;
      while (startIndex < newContent.length() &&
               startIndex < oldContent.length() &&
               newContent.charAt(startIndex) == oldContent.charAt(startIndex))
      {
        ++startIndex;
      }
      int newEndIndex = newContent.length() - 1;
      int oldEndIndex = oldContent.length() - 1;
      while (newEndIndex >= startIndex &&
               oldEndIndex >= startIndex &&
               newContent.charAt(newEndIndex) == oldContent.charAt(oldEndIndex))
      {
        --newEndIndex;
        --oldEndIndex;
      }

      String replacement = newContent.substring(startIndex, newEndIndex + 1);
      int length = oldEndIndex - startIndex + 1;
      handledStructuredModelChange = true;
      document.replace(startIndex, length, replacement);
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  /**
   * This handles part activation.
   */
  protected void handlePartActivated(IWorkbenchPart workbenchPart)
  {
    if (workbenchPart == this)
    {
      if (getActivePage() == 0)
      {
        setCurrentViewer((Viewer)sourceViewer);
      }
    }
    else if (workbenchPart instanceof ContentOutline)
    {
      if (((ContentOutline)workbenchPart).getCurrentPage() == contentOutlinePage)
      {
        getEditorSite().getActionBarContributor().setActiveEditor(XSDEditor.this);

        setCurrentViewer(contentOutlineViewer);
      }
    }
    else if (workbenchPart instanceof PropertySheet)
    {
      if (((PropertySheet)workbenchPart).getCurrentPage() == propertySheetPage)
      {
        getActionBarContributor().setActiveEditor(XSDEditor.this);
      }
    }
  }

  /**
   * This is here for the listener to be able to call it.
   */
  protected void firePropertyChange(int action)
  {
    super.firePropertyChange(action);
  }

  /**
   * This sets the selection into whichever viewer is active.
   */
  public void setSelectionToViewer(final Collection collection)
  {
    // Make sure it's okay.
    //
    if (collection != null && !collection.isEmpty())
    {
      // I don't know if this should be run this deferred
      // because we might have to give the editor a chance to process the viewer update events
      // and hence to update the views first.
      //
      //
      Runnable runnable =
       new Runnable()
        {
          public void run()
          {
            // Try to select the items in the current content viewer of the editor.
            //
            if (currentViewer != null)
            {
              currentViewer.setSelection(new StructuredSelection(collection.toArray()), true);
            }
          }
        };
      runnable.run();
    }
  }

  /**
   * This returns the editing domain as required by the {@link IEditingDomainProvider} interface.
   * This is important for implementing the static methods of {@link AdapterFactoryEditingDomain}
   * and for supporting {@link CommandAction}.
   */
  public EditingDomain getEditingDomain()
  {
    return editingDomain;
  }

  public void setCurrentViewerPane(ViewerPane viewerPane)
  {
    if (currentViewerPane != viewerPane)
    {
      if (currentViewerPane != null)
      {
        currentViewerPane.showFocus(false);
      }
      currentViewerPane = viewerPane;
    }

    if (currentViewerPane != null)
    {
      setCurrentViewer(currentViewerPane.getViewer());
    }
  }

  /**
   * This makes sure that one content viewer, either for the current page or the outline view, if it has focus,
   * is the current one.
   */
  public void setCurrentViewer(Viewer viewer)
  {
    // If it is changing...
    //
    if (currentViewer != viewer)
    {
      if (selectionChangedListener == null)
      {
        // Create the listener on demand.
        //
        selectionChangedListener =
          new ISelectionChangedListener()
          {
            // This just notifies those things that are affected by the section.
            //
            public void selectionChanged(SelectionChangedEvent selectionChangedEvent)
            {
              setSelection(selectionChangedEvent.getSelection());
            }
          };
      }

      // Stop listening to the old one.
      //
      if (currentViewer != null)
      {
        currentViewer.removeSelectionChangedListener(selectionChangedListener);
      }

      // Start listening to the new one.
      //
      if (viewer != null)
      {
        viewer.addSelectionChangedListener(selectionChangedListener);
      }

      // Remember it.
      //
      currentViewer = viewer;

      // Set the editors selection based on the current viewer's selection.
      //
      setSelection(currentViewer == null ? StructuredSelection.EMPTY : currentViewer.getSelection());
    }
  }

  /**
   * This is the contributor for the XSD model editor.
   */
  static public class ActionBarContributor
    extends EditingDomainActionBarContributor
    implements ISelectionChangedListener
  {
    protected IEditorPart activeEditorPart;
    protected ISelectionProvider selectionProvider;

    /**
     * This will contain one CreateChildAction corresponding to each
     * descriptor generated for the current selection.
     */
    protected Collection createChildActions = Collections.EMPTY_LIST;

    /**
     * This is the menu manager into which menu contribution items should be
     * added for the child creation actions.
     */
    protected IMenuManager createChildMenuManager;

    /**
     * This will contain one CreateSiblingAction corresponding to each
     * descriptor generated for the current selection's parent.
     */
    protected Collection createSiblingActions = Collections.EMPTY_LIST;

    /**
     * This is the menu manager into which menu contribution items should be
     * added for sibling creation actions.
     */
    protected IMenuManager createSiblingMenuManager;

    /**
     * This creates an instance of the contributor.
     */
    public ActionBarContributor()
    {
    }

    /**
     * This adds to the menu bar a menu for editor actions, duplicating
     * the menu contribution made in the plugin.xml, so that the new menu is
     * accessible for modification in code.  Also, sub-menus are created for
     * the addition and removal of child and sibling creation items.
     */
    public void contributeToMenu(IMenuManager menuManager)
    {
      super.contributeToMenu(menuManager);

      // duplicate the menu contribution in the plugin.xml
      IMenuManager submenuManager = new MenuManager(XSDEditorPlugin.INSTANCE.getString("_UI_XSDEditor_menu"), "org.eclipse.xsdMenuID");
      menuManager.insertAfter("additions", submenuManager);
      submenuManager.add(new Separator("settings"));
      submenuManager.add(new Separator("additions"));

      // prepare for child and sibling creation item addition/removal
      createChildMenuManager = new MenuManager(XSDEditorPlugin.INSTANCE.getString("_UI_CreateChild_menu_item"));
      createSiblingMenuManager = new MenuManager(XSDEditorPlugin.INSTANCE.getString("_UI_CreateSibling_menu_item"));
      submenuManager.insertBefore("additions", new Separator("actions"));
      submenuManager.insertBefore("additions", createChildMenuManager);
      submenuManager.insertBefore("additions", createSiblingMenuManager);
    }

    /**
     * This adds Separators to the tool bar.
     */
    public void contributeToToolBar(IToolBarManager toolBarManager)
    {
      toolBarManager.add(new Separator("xsd-settings"));
      toolBarManager.add(new Separator("xsd-additions"));
    }

    /**
     * When the active editor changes, this remembers the change, and
     * registers with it as a selection provider.
     */
    public void setActiveEditor(IEditorPart part)
    {
      super.setActiveEditor(part);
      activeEditorPart = part;

      // switch to the new selection provider
      if (selectionProvider != null)
      {
        selectionProvider.removeSelectionChangedListener(this);
      }
      selectionProvider = part.getSite().getSelectionProvider();
      selectionProvider.addSelectionChangedListener(this);

      // fake a selection changed event to update the menus
      if (selectionProvider.getSelection() != null)
        selectionChanged(new SelectionChangedEvent(selectionProvider, selectionProvider.getSelection()));
    }

    /**
     * This implements {@link ISelectionChangedListener}, handling
     * SelectionChangedEvents by querying for the children and siblings that
     * can be added to the selected object and updating the menus
     * accordingly.
     */
    public void selectionChanged(SelectionChangedEvent event)
    {
      // remove any menu items for old selection
      if (createChildMenuManager != null)
      {
        depopulateManager(createChildMenuManager, createChildActions);
      }
      if (createSiblingMenuManager != null)
      {
        depopulateManager(createSiblingMenuManager, createSiblingActions);
      }

      // query new selection for appropriate new child/sibling descriptors...
      Collection newChildDescriptors = Collections.EMPTY_LIST;
      Collection newSiblingDescriptors = Collections.EMPTY_LIST;
      ISelection sel = event.getSelection();

      if (sel instanceof IStructuredSelection
          && ((IStructuredSelection) sel).size() == 1)
      {
        Object object = ((IStructuredSelection) sel).getFirstElement();
        EditingDomain domain =
          ((IEditingDomainProvider) activeEditorPart).getEditingDomain();

        newChildDescriptors = domain.getNewChildDescriptors(object, null);
        newSiblingDescriptors = domain.getNewChildDescriptors(
          domain.getParent(object), object);
      }

      // generate actions for selection, populate and redraw menu
      createChildActions = generateCreateChildActions(
        newChildDescriptors, sel);
      createSiblingActions = generateCreateSiblingActions(
        newSiblingDescriptors, sel);

      if (createChildMenuManager != null)
      {
        populateManager(createChildMenuManager, createChildActions, null);
        createChildMenuManager.update(true);
      }
      if (createSiblingMenuManager != null)
      {
        populateManager(createSiblingMenuManager, createSiblingActions, null);
        createSiblingMenuManager.update(true);
      }
    }

    /**
     * This generates a {@link CreateChildAction} for each object in
     * <code>descriptors</code>, and returns the collection of these actions.
     */
    protected Collection generateCreateChildActions(Collection descriptors,
                                                    ISelection sel)
    {
      Collection actions = new LinkedList();
      for (Iterator i = descriptors.iterator(); i.hasNext(); )
      {
        actions.add(new CreateChildAction(activeEditorPart, sel, i.next()));
      }
      return actions;
    }

    /**
     * This generates a {@link CreateSiblingAction} for each object in
     * <code>descriptors</code>, and returns the collection of these actions.
     */
    protected Collection generateCreateSiblingActions(Collection descriptors,
                                                      ISelection sel)
    {
      Collection actions = new LinkedList();
      for (Iterator i = descriptors.iterator(); i.hasNext(); )
      {
        actions.add(new CreateSiblingAction(activeEditorPart, sel, i.next()));
      }
      return actions;
    }

    /**
     * This populates the specified IContributionManager with
     * ActionContributionItems based on the IActions contained in
     * the actions collection, by inserting them before the specified
     * contribution item ID.  If ID is null, they are simply added.
     */
    protected void populateManager(IContributionManager manager,
                                   Collection actions,  String ID)
    {
      for (Iterator i = actions.iterator(); i.hasNext(); )
      {
        IAction action = (IAction) i.next();
        if (ID != null)
        {
          manager.insertBefore(ID, action);
        }
        else
        {
          manager.add(action);
        }
      }
    }

    /**
     * This removes from the specified IContributionManager all
     * ActionContributionItems based on the IActions contained in the
     * actions collection.
     */
    protected void depopulateManager(IContributionManager manager,
                                     Collection actions)
    {
      IContributionItem[] item = manager.getItems();
      for (int i = 0; i < item.length; i++)
      {
        // look into SubContributionItems
        IContributionItem curItem = item[i];
        while (curItem instanceof SubContributionItem)
        {
          curItem = ((SubContributionItem) curItem).getInnerItem();
        }

        // delete ActionContributionItems with matching action
        if (curItem instanceof ActionContributionItem)
        {
          IAction action = ((ActionContributionItem) curItem).getAction();
          if (actions.contains(action))
          {
            manager.remove(curItem);
          }
        }
      }
    }

    /**
     * This populates the pop-up menu before it appears.
     */
    public void menuAboutToShow(IMenuManager menuManager)
    {
      super.menuAboutToShow(menuManager);

      menuManager.insertAfter("additions", new Separator());

      MenuManager submenuManager = new MenuManager(XSDEditorPlugin.INSTANCE.getString("_UI_CreateSibling_menu_item"));
      populateManager(submenuManager, createSiblingActions, null);
      menuManager.insertAfter("additions", submenuManager);

      submenuManager = new MenuManager(XSDEditorPlugin.INSTANCE.getString("_UI_CreateChild_menu_item"));
      populateManager(submenuManager, createChildActions, null);
      menuManager.insertAfter("additions", submenuManager);
    }
  }

  /**
   * This creates a context menu for the viewer and adds a listener as well registering the menu for extension.
   */
  protected void createContextMenuFor(StructuredViewer viewer)
  {
    MenuManager contextMenu = new MenuManager("#PopUp");
    contextMenu.add(new Separator("additions"));
    contextMenu.setRemoveAllWhenShown(true);
    contextMenu.addMenuListener(this);
    Menu menu= contextMenu.createContextMenu(viewer.getControl());
    viewer.getControl().setMenu(menu);
    getSite().registerContextMenu(contextMenu, viewer);

    int dndOperations = DND.DROP_COPY | DND.DROP_MOVE | DND.DROP_LINK;
    Transfer[] transfers = new Transfer[] { LocalTransfer.getInstance() };
    viewer.addDragSupport(dndOperations, transfers, new ViewerDragAdapter(viewer));
    viewer.addDropSupport(dndOperations, transfers, new EditingDomainViewerDropAdapter(editingDomain, viewer));
  }


  /**
   * This is the method used by the framework to install your own controls.
   */
  public void createPages()
  {
    createSourcePage();
    createSemanticsPage();
    createSyntaxPage();

    setActivePage(0);
    setCurrentViewer((Viewer)sourceViewer);
  }

  protected void createResource(String uri)
  {
    extendedCreateResource(uri);
  }

  protected void standardCreateResource(String uri)
  {

    // Load the resource through the editing domain.
    // This will creat a context and associate it with the resource set.
    //
    XSDResourceImpl xsdResource = (XSDResourceImpl)editingDomain.loadResource(uri);
    xsdSchema = xsdResource.getSchema();
  }

  protected void extendedCreateResource(String uri)
  {
    editingDomain.getResourceSet().getLoadOptions().put(XSDResourceImpl.XSD_TRACK_LOCATION, Boolean.TRUE);
    try
    {
      XSDResourceImpl xsdResource = (XSDResourceImpl)editingDomain.getResourceSet().getResource(URI.createDeviceURI(uri), true);
      xsdSchema = xsdResource.getSchema();
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  protected void createModel()
  {
    // Do the work within an operation because this is a long running activity that modifies the workbench.
    //
    WorkspaceModifyOperation operation =
      new WorkspaceModifyOperation()
      {
        // This is the method that gets invoked when the operation runs.
        //
        protected void execute(IProgressMonitor progressMonitor) throws CoreException
        {
          try
          {
            progressMonitor.beginTask("", 10);

            IFileEditorInput modelFile = (IFileEditorInput)getEditorInput();
            IFile file = modelFile.getFile();

            editingDomain.getResourceSet().getLoadOptions().put(XSDResourceImpl.XSD_PROGRESS_MONITOR, progressMonitor);
            createResource("platform:/resource" + file.getFullPath());
            editingDomain.getResourceSet().getLoadOptions().remove(XSDResourceImpl.XSD_PROGRESS_MONITOR);

            progressMonitor.worked(1);
            progressMonitor.subTask(XSDEditorPlugin.INSTANCE.getString("_UI_Validating_message"));
            if (xsdSchema.getDiagnostics().isEmpty())
            {
              xsdSchema.validate();
            }

            progressMonitor.worked(1);
            progressMonitor.subTask(XSDEditorPlugin.INSTANCE.getString("_UI_ReportingErrors_message"));

            handleDiagnostics(progressMonitor);
          }
          finally
          {
            progressMonitor.done();
          }
        }
      };

    try
    {
      // This runs the operation, and shows progress.
      // (It appears to be a bad thing to fork this onto another thread.)
      //
      new ProgressMonitorDialog(getSite().getShell()).run(false, false, operation);
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  protected void handleSourceCaretPosition()
  {
    int offset = sourceViewer.getTextWidget().getCaretOffset();
    Element element = xsdSchema.getElement();
    if (element != null)
    {
      IDocument document = sourceViewer.getDocument();
      int line = 0;
      int lineOffset = 0;
      try
      {
        line = document.getLineOfOffset(offset);
        lineOffset = document.getLineOffset(line);
      }
      catch (BadLocationException exception)
      {
      }
      int column = offset - lineOffset;
      // System.out.println("[" + line + "," + column + "]");

      Element bestElement = findBestElement(element, line + 1, column + 1);
      if (bestElement != null)
      {
        handleSelectedNodes(Collections.singleton(bestElement));
      }
    }
  }

  public Element findBestElement(Element element, int line, int column)
  {
    int startLine = XSDParser.getStartLine(element);
    int startColumn = XSDParser.getStartColumn(element);
    int endLine = XSDParser.getEndLine(element);
    int endColumn = XSDParser.getEndColumn(element);

    Element candidate = null;
    if ((line == startLine ? column >= startColumn : line > startLine)  &&
          (line == endLine ? column <= endColumn : line < endLine))
    {
      candidate = element;
      for (Node child = element.getFirstChild(); child != null; child = child.getNextSibling())
      {
        if (child.getNodeType() == Node.ELEMENT_NODE)
        {
          Element childElement = (Element)child;
          Element betterCandidate = findBestElement(childElement, line, column);
          if (betterCandidate != null)
          {
            candidate = betterCandidate;
            break;
          }
        }
      }
    }
    return candidate;
  }


  public void handleSelectedNodes(Collection nodes)
  {
    Collection selection = new ArrayList();
    for (Iterator i = nodes.iterator(); i.hasNext(); )
    {
      Node node = (Node)i.next();
      Collection parents = new HashSet();
      XSDConcreteComponent bestXSDConcreteComponent = ((XSDSchema)xsdSchema).getCorrespondingComponent(node);
      if (bestXSDConcreteComponent != null)
      {
        boolean add = true;
        for (XSDConcreteComponent parent = bestXSDConcreteComponent;
             parent != null;
             parent = parent.getContainer())
        {
          if (selection.contains(parent))
          {
            add = false;
            break;
          }
        }
        if (add)
        {
          XSDConcreteComponent container = bestXSDConcreteComponent.getContainer();
          if (container instanceof XSDParticle || container instanceof XSDAttributeUse)
          {
            bestXSDConcreteComponent = container;
          }
          selection.add(bestXSDConcreteComponent);
        }
      }
    }
    if (!selection.isEmpty())
    {
      ISelection newSelection = new StructuredSelection(selection.toArray());
      if (contentOutlineViewer != null)
      {
        contentOutlineViewer.setSelection(newSelection, true);
      }
      setSelection(newSelection);
      handleContentOutlineSelectionForTextEditor(newSelection, false);
    }
  }

  protected void handleDocumentChange()
  {
    try
    {
      XSDParser xsdParser = new XSDParser();
      String documentContent = sourceViewer.getDocument().get();
      Document domDocument = xsdSchema.getElement().getOwnerDocument();
      byte [] bytes =
        // domDocument.getEncoding() == null ? documentContent.getBytes() : documentContent.getBytes(domDocument.getEncoding());
        documentContent.getBytes();
      xsdParser.parse(new ByteArrayInputStream(bytes));
      xsdParser.setSchema(xsdSchema);
      xsdSchema.validate();

      handleDiagnostics(null);
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  protected void createSourcePage()
  {
    try
    {
      // Create the SED Editor.
      //
      textEditor =
        new TextEditor()
        {
          public ISourceViewer createSourceViewer(Composite parent, IVerticalRuler ruler, int styles)
          {
            final ISourceViewer result = super.createSourceViewer(parent, ruler, styles);
            result.getTextWidget().addMouseListener
              (new MouseAdapter()
               {
                 public void mouseDown(MouseEvent event)
                 {
                   handleSourceCaretPosition();
                 }
               });
            result.getTextWidget().addKeyListener
              (new KeyAdapter()
               {
                 public void keyPressed(KeyEvent event)
                 {
                   switch (event.keyCode)
                   {
                     case SWT.ARROW_UP:
                     case SWT.ARROW_DOWN:
                     case SWT.ARROW_LEFT:
                     case SWT.ARROW_RIGHT:
                     case SWT.PAGE_UP:
                     case SWT.PAGE_DOWN:
                     {
                       handleSourceCaretPosition();
                       break;
                     }
                   }
                 }
               });
            sourceViewer = result;
            return result;
          }
        };

      IFileEditorInput modelFile = (IFileEditorInput)getEditorInput();
      int pageIndex = addPage(textEditor, modelFile);
      setPageText(pageIndex, "Source");

      createModel();

      IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
      document.addDocumentListener
        (new IDocumentListener()
         {
           protected Timer timer = new Timer();
           protected TimerTask timerTask;

           public void documentAboutToBeChanged(DocumentEvent documentEvent)
           {
           }
           public void documentChanged(final DocumentEvent documentEvent)
           {
             try
             {
               // This is need for the Properties view.
               //
               // setSelection(StructuredSelection.EMPTY);

               if (timerTask != null)
               {
                 timerTask.cancel();
               }

               if (handledStructuredModelChange)
               {
                 handledStructuredModelChange = false;
                 handleDocumentChange();
               }
               else
               {
                 timerTask =
                   new TimerTask()
                   {
                     public void run()
                     {
                       getSite().getShell().getDisplay().asyncExec
                         (new Runnable()
                          {
                            public void run()
                            {
                              handleDocumentChange();
                            }
                          });
                     }
                   };

                 timer.schedule(timerTask, 1000);
               }
             }
             catch (Exception exception)
             {
               XSDEditorPlugin.INSTANCE.log(exception);
             }
           }
         });
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  protected void createSemanticsPage()
  {
    // Create a page for the selection tree view.
    //
    {
      ViewerPane viewerPane =
        new ViewerPane(getSite().getPage(), XSDEditor.this)
        {
          public Viewer createViewer(Composite composite)
          {
            Tree tree = new Tree(composite, SWT.MULTI);
            TreeViewer newTreeViewer = new TreeViewer(tree);
            return newTreeViewer;
          }
          public void requestActivation()
          {
            super.requestActivation();
            setCurrentViewerPane(this);
          }
        };
      viewerPane.createControl(getContainer());

      semanticSelectionViewer = (TreeViewer)viewerPane.getViewer();
      semanticSelectionViewer.setContentProvider(new AdapterFactoryContentProvider(semanticAdapterFactory));
      semanticSelectionViewer.setLabelProvider(new AdapterFactoryLabelProvider(semanticAdapterFactory));
      semanticSelectionViewer.setAutoExpandLevel(2);

      semanticSelectionViewer.addSelectionChangedListener
        (new ISelectionChangedListener()
         {
           // This just notifies those things that are affected by the section.
           //
           public void selectionChanged(SelectionChangedEvent selectionChangedEvent)
           {
             if (currentViewer == semanticSelectionViewer && contentOutlineViewer != null)
             {
               contentOutlineViewer.setSelection(selectionChangedEvent.getSelection(), true);
             }
           }
         });

      semanticSelectionViewer.setInput(new ItemProvider(Collections.singleton(xsdSchema)));
      viewerPane.setTitle(xsdSchema);

      new AdapterFactoryTreeEditor(semanticSelectionViewer.getTree(), semanticAdapterFactory);

      createContextMenuFor(semanticSelectionViewer);
      int pageIndex = addPage(viewerPane.getControl());
      setPageText(pageIndex, XSDEditorPlugin.INSTANCE.getString("_UI_Semantics_title"));
    }
  }

  protected void createSyntaxPage()
  {
    // Create a page for the selection tree view.
    //
    {
      ViewerPane viewerPane =
        new ViewerPane(getSite().getPage(), XSDEditor.this)
        {
          public Viewer createViewer(Composite composite)
          {
            Tree tree = new Tree(composite, SWT.MULTI);
            TreeViewer newTreeViewer = new TreeViewer(tree);
            return newTreeViewer;
          }
          public void requestActivation()
          {
            super.requestActivation();
            setCurrentViewerPane(this);
          }
        };
      viewerPane.createControl(getContainer());

      syntacticSelectionViewer = (TreeViewer)viewerPane.getViewer();
      syntacticSelectionViewer.setContentProvider(new AdapterFactoryContentProvider(syntacticAdapterFactory));
      syntacticSelectionViewer.setLabelProvider(new AdapterFactoryLabelProvider(syntacticAdapterFactory));
      syntacticSelectionViewer.setAutoExpandLevel(2);

      syntacticSelectionViewer.setInput(new ItemProvider(Collections.singleton(xsdSchema)));
      viewerPane.setTitle(xsdSchema);

      new AdapterFactoryTreeEditor(syntacticSelectionViewer.getTree(), syntacticAdapterFactory);

      createContextMenuFor(syntacticSelectionViewer);
      int pageIndex = addPage(viewerPane.getControl());
      setPageText(pageIndex, XSDEditorPlugin.INSTANCE.getString("_UI_Syntax_title"));
    }
  }

  protected void initializeMarkerPosition(IMarker marker, XSDDiagnostic xsdDiagnostic) throws CoreException
  {
    Node node = xsdDiagnostic.getNode();
    if (node != null && node.getNodeType() == Node.ATTRIBUTE_NODE)
    {
      node = ((Attr)node).getOwnerElement();
    }
    if (node != null && /* !xsdDiagnostic.isSetLine() && */ XSDParser.getUserData(node) instanceof Map)
    {
      int startLine = XSDParser.getStartLine(node) - 1;
      int startColumn = XSDParser.getStartColumn(node);
      int endLine = XSDParser.getEndLine(node) - 1;
      int endColumn = XSDParser.getEndColumn(node);

      marker.setAttribute(IMarker.LINE_NUMBER, startLine);

      try
      {
        IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
        marker.setAttribute(IMarker.CHAR_START, document.getLineOffset(startLine) + startColumn - 1);
        marker.setAttribute(IMarker.CHAR_END, document.getLineOffset(endLine) + endColumn - 1);
      }
      catch (BadLocationException exception)
      {
      }
    }
    else
    {
      marker.setAttribute(IMarker.LINE_NUMBER, xsdDiagnostic.getLine());
    }
  }

  protected void handleDiagnostics(IProgressMonitor progressMonitor)
  {
    if (progressMonitor == null)
    {
      // Do the work within an operation because this is a long running activity that modifies the workbench.
      //
      IWorkspaceRunnable operation =
        new IWorkspaceRunnable()
        {
          // This is the method that gets invoked when the operation runs.
          //
          public void run(IProgressMonitor localProgressMonitor) throws CoreException
          {
            handleDiagnostics(localProgressMonitor);
          }
        };

      try
      {
        ResourcesPlugin.getWorkspace().run(operation, new NullProgressMonitor());
        // getSite().getWorkbenchWindow().run(false, false, operation);
      }
      catch (Exception exception)
      {
        XSDEditorPlugin.INSTANCE.log(exception);
      }
    }
    else
    {
      XSDConcreteComponent newSelection = null;
      try
      {
        // I assume that the input is a file object.
        //
        IFileEditorInput modelFile = (IFileEditorInput)getEditorInput();
        IFile file = modelFile.getFile();

        IMarker[] markers = file.findMarkers(IMarker.PROBLEM, true, IResource.DEPTH_ZERO);
        Collection deletableMarkers = new ArrayList(Arrays.asList(markers));

        for (Iterator xsdDiagnostics = xsdSchema.getAllDiagnostics().iterator(); xsdDiagnostics.hasNext(); )
        {
          XSDDiagnostic xsdDiagnostic = (XSDDiagnostic)xsdDiagnostics.next();
          String uriReferencePath = xsdSchema.eResource().getURIFragment(xsdDiagnostic);

          IMarker marker = null;
          for (int i = 0; i < markers.length; ++i)
          {
            if (markers[i].getAttribute(XSDDiagnostic.URI_FRAGMENT_ATTRIBUTE, "").equals(uriReferencePath))
            {
              marker = markers[i];
              deletableMarkers.remove(marker);
              break;
            }
          }

          if (marker == null)
          {
            marker = file.createMarker(XSDDiagnostic.MARKER);
            marker.setAttribute(XSDDiagnostic.URI_FRAGMENT_ATTRIBUTE, uriReferencePath);
          }

          initializeMarkerPosition(marker, xsdDiagnostic);

          marker.setAttribute(IMarker.MESSAGE, xsdDiagnostic.getMessage());

          switch (xsdDiagnostic.getSeverity().getValue())
          {
            case XSDDiagnosticSeverity.FATAL:
            case XSDDiagnosticSeverity.ERROR:
            {
              if (newSelection == null)
              {
                newSelection = xsdDiagnostic.getPrimaryComponent();
              }
              marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_ERROR);
              break;
            }
            case XSDDiagnosticSeverity.WARNING:
            {
              marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_WARNING);
              break;
            }
            case XSDDiagnosticSeverity.INFORMATION:
            {
              marker.setAttribute(IMarker.SEVERITY, IMarker.SEVERITY_INFO);
              break;
            }
          }
        }

        for (Iterator i = deletableMarkers.iterator(); i.hasNext(); )
        {
          IMarker marker = (IMarker)i.next();
          marker.delete();
        }
      }
      catch (Exception exception)
      {
        XSDEditorPlugin.INSTANCE.log(exception);
      }

      // This will refresh the status.
      //
      if (editorSelection != null)
      {
        setSelection(editorSelection);
      }
      // This is the startup case.
      //
      else if (newSelection != null)
      {
        final IStructuredSelection errorSelection = new StructuredSelection(newSelection);
        getSite().getShell().getDisplay().asyncExec
          (new Runnable()
           {
             public void run()
             {
               if (contentOutlineViewer != null)
               {
                 contentOutlineViewer.setSelection(errorSelection, true);
               }
               if (sourceViewer != null)
               {
                 handleContentOutlineSelectionForTextEditor(errorSelection, true);
               }
               setSelection(errorSelection);
               handleSourceCaretPosition();
             }
           });
      }
    }
  }

  /**
   * This is used to track the active viewer.
   */
  protected void pageChange(int pageIndex)
  {
    super.pageChange(pageIndex);

    if (pageIndex == 0)
    {
      setCurrentViewerPane(null);
      setCurrentViewer((Viewer)sourceViewer);
    }

    // This is a temporary workaround... EATM
    //
    Control control = (Control) getControl(pageIndex);
    if (control != null)
    {
      control.setVisible(true);
      control.setFocus();
    }

    handleContentOutlineSelection(contentOutlinePage.getSelection());
  }

  /**
   * This is how the framework determines which interfaces we implement.
   */
  public Object getAdapter(Class key)
  {
    if (key.equals(IContentOutlinePage.class))
    {
      return getContentOutlinePage();
    }
    else if (key.equals(IPropertySheetPage.class))
    {
      return getPropertySheetPage();
    }
    else
    {
      return textEditor.getAdapter(key);
    }
  }

  /**
   * This is a utility function to resolve a component.
   */
  public static XSDConcreteComponent getResolvedObject(XSDConcreteComponent xsdConcreteComponent)
  {
    XSDConcreteComponent result =
      (XSDConcreteComponent)new XSDSwitch()
      {
        public Object caseXSDAttributeUse(XSDAttributeUse xsdAttributeUse)
        {
          return xsdAttributeUse.getAttributeDeclaration().getResolvedAttributeDeclaration();
        }
        public Object caseXSDAttributeDeclaration(XSDAttributeDeclaration xsdAttributeDeclaration)
        {
          return xsdAttributeDeclaration.getResolvedAttributeDeclaration();
        }
        public Object caseXSDAttributeGroupDefinition(XSDAttributeGroupDefinition xsdAttributeGroupDefinition)
        {
          return xsdAttributeGroupDefinition.getResolvedAttributeGroupDefinition();
        }
        public Object caseXSDElementDeclaration(XSDElementDeclaration xsdElementDeclaration)
        {
          return xsdElementDeclaration.getResolvedElementDeclaration();
        }
        public Object caseXSDModelGroupDefinition(XSDModelGroupDefinition xsdModelGroupDefinition)
        {
          return xsdModelGroupDefinition.getResolvedModelGroupDefinition();
        }
        public Object caseXSDParticle(XSDParticle xsdParticle)
        {
          Object resolvedObject = getResolvedObject(xsdParticle.getContent());
          if (resolvedObject instanceof XSDModelGroup)
          {
            return xsdParticle;
          }
          else
          {
            return resolvedObject;
          }
        }
      }.doSwitch(xsdConcreteComponent);

    return result == null ? xsdConcreteComponent : result;
  }

  /**
   * This accesses a cached version of the content outliner.
   */
  public IContentOutlinePage getContentOutlinePage()
  {
    if (contentOutlinePage == null)
    {
      // The content outline is just a tree.
      //
      class MyContentOutlinePage extends ContentOutlinePage
      {
        public void createControl(Composite parent)
        {
          super.createControl(parent);
          contentOutlineViewer = getTreeViewer();
          contentOutlineViewer.addSelectionChangedListener(this);
          contentOutlineViewer.setAutoExpandLevel(2);

          selectNextDiagnosticsAction = new SelectDiagnosticAction(true, contentOutlineViewer);
          selectPreviousDiagnosticsAction = new SelectDiagnosticAction(false, contentOutlineViewer);

          selectNextUseAction = new SelectUseAction(true, contentOutlineViewer);
          selectPreviousUseAction = new SelectUseAction(false, contentOutlineViewer);

          contentOutlineViewer.getTree().addMouseListener
            (new MouseAdapter()
             {
               public void mouseDoubleClick(MouseEvent event)
               {
                 // Do fancy navigation selections when double clicking.
                 //
                 IStructuredSelection selection = (IStructuredSelection)contentOutlineViewer.getSelection();
                 for (Iterator objects = selection.toList().iterator(); objects.hasNext(); )
                 {
                   XSDConcreteComponent object = (XSDConcreteComponent)objects.next();
                   Object resolvedObject = getResolvedObject(object);
                   if (object != resolvedObject)
                   {
                     contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {resolvedObject}), true);
                     break;
                   }
                   else if (object instanceof XSDAttributeDeclaration)
                   {
                     XSDAttributeDeclaration xsdAttributeDeclaration = (XSDAttributeDeclaration)object;
                     XSDSimpleTypeDefinition typeDefinition = xsdAttributeDeclaration.getTypeDefinition();
                     if (typeDefinition != null && typeDefinition.getSchema() == xsdAttributeDeclaration.getSchema())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {typeDefinition}), true);
                       break;
                     }
                   }
                   else if (object instanceof XSDElementDeclaration)
                   {
                     XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)object;
                     XSDTypeDefinition typeDefinition = xsdElementDeclaration.getTypeDefinition();
                     if (typeDefinition != null && typeDefinition.getSchema() == xsdElementDeclaration.getSchema())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {typeDefinition}), true);
                       break;
                     }
                   }
                   else if (object instanceof XSDSimpleTypeDefinition)
                   {
                     XSDSimpleTypeDefinition xsdSimpleTypeDefinition = (XSDSimpleTypeDefinition)object;
                     XSDSimpleTypeDefinition baseTypeDefinition = xsdSimpleTypeDefinition.getBaseTypeDefinition();
                     if (baseTypeDefinition != null && baseTypeDefinition.getSchema() == xsdSimpleTypeDefinition.getSchema())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {baseTypeDefinition}), true);
                       break;
                     }
                     XSDSimpleTypeDefinition itemTypeDefinition = xsdSimpleTypeDefinition.getItemTypeDefinition();
                     if (itemTypeDefinition != null && itemTypeDefinition.getSchema() == xsdSimpleTypeDefinition.getSchema())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {itemTypeDefinition}), true);
                       break;
                     }
                     List memberTypeDefinitions = xsdSimpleTypeDefinition.getMemberTypeDefinitions();
                     if (!memberTypeDefinitions.isEmpty())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(memberTypeDefinitions.toArray()), true);
                       break;
                     }
                   }
                   else if (object instanceof XSDComplexTypeDefinition)
                   {
                     XSDComplexTypeDefinition xsdComplexTypeDefinition = (XSDComplexTypeDefinition)object;
                     XSDTypeDefinition baseTypeDefinition = xsdComplexTypeDefinition.getBaseTypeDefinition();
                     if (baseTypeDefinition != null && baseTypeDefinition.getSchema() == xsdComplexTypeDefinition.getSchema())
                     {
                       contentOutlineViewer.setSelection(new StructuredSelection(new Object [] {baseTypeDefinition}), true);
                       break;
                     }
                   }
                 }
               }
             });

          // Set up the tree viewer.
          //
          contentOutlineViewer.setContentProvider(new AdapterFactoryContentProvider(syntacticAdapterFactory));
          contentOutlineViewer.setLabelProvider(new AdapterFactoryLabelProvider(syntacticAdapterFactory));
          contentOutlineViewer.setInput(new ItemProvider(Collections.singleton(xsdSchema)));

          // Make sure our popups work.
          //
          createContextMenuFor(contentOutlineViewer);

          // Select the root object in the view.
          //
          ArrayList selection = new ArrayList();
          selection.add(xsdSchema);
          contentOutlineViewer.setSelection(new StructuredSelection(selection), true);

          // Listen to selection so that we can handle it is a special way.
          //
          this.addSelectionChangedListener
            (new ISelectionChangedListener()
             {
               // This ensures that we handle selections correctly.
               //
               public void selectionChanged(SelectionChangedEvent event)
               {
                 ISelection s = event.getSelection();
                 if (contentOutlineViewer == currentViewer)
                 {
                   handleContentOutlineSelection(s);
                 }
                 selectNextDiagnosticsAction.setCurrentObjects(((IStructuredSelection)s).toList());
                 selectPreviousDiagnosticsAction.setCurrentObjects(((IStructuredSelection)s).toList());

                 selectNextUseAction.setCurrentObjects(((IStructuredSelection)s).toList());
                 selectPreviousUseAction.setCurrentObjects(((IStructuredSelection)s).toList());
               }
             });
        }

        public void setActionBars(IActionBars actionBars)
        {
          super.setActionBars(actionBars);

          contentOutlineStatusLineManager = actionBars.getStatusLineManager();

          actionBars.getToolBarManager().add(selectNextUseAction);
          actionBars.getToolBarManager().add(selectPreviousUseAction);

          actionBars.getToolBarManager().add(selectNextDiagnosticsAction);
          actionBars.getToolBarManager().add(selectPreviousDiagnosticsAction);


          actionBars.getMenuManager().add(selectNextDiagnosticsAction);
          actionBars.getMenuManager().add(selectPreviousDiagnosticsAction);

          actionBars.getMenuManager().add(selectNextUseAction);
          actionBars.getMenuManager().add(selectPreviousUseAction);

          getActionBarContributor().shareGlobalActions(this, actionBars);
        }
      }

      contentOutlinePage = new MyContentOutlinePage();

      // Listen to selection so that we can handle it is a special way.
      //
      contentOutlinePage.addSelectionChangedListener
        (new ISelectionChangedListener()
         {
           // This ensures that we handle selections correctly.
           //
           public void selectionChanged(SelectionChangedEvent event)
           {
             if (contentOutlineViewer == currentViewer)
             {
               handleContentOutlineSelection(event.getSelection());
             }
           }
         });
    }

    return contentOutlinePage;
  }

  /**
   * This accesses a cached version of the property sheet.
   */
  public IPropertySheetPage getPropertySheetPage()
  {
    if (propertySheetPage == null)
    {
      propertySheetPage =
        new PropertySheetPage()
        {
          public void makeContributions(IMenuManager menuManager, IToolBarManager toolBarManager, IStatusLineManager statusLineManager)
          {
            super.makeContributions(menuManager, toolBarManager, statusLineManager);
          }
        };
      propertySheetPage.setPropertySourceProvider
        (new AdapterFactoryContentProvider(syntacticAdapterFactory)
         {
           public void notifyChanged(Object object, int eventType, Object feature, Object oldValue, Object newValue, int index)
           {
             getContainer().getDisplay().asyncExec
               (new Runnable()
                {
                  public void run()
                  {
                    propertySheetPage.refresh();
                  }
                });
           }

           protected IPropertySource createPropertySource(Object object, IItemPropertySource itemPropertySource)
           {
             return
               new PropertySource(object, itemPropertySource)
               {
                 protected IPropertyDescriptor createPropertyDescriptor(IItemPropertyDescriptor itemPropertyDescriptor)
                 {
                   return
                     new PropertyDescriptor(this.object, itemPropertyDescriptor)
                     {
                       public CellEditor createPropertyEditor(Composite composite)
                       {
                         if (!this.itemPropertyDescriptor.canSetProperty(this.object))
                         {
                           return null;
                         }

                         CellEditor result = null;

                         Object genericFeature = this.itemPropertyDescriptor.getFeature(this.object);
                         if (genericFeature instanceof EStructuralFeature)
                         {
                           EStructuralFeature feature = (EStructuralFeature)genericFeature;
                           EObject getEType = feature.getEType();
                           if (getEType == ecorePackage.getEBoolean())
                           {
                             Collection choiceOfValues = this.itemPropertyDescriptor.getChoiceOfValues(this.object);
                             if (choiceOfValues != null)
                             {
                               result =
                                 new ExtendedComboBoxCellEditor
                                   (composite, new ArrayList(choiceOfValues), getLabelProvider(), true);
                             }
                           }
                         }
                         if (result == null)
                         {
                           result = super.createPropertyEditor(composite);
                         }
                         return result;
                       }
                     };
                 }
               };
           }
         });
    }

    return propertySheetPage;
  }

  /**
   * This deals with how we want selection in the outliner to affect the other views.
   */
  public void handleContentOutlineSelection(ISelection selection)
  {
    if ((currentViewerPane != null || getActivePage() == 0) && !selection.isEmpty() && selection instanceof IStructuredSelection)
    {
      if (getActivePage() == 0)
      {
        handleContentOutlineSelectionForTextEditor(selection, true);
      }
      else if (currentViewerPane.getViewer() == syntacticSelectionViewer)
      {
        // Set the selection to the widget.
        //
        syntacticSelectionViewer.setSelection(selection);
      }
      else if (currentViewerPane.getViewer() == semanticSelectionViewer)
      {
        ArrayList selectionList = new ArrayList();
        for (Iterator elements = ((IStructuredSelection)selection).iterator(); elements.hasNext(); )
        {
          selectionList.add(getResolvedObject((XSDConcreteComponent)elements.next()));
        }

        // Set the selection to the widget.
        //
        semanticSelectionViewer.setSelection(new StructuredSelection(selectionList));
      }
    }
  }

  /**
   * This deals with how we want selection in the outliner to affect the text editor.
   */
  public void handleContentOutlineSelectionForTextEditor(ISelection selection, boolean reveal)
  {
    XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)((IStructuredSelection)selection).iterator().next();
    if (xsdConcreteComponent instanceof XSDParticle)
    {
      XSDParticle xsdParticle = (XSDParticle)xsdConcreteComponent;
      XSDConcreteComponent content = xsdParticle.getContent();
      if (content != null)
      {
        xsdConcreteComponent = content;
      }
    }

    Element element = xsdConcreteComponent.getElement();
    if (element != null)
    {
      try
      {
        IDocument document = textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
        int startLine = XSDParser.getStartLine(element);
        int startColumn = XSDParser.getStartColumn(element);
        int endLine = XSDParser.getEndLine(element);
        int endColumn = XSDParser.getEndColumn(element);

        int startOffset = document.getLineOffset(startLine - 1);
        startOffset += startColumn - 1;
        int endOffset = document.getLineOffset(endLine - 1);
        endOffset += endColumn - 1;
        if (startLine == endLine)
        {
          textEditor.setHighlightRange(startOffset, endOffset - startOffset, false);
          if (reveal)
          {
            textEditor.selectAndReveal(startOffset, endOffset - startOffset);
          }
        }
        else
        {
          textEditor.setHighlightRange(startOffset, endOffset - startOffset, reveal);
        }
      }
      catch (Exception exception)
      {
        XSDEditorPlugin.INSTANCE.log(exception);
      }
    }
  }

  /**
   * This is for implementing {@link IEditorPart} and simply tests the command stack.
   */
  public boolean isDirty()
  {
    return ((BasicCommandStack)editingDomain.getCommandStack()).isSaveNeeded() || textEditor != null && textEditor.isDirty();
  }

  /**
   * This is for implementing {@link IEditorPart} and simply saves the model file.
   */
  public void doSave(IProgressMonitor progressMonitor)
  {
    // Do the work within an operation because this is a long running activity that modifies the workbench.
    //
    WorkspaceModifyOperation operation =
      new WorkspaceModifyOperation()
      {
        // This is the method that gets invoked when the operation runs.
        //
        protected void execute(IProgressMonitor monitor) throws CoreException
        {
          try
          {
            // Save the resource to the file system.
            //
            xsdSchema.eResource().save(Collections.EMPTY_MAP);
          }
          catch (Exception exception)
          {
            XSDEditorPlugin.INSTANCE.log(exception);
          }
        }
      };

    try
    {
      // This runs the operation, and shows progress.
      // (It appears to be a bad thing to fork this onto another thread.)
      //
      new ProgressMonitorDialog(getSite().getShell()).run(false, false, operation);

      // Refresh the necessary state.
      //
      ((BasicCommandStack)editingDomain.getCommandStack()).saveIsDone();
      firePropertyChange(IEditorPart.PROP_DIRTY);
    }
    catch (Exception exception)
    {
      // Something went wrong that shouldn't.
      //
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  /**
   * This always returns false because it is not current supported.
   */
  public boolean isSaveAsAllowed()
  {
    return true;
  }

  /**
   * This also changes the editor's input.
   */
  public void doSaveAs()
  {
    SaveAsDialog saveAsDialog= new SaveAsDialog(getSite().getShell());
    saveAsDialog.open();
    IPath path= saveAsDialog.getResult();
    if (path != null)
    {
      IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
      if (file != null)
      {
        if (!file.exists() ||
              MessageDialog.openQuestion
                (getSite().getShell(),
                 XSDEditorPlugin.INSTANCE.getString("_UI_FileExists_title"),
                 XSDEditorPlugin.INSTANCE.getString("_UI_FileExists_description", new Object [] { file.getFullPath() })))
        {
          xsdSchema.eResource().setURI(URI.createURI("platform:/resource" + file.getFullPath()));
          IFileEditorInput modelFile = new FileEditorInput(file);
          setInput(modelFile);
          setTitle(file.getName());
          doSave(getActionBars().getStatusLineManager().getProgressMonitor());
        }
      }
    }
  }

  public void gotoMarker(IMarker marker)
  {
    try
    {
/*
      if (marker.getType().equals(XSDDiagnostic.MARKER) && xsdSchema != null)
      {
        XSDDiagnostic xsdDiagnostic =
          xsdSchema.getDiagnosticForURIReferencePath(marker.getAttribute(XSDDiagnostic.URI_FRAGMENT_ATTRIBUTE, "/0/"));
      }
*/
      setActivePage(0);
      textEditor.gotoMarker(marker);
    }
    catch (Exception exception)
    {
      XSDEditorPlugin.INSTANCE.log(exception);
    }
  }

  /**
   * This is called during startup.
   */
  public void init(IEditorSite site, IEditorInput editorInput) throws PartInitException
  {
    if (editorInput instanceof IFileEditorInput)
    {
      setSite(site);
      setInput(editorInput);
      setTitle(((IFileEditorInput)editorInput).getFile().getName());
      site.setSelectionProvider(new MultiPageSelectionProvider(this)); // EATM
      site.getPage().addPartListener(partListener);
    }
    else
    {
      throw new PartInitException("Invalid Input: Must be IFileEditorInput.");
    }
  }

  public void setFocus()
  {
    getControl(getActivePage()).setFocus();
  }

  /**
   * This implements {@link org.eclipse.jface.viewers.ISelectionProvider}.
   */
  public void addSelectionChangedListener(ISelectionChangedListener listener)
  {
    selectionChangedListeners.add(listener);
  }

  /**
   * This implements {@link org.eclipse.jface.viewers.ISelectionProvider}.
   */
  public void removeSelectionChangedListener(ISelectionChangedListener listener)
  {
    selectionChangedListeners.remove(listener);
  }

  /**
   * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to return this editor's overall selection.
   */
  public ISelection getSelection()
  {
    return editorSelection;
  }

  /**
   * This implements {@link org.eclipse.jface.viewers.ISelectionProvider} to set this editor's overall selection.
   * Calling this result will notify the listeners.
   */
  public void setSelection(ISelection selection)
  {
    editorSelection = selection;

    SelectionChangedEvent selectionChangedEvent = new SelectionChangedEvent(this, selection);
    ((MultiPageSelectionProvider)getSite().getSelectionProvider()).fireSelectionChanged(selectionChangedEvent);
    for (Iterator listeners = selectionChangedListeners.iterator(); listeners.hasNext(); )
    {
      ISelectionChangedListener listener = (ISelectionChangedListener)listeners.next();
      listener.selectionChanged(selectionChangedEvent);
    }

    setStatusLineManager(selection);
  }

  /**
   *  This shows the selection on the status line.
   */
  public void setStatusLineManager(ISelection selection)
  {
    IStatusLineManager statusLineManager = getActionBars().getStatusLineManager();
    if (currentViewer == contentOutlineViewer)
    {
      statusLineManager = contentOutlineStatusLineManager;
    }

    if (statusLineManager != null)
    {
      if (selection instanceof IStructuredSelection)
      {
        Collection collection = ((IStructuredSelection)selection).toList();
        switch (collection.size())
        {
          case 0:
          {
            statusLineManager.setMessage(XSDEditorPlugin.INSTANCE.getString("_UI_NoObjectSelected"));
            break;
          }
          case 1:
          {
            Object object = collection.iterator().next();
            String text = new AdapterFactoryItemDelegator(syntacticAdapterFactory).getText(object);
            text = XSDEditorPlugin.INSTANCE.getString("_UI_SingleObjectSelected", new Object [] { text });
            if (object instanceof XSDConcreteComponent)
            {
              XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)object;
              if (!xsdConcreteComponent.getDiagnostics().isEmpty())
              {
                text = ((XSDDiagnostic)xsdConcreteComponent.getDiagnostics().get(0)).getMessage();
              }
            }

            statusLineManager.setMessage(text);
            break;
          }
          default:
          {
            statusLineManager.setMessage
              (XSDEditorPlugin.INSTANCE.getString("_UI_MultiObjectSelected", new Object [] { Integer.toString(collection.size()) }));
            break;
          }
        }
      }
      else
      {
        statusLineManager.setMessage("");
      }
    }
  }

  /**
   * This implements {@link org.eclipse.jface.action.IMenuListener} to help fill the context menus with contributions from the Edit menu.
   */
  public void menuAboutToShow(IMenuManager menuManager)
  {
    ((IMenuListener)getEditorSite().getActionBarContributor()).menuAboutToShow(menuManager);
  }

  /**
   * This convenience method provides typed access to the contributor.
   */
  public EditingDomainActionBarContributor getActionBarContributor()
  {
    return (EditingDomainActionBarContributor)getEditorSite().getActionBarContributor();
  }

  /**
   * This convenience method provides access to the actionbars.
   */
  public IActionBars getActionBars()
  {
    return getActionBarContributor().getActionBars();
  }

  /**
   * This is called when the editor is disposed.
   */
  public void dispose()
  {
    super.dispose();
    getSite().getPage().removePartListener(partListener);
    ((IDisposable)semanticAdapterFactory).dispose();
    ((IDisposable)syntacticAdapterFactory).dispose();
  }

  /**
   * This is the base action for the outline actions.
   */
  class SelectObjectAction extends Action
  {
    protected Collection objectsToSelect;
    protected StructuredViewer structuredViewer;

    public SelectObjectAction(StructuredViewer structuredViewer, String text, ImageDescriptor imageDescriptor)
    {
      super(text, imageDescriptor);
      this.structuredViewer = structuredViewer;
      setEnabled(false);
    }

    public void setObjectToSelect(Object objectToSelect)
    {
      setObjectsToSelect
        (objectToSelect != null ?
           Collections.singleton(objectToSelect) :
           (Collection)Collections.EMPTY_LIST);
    }

    public void setObjectsToSelect(Collection objectsToSelect)
    {
      this.objectsToSelect = new ArrayList(objectsToSelect);
      setEnabled(!objectsToSelect.isEmpty());
    }

    public void run()
    {
      structuredViewer.setSelection(new StructuredSelection(objectsToSelect.toArray()), true);
    }
  }

  /**
   * This is used to implement the select next/previous unresolved component action.
   */
  class SelectDiagnosticAction extends SelectObjectAction
  {
    boolean isForward;
    public SelectDiagnosticAction(boolean isForward, StructuredViewer structuredViewer)
    {
      super
        (structuredViewer,
         isForward ?  "Select &Next Diagnosed Object" : "Select &Previous Diagnosed Object",
         ExtendedImageRegistry.INSTANCE.getImageDescriptor
           (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/elcl16/SelectNextDiagnosticObject" : "full/elcl16/SelectPreviousDiagnosticObject")));
      this.isForward = isForward;

      setHoverImageDescriptor
        (ExtendedImageRegistry.INSTANCE.getImageDescriptor
          (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/clcl16/SelectNextDiagnosticObject" : "full/clcl16/SelectPreviousDiagnosticObject")));

      setDisabledImageDescriptor
        (ExtendedImageRegistry.INSTANCE.getImageDescriptor
          (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/dlcl16/SelectNextDiagnosticObject" : "full/dlcl16/SelectPreviousDiagnosticObject")));
    }

    public void updateAction()
    {
      setCurrentObjects(((IStructuredSelection)structuredViewer.getSelection()).toList());
    }

    public void setCurrentObjects(List objects)
    {
      XSDConcreteComponent result = null;

      boolean isStarted = false;
      for (TreeIterator tree = editingDomain.treeIterator(xsdSchema); tree.hasNext(); )
      {
        XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)tree.next();
        if (!isForward && objects.contains(xsdConcreteComponent))
        {
          break;
        }
        else if (isStarted || !isForward)
        {
          if (!xsdConcreteComponent.getDiagnostics().isEmpty() ||
                 xsdConcreteComponent instanceof XSDParticle &&
                   !((XSDParticle)xsdConcreteComponent).getContent().getDiagnostics().isEmpty() ||
                 xsdConcreteComponent instanceof XSDAttributeUse &&
                   !((XSDAttributeUse)xsdConcreteComponent).getContent().getDiagnostics().isEmpty())
          {
            if (isStarted)
            {
              result = xsdConcreteComponent;
              break;
            }
            else
            {
              result = xsdConcreteComponent;
            }
          }
        }
        else if (objects.contains(xsdConcreteComponent))
        {
          isStarted = true;
        }
      }

      setObjectToSelect(result);
    }
  }

  /**
   * This is used to implement the select next/previous component use action.
   */
  class SelectUseAction extends SelectObjectAction
  {
    boolean isForward;
    public SelectUseAction(boolean isForward, StructuredViewer structuredViewer)
    {
      super
        (structuredViewer,
         isForward ?  "Select &Next Use" : "Select &Previous Use",
         ExtendedImageRegistry.INSTANCE.getImageDescriptor
           (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/elcl16/SelectNextUseObject" : "full/elcl16/SelectPreviousUseObject")));
      this.isForward = isForward;

      setHoverImageDescriptor
        (ExtendedImageRegistry.INSTANCE.getImageDescriptor
          (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/clcl16/SelectNextUseObject" : "full/clcl16/SelectPreviousUseObject")));
      setDisabledImageDescriptor
        (ExtendedImageRegistry.INSTANCE.getImageDescriptor
          (XSDEditorPlugin.INSTANCE.getImage
             (isForward ? "full/dlcl16/SelectNextUseObject" : "full/dlcl16/SelectPreviousUseObject")));
    }

    public void updateAction()
    {
      setCurrentObjects(((IStructuredSelection)structuredViewer.getSelection()).toList());
    }

    public void setCurrentObjects(List objects)
    {
      XSDConcreteComponent result = null;

      final List resolvedObjects = new ArrayList();
      for (Iterator i = objects.iterator(); i.hasNext(); )
      {
        XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)i.next();
        XSDConcreteComponent resolvedObject = getResolvedObject(xsdConcreteComponent);
        if (resolvedObject != this)
        {
          resolvedObjects.add(resolvedObject);
        }
      }

      boolean isStarted = false;
      for (TreeIterator tree = editingDomain.treeIterator(xsdSchema); tree.hasNext(); )
      {
        XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)tree.next();
        if (!isForward && objects.contains(xsdConcreteComponent))
        {
          break;
        }
        else if (isStarted || !isForward)
        {
          XSDConcreteComponent resolvedObject = getResolvedObject(xsdConcreteComponent);
          if (resolvedObjects.contains(resolvedObject))
          {
            if (isStarted)
            {
              result = xsdConcreteComponent;
              break;
            }
            else
            {
              result = xsdConcreteComponent;
            }
          }
        }
        else if (objects.contains(xsdConcreteComponent))
        {
          isStarted = true;
        }
      }

      setObjectToSelect(result);
    }
  }

  public static class MapBuilder
  {
    protected EObject mappingRoot;
    protected Object mapper;
    protected Method mapMethod;
    protected Method addInputMethod;
    protected Method addOutputMethod;
    protected Map xsdComponentToEModelElementMap = new HashMap();

    public MapBuilder()
    {
      if (Platform.getPluginRegistry().getPluginDescriptor("org.eclipse.emf.mapping.xsd2ecore.editor") != null)
      {
        Plugin plugin = Platform.getPlugin("org.eclipse.emf.mapping.xsd2ecore.editor");
        try
        {
          Class theClass =
            plugin.getClass().getClassLoader().loadClass
              ("org.eclipse.emf.mapping.xsd2ecore.presentation.XSD2EcoreEditor$XSD2EcoreMapper");

          mapper = theClass.newInstance();
          mapMethod = theClass.getDeclaredMethod("map", new Class [] { Collection.class, Collection.class });
          addInputMethod = theClass.getDeclaredMethod("addInput", new Class [] { EObject.class });
          addOutputMethod = theClass.getDeclaredMethod("addOutput", new Class [] { EObject.class });
        }
        catch (IllegalAccessException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (NoSuchMethodException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (InstantiationException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (ClassNotFoundException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
      }
    }

    protected void map(XSDComponent xsdComponent, EModelElement eModelElement)
    {
      if (mapMethod != null)
      {
        try
        {
          mappingRoot =
            (EObject)mapMethod.invoke
              (mapper,
               new Object []
               {
                 Collections.singleton(xsdComponent),
                 Collections.singleton(eModelElement)
               });
        }
        catch (InvocationTargetException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (IllegalAccessException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
      }

      xsdComponentToEModelElementMap.put(xsdComponent, eModelElement);
    }

    protected void addInput(EObject eObject)
    {
      if (addInputMethod != null)
      {
        try
        {
          mappingRoot =
            (EObject)addInputMethod.invoke
              (mapper,
               new Object []
               {
                 eObject
               });
        }
        catch (InvocationTargetException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (IllegalAccessException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
      }
    }

    protected void addOutput(EObject eObject)
    {
      if (addOutputMethod != null)
      {
        try
        {
          mappingRoot =
            (EObject)addOutputMethod.invoke
              (mapper,
               new Object []
               {
                 eObject
               });
        }
        catch (InvocationTargetException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
        catch (IllegalAccessException exception)
        {
          XSDEditorPlugin.INSTANCE.log(exception);
        }
      }
    }
  }

  public static class XSDEcoreBuilder extends MapBuilder
  {
    protected static final int TRIVIAL = 0x0001;
    protected int style;
    protected XSDSchema xsdSchema;
    protected Map targetNamespaceToEPackageMap = new HashMap();

    public XSDEcoreBuilder()
    {
      // style = TRIVIAL;
    }

    public XSDEcoreBuilder(int style)
    {
      this.style = style;
    }

    public XSDSchema getSchema()
    {
      return xsdSchema;
    }

    public Map getTargetNamespaceToEPackageMap()
    {
       return targetNamespaceToEPackageMap;
    }

    public Map getXSDComponentToEModelElementMap()
    {
      return xsdComponentToEModelElementMap;
    }

    public EPackage getEPackage(String targetNamespace)
    {
      EPackage ePackage = (EPackage)targetNamespaceToEPackageMap.get(targetNamespace);
      if (ePackage == null)
      {
        ePackage = EcoreFactory.eINSTANCE.createEPackage();
        addOutput(ePackage);
        if (targetNamespace == null)
        {
          ePackage.setName(validName(xsdSchema.eResource().getURI().trimFileExtension().lastSegment(), false));
          ePackage.setNsURI(xsdSchema.eResource().getURI().toString());
        }
        else
        {
          URI uri = URI.createURI(targetNamespace);
          String host = uri.host();
          if (host != null && host.startsWith("www."))
          {
            host = host.substring(4);
          }
          List parsedName = parseName(host, '.');
          Collections.reverse(parsedName);
          if (!parsedName.isEmpty())
          {
            parsedName.set(0, ((String)parsedName.get(0)).toLowerCase());
          }

          parsedName.addAll(parseName(uri.trimFileExtension().path(), '/'));
          StringBuffer qualifiedPackageName = new StringBuffer();
          for (Iterator i = parsedName.iterator(); i.hasNext(); )
          {
            String packageName = (String)i.next();
            if (packageName.length() > 0)
            {
              if (qualifiedPackageName.length() > 0)
              {
                qualifiedPackageName.append('.');
              }
              qualifiedPackageName.append(validName(packageName, false));
            }
          }
          ePackage.setName(qualifiedPackageName.toString());
          ePackage.setNsURI(targetNamespace);
        }

        ePackage.setNsPrefix(ePackage.getName());

        EAnnotation eAnnotation = createAnnotation();
        addAnnotationDetail(eAnnotation, "representation", "schema");
        addAnnotationDetail(eAnnotation, "targetNamespace", targetNamespace);
        ePackage.getEAnnotations().add(eAnnotation);

        targetNamespaceToEPackageMap.put(targetNamespace, ePackage);
      }

      return ePackage;
    }

    public EClassifier getEClassifier(XSDTypeDefinition xsdTypeDefinition)
    {
      EClassifier eClassifier = (EClassifier)xsdComponentToEModelElementMap.get(xsdTypeDefinition);
      if (eClassifier == null)
      {
        if (xsdTypeDefinition instanceof XSDSimpleTypeDefinition)
        {
          XSDSimpleTypeDefinition xsdSimpleTypeDefinition = (XSDSimpleTypeDefinition)xsdTypeDefinition;
          if (XSDConstants.isSchemaForSchemaNamespace(xsdSimpleTypeDefinition.getTargetNamespace()))
          {
            String name = xsdSimpleTypeDefinition.getName();
            if ((style & TRIVIAL) == 0 &&
                       ("anyURI".equals(name) || "QName".equals(name) || "IDREF".equals(name) || "IDREFS".equals(name)))
            {
              eClassifier = EcorePackage.eINSTANCE.getEObject();
            }
            else if ("string".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEString();
            }
            else if ("boolean".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEBoolean();
            }
            else if ("float".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEFloat();
            }
            else if ("double".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEDouble();
            }
            else if ("long".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getELong();
            }
            else if ("int".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEInt();
            }
            else if ("short".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEShort();
            }
            else if ("byte".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEByte();
            }
            else if ("unsignedByte".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEShort();
            }
            else if ("unsignedShort".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getEInt();
            }
            else if ("unsignedShort".equals(name))
            {
              eClassifier = EcorePackage.eINSTANCE.getELong();
            }
            else 
            {
              XSDSimpleTypeDefinition baseTypeDefinition = xsdSimpleTypeDefinition.getBaseTypeDefinition();
              if (baseTypeDefinition != null && !XSDConstants.isURType(baseTypeDefinition)) 
              {
                eClassifier = getEClassifier(baseTypeDefinition);
              }
            }
          }
          else 
          {
            XSDSimpleTypeDefinition baseTypeDefinition = xsdSimpleTypeDefinition.getBaseTypeDefinition();
            if (baseTypeDefinition != null)
            {
              if (!xsdSimpleTypeDefinition.getEnumerationFacets().isEmpty() &&
                    XSDConstants.isSchemaForSchemaNamespace(baseTypeDefinition.getTargetNamespace()) &&
                    "NCName".equals(baseTypeDefinition.getName()))
              {
                List enumerators = new ArrayList();
                for (Iterator i = xsdSimpleTypeDefinition.getEnumerationFacets().iterator(); 
                     i.hasNext(); )
                {
                  XSDEnumerationFacet xsdEnumerationFacet = (XSDEnumerationFacet)i.next();
                  String enumerator = xsdEnumerationFacet.getLexicalValue();
                  if (enumerator != null && enumerator.length() != 0)
                  {
                    if (Character.isJavaIdentifierStart(enumerator.charAt(0)))
                    {
                      for (int j = enumerator.length() - 1; j > 0; --j)
                      {
                        if (!Character.isJavaIdentifierPart(enumerator.charAt(j)))
                        {
                          enumerators.clear();
                          break;
                        }
                      }
                    }
                    enumerators.add(xsdEnumerationFacet);
                    continue;
                  }

                  enumerators.clear();
                  break;
                }

                if (!enumerators.isEmpty())
                {
                  EEnum eEnum = EcoreFactory.eINSTANCE.createEEnum();
                  for (ListIterator i = enumerators.listIterator(); i.hasNext(); )
                  {
                    XSDEnumerationFacet xsdEnumerationFacet = (XSDEnumerationFacet)i.next();
                    String enumerator = xsdEnumerationFacet.getLexicalValue();
                    EEnumLiteral eEnumLiteral = EcoreFactory.eINSTANCE.createEEnumLiteral();
                    eEnumLiteral.setName(enumerator);
                    eEnumLiteral.setValue(i.previousIndex());
                    eEnum.getELiterals().add(eEnumLiteral);
                    map(xsdEnumerationFacet, eEnumLiteral);
                  }
                  eClassifier = eEnum;
                }
              }

              if (eClassifier == null)
              {
                eClassifier = getEClassifier(baseTypeDefinition);
              }
            }
          }

          if (eClassifier == null)
          {
            eClassifier = EcoreFactory.eINSTANCE.createEDataType();
            String name = "java.lang.String";
            eClassifier.setInstanceClassName("java.lang.String");
          }
        }
        else
        {
          XSDComplexTypeDefinition xsdComplexTypeDefinition = (XSDComplexTypeDefinition)xsdTypeDefinition;
          EClass eClass = EcoreFactory.eINSTANCE.createEClass();

          // Do this early to prevent recursive loop.
          xsdComponentToEModelElementMap.put(xsdComplexTypeDefinition, eClass);

          eClassifier = eClass;

          XSDTypeDefinition baseTypeDefinition = xsdComplexTypeDefinition.getBaseTypeDefinition();
          if (!XSDConstants.isURType(baseTypeDefinition))
          {
            EClassifier baseType = getEClassifier(baseTypeDefinition);
            if (baseType instanceof EClass && baseType != EcorePackage.eINSTANCE.getEObject())
            {
              eClass.getESuperTypes().add(baseType);
            }
          }

          if (eClass.getESuperTypes().isEmpty() ||
                xsdComplexTypeDefinition.getDerivationMethod() == XSDDerivationMethod.EXTENSION_LITERAL)
          {
            if (xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.SIMPLE_LITERAL)
            {
              if (eClass.getEAllReferences().isEmpty())
              {
                EStructuralFeature eStructuralFeature =
                  createFeature
                    (eClass,
                     "value",
                     getEClassifier(xsdComplexTypeDefinition.getSimpleType()),
                     null);
              }
            }
            else if ((style & TRIVIAL) != 0 ||
                       xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.MIXED_LITERAL)
            {
              if (eClass.getEAllReferences().isEmpty())
              {
                EStructuralFeature eStructuralFeature =
                  createFeature
                    (eClass,
                     "contents",
                     EcorePackage.eINSTANCE.getEObject(),
                     xsdComplexTypeDefinition.getComplexType());

                eStructuralFeature.setLowerBound(0);
                eStructuralFeature.setUpperBound(-1);

                if ((style & TRIVIAL) != 0)
                {
                  List xsdParticles = collectParticles((XSDParticle)xsdComplexTypeDefinition.getContent());
                  for (Iterator i = xsdParticles.iterator(); i.hasNext(); )
                  {
                    XSDParticle xsdParticle = (XSDParticle)i.next();
                    XSDTerm xsdTerm = xsdParticle.getTerm();
                    if (xsdTerm instanceof XSDElementDeclaration)
                    {
                      getEClassifier((XSDElementDeclaration)xsdTerm);
                    }
                  }
                }
              }
            }
            else if (xsdComplexTypeDefinition.getContentTypeCategory() == XSDContentTypeCategory.ELEMENT_ONLY_LITERAL &&
                       xsdComplexTypeDefinition.getContent() != null)
            {
              List xsdParticles = collectParticles((XSDParticle)xsdComplexTypeDefinition.getContent());
              for (Iterator i = xsdParticles.iterator(); i.hasNext(); )
              {
                XSDParticle xsdParticle = (XSDParticle)i.next();
                XSDTerm xsdTerm = xsdParticle.getTerm();
                EStructuralFeature eStructuralFeature;
                if (xsdTerm instanceof XSDModelGroup)
                {
                  eStructuralFeature =
                    createFeature
                      (eClass,
                       "contents",
                       EcorePackage.eINSTANCE.getEObject(),
                       xsdParticle);

                  eStructuralFeature.setUpperBound(-1);
                }
                else if (xsdTerm instanceof XSDWildcard)
                {
                  eStructuralFeature =
                    createFeature
                      (eClass,
                       "any",
                       EcorePackage.eINSTANCE.getEObject(),
                       xsdParticle);
                }
                else
                {
                  XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)xsdTerm;
                  EClassifier type = getEClassifier(xsdElementDeclaration.getTypeDefinition());
                  if (type instanceof EDataType &&
                        xsdElementDeclaration.isGlobal() &&
                        xsdElementDeclaration.getDisallowedSubstitutions().size() != 3)
                  {
                    type = getEClassifier(xsdElementDeclaration);
                  }

                  eStructuralFeature =
                    createFeature
                      (eClass,
                       validName(xsdElementDeclaration.getName(), true),
                       type,
                       xsdParticle);
                }
              }
            }

            Collection baseAttributeUses = new HashSet();
            if (baseTypeDefinition instanceof XSDComplexTypeDefinition)
            {
              for (Iterator i = ((XSDComplexTypeDefinition)baseTypeDefinition).getAttributeUses().iterator(); i.hasNext(); )
              {
                baseAttributeUses.add(((XSDAttributeUse)i.next()).getAttributeDeclaration().getURI());
              }
            }

            for (Iterator i = xsdComplexTypeDefinition.getAttributeUses().iterator(); i.hasNext(); )
            {
              XSDAttributeUse xsdAttributeUse = (XSDAttributeUse)i.next();
              if (!baseAttributeUses.contains(xsdAttributeUse.getAttributeDeclaration().getURI()))
              {
                EStructuralFeature eStructuralFeature =
                  createFeature
                    (eClass,
                     validName(xsdAttributeUse.getAttributeDeclaration().getName(), true),
                     getEClassifier(xsdAttributeUse.getAttributeDeclaration().getTypeDefinition()),
                     xsdAttributeUse);
              }
            }
          }
        }

        if (eClassifier.eContainer() == null)
        {
          eClassifier.setName(validName(xsdTypeDefinition.getAliasName(), true));

          EAnnotation eAnnotation = createAnnotation();
          addAnnotationDetail(eAnnotation, "representation", "type");
          addAnnotationDetail(eAnnotation, "name", xsdTypeDefinition.getAliasName());
          addAnnotationDetail(eAnnotation, "targetNamespace", xsdTypeDefinition.getTargetNamespace());
          eClassifier.getEAnnotations().add(eAnnotation);

          EPackage ePackage = getEPackage(xsdTypeDefinition.getSchema().getTargetNamespace());
          ePackage.getEClassifiers().add(eClassifier);
         }

        map(xsdTypeDefinition, eClassifier);
      }

      return eClassifier;
    }

    protected EAnnotation createAnnotation()
    {
      EAnnotation eAnnotation = EcoreFactory.eINSTANCE.createEAnnotation();
      eAnnotation.setSource("http:///org/eclipse/emf/mapping/xsd2ecore/XSD2Ecore");
      return eAnnotation;
    }

    protected void addAnnotationDetail(EAnnotation eAnnotation, String key, String value)
    {
      eAnnotation.getDetails().put(key, value);
    }

    protected EStructuralFeature createFeature(EClass eClass, String name, EClassifier type, XSDComponent xsdComponent)
    {
      if (type instanceof EClass)
      {
        EReference eReference = EcoreFactory.eINSTANCE.createEReference();
        eReference.setName(Character.toLowerCase(name.charAt(0)) + name.substring(1));
        eReference.setEType(type);

        eClass.getEReferences().add(eReference);
        if (xsdComponent != null)
        {
          map(xsdComponent, eReference);
          if (xsdComponent instanceof XSDParticle)
          {
            eReference.setContainment(true);

            XSDParticle xsdParticle = (XSDParticle)xsdComponent;

            eReference.setLowerBound(xsdParticle.getMinOccurs());
            eReference.setUpperBound(xsdParticle.getMaxOccurs());

            EAnnotation eAnnotation = createAnnotation();
            XSDTerm xsdTerm = ((XSDParticle)xsdComponent).getTerm();
            if (xsdTerm instanceof XSDElementDeclaration)
            {
              XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)xsdTerm;
              if (xsdElementDeclaration.getTypeDefinition() instanceof XSDSimpleTypeDefinition)
              {
                eReference.setContainment(false);
              }

              addAnnotationDetail(eAnnotation, "representation", "element");
              addAnnotationDetail(eAnnotation, "name", ((XSDElementDeclaration)xsdTerm).getName());
              addAnnotationDetail(eAnnotation, "targetNamespace", ((XSDElementDeclaration)xsdTerm).getTargetNamespace());
            }
            else if (xsdTerm instanceof XSDWildcard)
            {
              addAnnotationDetail(eAnnotation, "representation", "element-wildcard");
            }
            else
            {
              addAnnotationDetail(eAnnotation, "representation", "general-content");
            }

            eReference.getEAnnotations().add(eAnnotation);
          }
        }

        return eReference;
      }
      else
      {
        EAttribute eAttribute = EcoreFactory.eINSTANCE.createEAttribute();
        eAttribute.setName(Character.toLowerCase(name.charAt(0)) + name.substring(1));
        eAttribute.setEType(type);
        eClass.getEAttributes().add(eAttribute);

        EAnnotation eAnnotation = createAnnotation();
        if (xsdComponent == null)
        {
          addAnnotationDetail(eAnnotation, "representation", "simple-content");
        }
        else
        {
          map(xsdComponent, eAttribute);
          if (xsdComponent instanceof XSDAttributeUse)
          {
            XSDAttributeUse xsdAttributeUse = (XSDAttributeUse)xsdComponent;
            XSDAttributeDeclaration xsdAttributeDeclaration = xsdAttributeUse.getAttributeDeclaration();
            addAnnotationDetail(eAnnotation, "representation", "attribute");
            addAnnotationDetail(eAnnotation, "name", xsdAttributeDeclaration.getName());
            addAnnotationDetail(eAnnotation, "targetNamespace", xsdAttributeDeclaration.getTargetNamespace());
            if (xsdAttributeUse.isRequired())
            {
              eAttribute.setLowerBound(1);
            }
          }
          else if (xsdComponent instanceof XSDParticle)
          {
            XSDParticle xsdParticle = (XSDParticle)xsdComponent;

            eAttribute.setLowerBound(xsdParticle.getMinOccurs());
            eAttribute.setUpperBound(xsdParticle.getMaxOccurs());

            XSDTerm xsdTerm = ((XSDParticle)xsdComponent).getTerm();
            if (xsdTerm instanceof XSDElementDeclaration)
            {
              addAnnotationDetail(eAnnotation, "representation", "element");
              addAnnotationDetail(eAnnotation, "name", ((XSDElementDeclaration)xsdTerm).getName());
              addAnnotationDetail(eAnnotation, "targetNamespace", ((XSDElementDeclaration)xsdTerm).getTargetNamespace());
            }
            else if (xsdTerm instanceof XSDWildcard)
            {
              addAnnotationDetail(eAnnotation, "representation", "element-wildcard");
            }
            else
            {
              addAnnotationDetail(eAnnotation, "representation", "general-content");
            }
          }
          else if (xsdComponent instanceof XSDWildcard)
          {
            addAnnotationDetail(eAnnotation, "representation", "attribute-wildcard");
          }
        }
        eAttribute.getEAnnotations().add(eAnnotation);

        return eAttribute;
      }
    }

    public List collectParticles(XSDParticle xsdParticle)
    {
      List result = new ArrayList();
      collectParticlesHelper(result, xsdParticle);
      return result;
    }

    public void collectParticlesHelper(List result, XSDParticle xsdParticle)
    {
      if ((style & TRIVIAL) != 0 || xsdParticle.getMaxOccurs() == 1)
      {
        XSDTerm xsdTerm = xsdParticle.getTerm();
        if (xsdTerm instanceof XSDModelGroup)
        {
          for (Iterator i = ((XSDModelGroup)xsdTerm).getParticles().iterator(); i.hasNext(); )
          {
            XSDParticle childXSDParticle = (XSDParticle)i.next();
            collectParticlesHelper(result, childXSDParticle);
          }
        }
        else
        {
          result.add(xsdParticle);
        }
      }
      else
      {
        result.add(xsdParticle);
      }
    }

    public EClassifier getEClassifier(XSDElementDeclaration xsdElementDeclaration)
    {
      EClassifier eClassifier = (EClassifier)xsdComponentToEModelElementMap.get(xsdElementDeclaration);
      if (eClassifier == null)
      {
        EClassifier type = getEClassifier(xsdElementDeclaration.getType());

        EClass elementEClass;
        if (xsdElementDeclaration.getAnonymousTypeDefinition() != null && type instanceof EClass)
        {
          // Fold in the interface for the anonymous type
          //
          elementEClass = (EClass)type;
          elementEClass.getEAnnotations().clear();
        }
        else
        {
          elementEClass = EcoreFactory.eINSTANCE.createEClass();
          EPackage ePackage = getEPackage(xsdElementDeclaration.getSchema().getTargetNamespace());
          ePackage.getEClassifiers().add(elementEClass);
        }

        elementEClass.setName(validName(xsdElementDeclaration.getName(), true));

        EAnnotation eAnnotation = createAnnotation();
        addAnnotationDetail(eAnnotation, "representation", "element");
        addAnnotationDetail(eAnnotation, "name", xsdElementDeclaration.getAliasName());
        addAnnotationDetail(eAnnotation, "targetNamespace", xsdElementDeclaration.getTargetNamespace());
        elementEClass.getEAnnotations().add(eAnnotation);

        XSDElementDeclaration substitutionGroupAffiliation = xsdElementDeclaration.getSubstitutionGroupAffiliation();
        if (substitutionGroupAffiliation != null)
        {
          elementEClass.getESuperTypes().add(getEClassifier(substitutionGroupAffiliation));
        }

        if (type instanceof EClass)
        {
          if (type != elementEClass && type != EcorePackage.eINSTANCE.getEObject())
          {
            elementEClass.getESuperTypes().add(type);
          }
        }
        else
        {
          if (elementEClass.getEAllReferences().isEmpty())
          {
            EStructuralFeature eStructuralFeature =
              createFeature
                (elementEClass,
                 "value",
                 type,
                 null);
          }
        }

        map(xsdElementDeclaration, elementEClass);

        eClassifier = elementEClass;
      }

      return eClassifier;
    }

    public Collection generate(URI uri)
    {
      ResourceSet resourceSet = new ResourceSetImpl();
      Resource resource = resourceSet.getResource(uri, true);
      if (!resource.getContents().isEmpty() && resource.getContents().get(0) instanceof XSDSchema)
      {
        generate((XSDSchema)resource.getContents().get(0));
      }

      List result = new ArrayList(targetNamespaceToEPackageMap.values());
      result.add(mappingRoot);
      return result;
    }

    public Collection generateResources(URI uri)
    {
      ResourceSet resourceSet = new ResourceSetImpl();
      Resource resource = resourceSet.getResource(uri, true);
      if (!resource.getContents().isEmpty() && resource.getContents().get(0) instanceof XSDSchema)
      {
        generate((XSDSchema)resource.getContents().get(0));
      }

      for (Iterator i = targetNamespaceToEPackageMap.values().iterator(); i.hasNext(); )
      {
        EPackage ePackage = (EPackage)i.next();
        Resource ecoreResource = resourceSet.createResource(URI.createURI("*.ecore"));
        ecoreResource.setURI
          (URI.createURI(ePackage.getNsURI().endsWith(".ecore") ? ePackage.getNsURI() : ePackage.getNsURI() + ".ecore"));
        ecoreResource.getContents().add(ePackage);
      }

      return resourceSet.getResources();
    }

    public void generate(XSDSchema xsdSchema)
    {
      this.xsdSchema = xsdSchema;
      addInput(xsdSchema);

      for (Iterator i = xsdSchema.getElementDeclarations().iterator(); i.hasNext(); )
      {
        XSDElementDeclaration xsdElementDeclaration = (XSDElementDeclaration)i.next();
        EClassifier eClassifier = getEClassifier(xsdElementDeclaration);
      }

      for (Iterator i = xsdSchema.getTypeDefinitions().iterator(); i.hasNext(); )
      {
        XSDTypeDefinition xsdTypeDefinition = (XSDTypeDefinition)i.next();
        EClassifier eClassifier = getEClassifier(xsdTypeDefinition);
      }
    }

    protected String validName(String name, boolean isUpperCase)
    {
      List parsedName = parseName(name, '_');
      StringBuffer result = new StringBuffer();
      for (Iterator i = parsedName.iterator(); i.hasNext(); )
      {
        String nameComponent = (String)i.next();
        if (nameComponent.length() > 0)
        {
          result.append(Character.toUpperCase(nameComponent.charAt(0)));
          result.append(nameComponent.substring(1));
        }
      }

      return
        result.length() == 0 ?
          "_" :
          Character.isJavaIdentifierStart(result.charAt(0)) ?
            isUpperCase ?
              result.toString() :
              uncapName(result.toString()) :
            "_" + result;
    }

    protected List parseName(String sourceName, char separator)
    {
      List result = new ArrayList();
      if (sourceName != null)
      {
        StringBuffer currentWord = new StringBuffer();
        boolean lastIsLower = false;
        for (int index = 0, length = sourceName.length(); index < length; ++index)
        {
          char curChar = sourceName.charAt(index);
          if (!Character.isJavaIdentifierPart(curChar))
          {
            curChar = separator;
          }
          if (Character.isUpperCase(curChar) || (!lastIsLower && Character.isDigit(curChar)) || curChar == separator)
          {
            if ((lastIsLower || curChar == separator) && currentWord.length() > 0)
            {
              result.add(currentWord.toString());
              currentWord = new StringBuffer();
            }
            lastIsLower = false;
          }
          else
          {
            if (!lastIsLower)
            {
              int currentWordLength = currentWord.length();
              if (currentWordLength > 1)
              {
                char lastChar = currentWord.charAt(--currentWordLength);
                currentWord.setLength(currentWordLength);
                result.add(currentWord.toString());
                currentWord = new StringBuffer();
                currentWord.append(lastChar);
              }
            }
            lastIsLower = true;
          }

          if (curChar != separator)
          {
            currentWord.append(curChar);
          }
        }

        result.add(currentWord.toString());
      }
      return result;
    }

    public String uncapName(String name)
    {
      if (name.length() == 0)
      {
        return name;
      }
      else
      {
        String lowerName = name.toLowerCase();
        int i;
        for (i = 0; i < name.length(); i++)
        {
          if (name.charAt(i) == lowerName.charAt(i))
          {
            break;
          }
        }
        if (i > 1 && i < name.length())
        {
          --i;
        }
        return name.substring(0, i).toLowerCase() + name.substring(i);
      }
    }
  }

  /**
   * This class produces an XSDSchema given an Ecore EPackage.
   * The XSDSchema is an XML schema.
   */
  public static class EcoreXMLSchemaBuilder extends MapBuilder
  {
    protected static final String EMF_SCHEMA_URI = "http://org.eclipse.emf/xsd";
    protected static final String EMF_SCHEMA_PREFIX = "emf";
    protected static final String EMF_SCHEMA_NAME = "EMF.xsd";
    protected static final String REFERENCE_TYPE_NAME = "string";

    protected XSDSchema xsdSchema;
    protected XSDSchema emfSchema;
    protected EPackage ePackage;
    protected Map ecoreToSchemaName;
    protected Map ePackageToXSDSchemaMap = new HashMap();
    protected XMLResource.XMLMap xmlMap;

    public EcoreXMLSchemaBuilder()
    {
    }

    public Collection generate(EPackage ePackage)
    {
       return generate(ePackage, null);
    }

    public Collection generate(EPackage ePackage, XMLResource.XMLMap map)
    {
      xmlMap = map;
      addInput(ePackage);
      // addOutput(ePackage);

      this.ePackage = ePackage;
      createSchema();
      processEnums();
      processClasses();
      Collection result = new ArrayList();
      result.add(xsdSchema);
      if (emfSchema != null)
      {
        result.add(emfSchema);
      }
      if (mappingRoot != null)
      {
        result.add(mappingRoot);
      }
      return result;
    }

    protected void createSchema()
    {
      xsdSchema = XSDFactory.eINSTANCE.createXSDSchema();
      addOutput(xsdSchema);
      // addInput(xsdSchema);

      xsdSchema.setTargetNamespace(ePackage.getNsURI());
      xsdSchema.setSchemaForSchemaQNamePrefix("xsd");

      Map namespaces = xsdSchema.getQNamePrefixToNamespaceMap();
      namespaces.put(ePackage.getNsPrefix(), xsdSchema.getTargetNamespace());
      namespaces.put(xsdSchema.getSchemaForSchemaQNamePrefix(), XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);
    }

    protected void createEMFSchema()
    {
      emfSchema = XSDFactory.eINSTANCE.createXSDSchema();

      emfSchema.setTargetNamespace(EMF_SCHEMA_URI);
      emfSchema.setSchemaForSchemaQNamePrefix("xsd");

      Map namespaces = emfSchema.getQNamePrefixToNamespaceMap();
      namespaces.put(EMF_SCHEMA_PREFIX, emfSchema.getTargetNamespace());
      namespaces.put(emfSchema.getSchemaForSchemaQNamePrefix(), XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);

      XSDSimpleTypeDefinition list = XSDFactory.eINSTANCE.createXSDSimpleTypeDefinition();
      list.setName(REFERENCE_TYPE_NAME);

      XSDSimpleTypeDefinition union = XSDFactory.eINSTANCE.createXSDSimpleTypeDefinition();
      union.getMemberTypeDefinitions().add(emfSchema.getSchemaForSchema().resolveSimpleTypeDefinition("IDREF"));
      union.getMemberTypeDefinitions().add(emfSchema.getSchemaForSchema().resolveSimpleTypeDefinition("QName"));
      union.getMemberTypeDefinitions().add(emfSchema.getSchemaForSchema().resolveSimpleTypeDefinition("anyURI"));

      list.setItemTypeDefinition(union);
      list.getContents().add(union);

      emfSchema.getContents().add(list);
    }

    protected void addEMFSchema()
    {
    }

    protected XSDTypeDefinition typeInOtherSchema(EClassifier classifier)
    {
      EPackage typePkg = classifier.getEPackage();
      Map namespaces = xsdSchema.getQNamePrefixToNamespaceMap();

      if (namespaces.get(typePkg.getNsPrefix()) == null)
      {
        namespaces.put(typePkg.getNsPrefix(), typePkg.getNsURI());
        addImport(typePkg.getNsURI(), getName(typePkg) + ".xsd");
        createOtherSchema(typePkg);
      }

      XSDSchema otherXSDSchema = (XSDSchema) ePackageToXSDSchemaMap.get(typePkg);
      return otherXSDSchema.resolveTypeDefinition(getName(classifier));
    }

    protected void addImport(String namespace, String schemaLocation)
    {
      XSDImport xsdImport = XSDFactory.eINSTANCE.createXSDImport();
      xsdImport.setNamespace(namespace);
      xsdImport.setSchemaLocation(schemaLocation);
      xsdSchema.getContents().add(0, xsdImport);
    }

    protected void createOtherSchema(EPackage ePackage)
    {
      XSDSchema otherSchema = XSDFactory.eINSTANCE.createXSDSchema();

      otherSchema.setTargetNamespace(ePackage.getNsURI());
      otherSchema.setSchemaForSchemaQNamePrefix("xsd");

      Map namespaces = otherSchema.getQNamePrefixToNamespaceMap();
      namespaces.put(ePackage.getNsPrefix(), otherSchema.getTargetNamespace());
      namespaces.put(otherSchema.getSchemaForSchemaQNamePrefix(), XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);

      ePackageToXSDSchemaMap.put(ePackage, otherSchema);
    }

    protected void processClasses()
    {
      for (Iterator classifiers = ePackage.getEClassifiers().iterator(); classifiers.hasNext(); )
      {
        EClassifier classifier = (EClassifier) classifiers.next();
        if (classifier instanceof EClass)
        {
          processClass((EClass) classifier);
        }
      }
    }

    protected void processEnums()
    {
      for (Iterator classifiers = ePackage.getEClassifiers().iterator(); classifiers.hasNext(); )
      {
        EClassifier classifier = (EClassifier) classifiers.next();
        if (classifier instanceof EEnum)
        {
          processEnum((EEnum) classifier);
        }
      }
    }

    /**
     * Return the XMLInfo object for the given element, if there is
     * one.
     */
    protected XMLResource.XMLInfo getInfo(ENamedElement element)
    {
      if (xmlMap == null)
      {
        return null;
      }
      else
      {
        return xmlMap.getInfo(element);
      }
    }

    /**
     * Return the given name, or the name from the XMLMap, if there is one.
     */
    protected String getName(ENamedElement element)
    {
      XMLResource.XMLInfo info = getInfo(element);

      if (info == null || info.getName() == null)
      {
        return element.getName();
      }
      else
      {
        return info.getName();
      }
    }

    /**
     * Each EEnum is mapped to a schema simple type that restricts the string simple type;
     * the schema type has an enumeration facet for each EEnumLiteral.
     */
    protected void processEnum(EEnum enum)
    {
      XSDSimpleTypeDefinition enumType = XSDFactory.eINSTANCE.createXSDSimpleTypeDefinition();
      enumType.setName(getName(enum));
      enumType.setBaseTypeDefinition(xsdSchema.getSchemaForSchema().resolveSimpleTypeDefinition("NCName"));
      xsdSchema.getContents().add(enumType);

      map(enumType, enum);

      for (Iterator literals = enum.getELiterals().iterator(); literals.hasNext(); )
      {
        EEnumLiteral literal = (EEnumLiteral) literals.next();
        XSDEnumerationFacet facet = XSDFactory.eINSTANCE.createXSDEnumerationFacet();
        facet.setLexicalValue(getName(literal));
        enumType.getFacetContents().add(facet);

        map(facet, literal);
      }
    }

    /**
     * Map each EClass to a schema complex type and an element declaration of that type;
     * the names of the complex type and element declaration are the name of the class.
     * If the class is abstract, only a complex type is created from it; there is no
     * corresponding element declaration.
     */
    protected void processClass(EClass eClass)
    {
      XSDComplexTypeDefinition xsdComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      xsdComplexTypeDefinition.setName(getName(eClass));
      EList superClasses = eClass.getESuperTypes();

      if (superClasses.size() > 0 )
      {
        xsdComplexTypeDefinition.setDerivationMethod(XSDDerivationMethod.EXTENSION_LITERAL);
        EClass superClass = (EClass) superClasses.get(0);

        if (superClass.getEPackage() == ePackage)
        {
          xsdComplexTypeDefinition.setBaseTypeDefinition(xsdSchema.resolveTypeDefinition(getName(superClass)));
        }
        else
        {
          xsdComplexTypeDefinition.setBaseTypeDefinition(typeInOtherSchema(superClass));
        }
      }

      xsdSchema.getContents().add(xsdComplexTypeDefinition);
      map(xsdComplexTypeDefinition, eClass);

      List features = getFeatures(eClass, superClasses);
      for (Iterator fs = features.iterator(); fs.hasNext(); )
      {
        EStructuralFeature f = (EStructuralFeature) fs.next();

        if (f instanceof EAttribute)
          processAttribute((EAttribute) f, xsdComplexTypeDefinition);
        else if (f instanceof EReference)
          processReference((EReference) f, xsdComplexTypeDefinition);
      }

      if (makeClassElementDeclaration(eClass))
      {
        XSDElementDeclaration xsdElementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
        xsdElementDeclaration.setName(getName(eClass));
        xsdElementDeclaration.setTypeDefinition(xsdComplexTypeDefinition);
        xsdSchema.getContents().add(xsdElementDeclaration);

        map(xsdElementDeclaration, eClass);
      }

      additionalProcessing(eClass, xsdComplexTypeDefinition);
    }

    protected boolean makeClassElementDeclaration(EClass eClass)
    {
      return !eClass.isAbstract();
    }

    protected void additionalProcessing(EClass cls, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
    }

    /**
     * Return the inherited features from the super classes (except the first one, which
     * is the base type for the generated schema type for the class), and the local
     * features.
     */
    protected List getFeatures(EClass eClass, List superClasses)
    {
      List features = new ArrayList();

      if (superClasses.size() > 0)
      {
        List allSupers = ((EClass) superClasses.get(0)).getEAllSuperTypes();
        Set allSuperClasses = new HashSet();
        allSuperClasses.addAll(allSupers);

        for (int i = 1; i < superClasses.size(); i++)
        {
          features.addAll(getAllFeatures((EClass) superClasses.get(i), allSuperClasses));
        }
      }

      features.addAll(eClass.getEAttributes());
      features.addAll(eClass.getEReferences());
      return features;
    }

    /**
     * This method returns the list of features for the given class and all inherited
     * features; ignoring features from classes in the classesToIgnore set.
     */
    protected List getAllFeatures(EClass eClass, Set classesToIgnore) {
       List features     = new ArrayList();
       List superClasses = eClass.getESuperTypes();

       for (int i = 0; i < superClasses.size(); i++)
          if (!classesToIgnore.contains(superClasses.get(i)))
             features.addAll(getAllFeatures((EClass) superClasses.get(i), classesToIgnore));

       features.addAll(eClass.getEAttributes());
       features.addAll(eClass.getEReferences());
       return features;
    }

    /**
     * Map each EAttribute to an XML attribute declaration in the complex type
     * corresponding to the class, or an XML element declaration, depending on
     * the multiplicity of the EAttribute.
     * <p>
     * The types are mapped as follows:
     *    Ecore type             Schema type
     *   ------------           -------------
     *     EBoolean                boolean
     *     EBooleanObject          boolean
     *     EInt                    int
     *     EIntegerObject          int
     *     ELong                   long
     *     ELongObject             long
     *     EFloat                  float
     *     EFloatObject            float
     *     EDouble                 double
     *     EDoubleObject           double
     *     EString                 string
     * <p>
     * If the attribute has no type or if the type is an EDataType that is not
     * an EEnum, the schema string type is used.
     * <p>
     * If the EAttribute is required, the attribute declaration includes
     * use="required".
     * <p>
     * If the EAttribute has a defaultValueLiteral, the attribute declaration
     * has a default attribute whose value is the defaultValueLiteral.
     * <p>
     * If the EAttribute is transient, or its type is not serializable, an
     * attribute declaration is not generated.
     * <p>
     * If the EAttribute can have many values, an element declaration is created
     * rather than an attribute declaration. The lower bound of the attribute is set
     * to the minOccurs value of the element declaration. The upper bound of the
     * attribute is set to the maxOccurs value of the element declaration (converting
     * -1 to unbounded as appropriate). The type of the element is set as above for
     * attribute declarations.
     */
    protected void processAttribute(EAttribute attribute, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      if (processAttribute(attribute))
      {
        if (makeAttributeDeclaration(attribute))
        {
          createAttributeDeclaration(attribute, xsdComplexTypeDefinition);
        }

        if (makeAttributeElementDeclaration())
        {
          createAttributeElementDeclaration(attribute, xsdComplexTypeDefinition);
        }
      }
    }

    protected void createAttributeDeclaration(EAttribute attribute, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDAttributeDeclaration attrDecl = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      attrDecl.setName(getName(attribute));
      setAttributeType(attribute, attrDecl);
      setDefaultValue(attribute, attrDecl);

      XSDAttributeUse attrUse = XSDFactory.eINSTANCE.createXSDAttributeUse();
      setUseToRequired(attribute, attrUse);
      attrUse.setContent(attrDecl);
      xsdComplexTypeDefinition.getAttributeContents().add(attrUse);

      map(attrUse, attribute);
    }

    protected void createAttributeElementDeclaration(EAttribute attribute, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDModelGroup modelGroup = getModelGroup(xsdComplexTypeDefinition);
      XSDElementDeclaration xsdElementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      xsdElementDeclaration.setName(getName(attribute));
      xsdElementDeclaration.setNillable(true);
      XSDSimpleTypeDefinition attrType = getType(attribute.getEAttributeType());

      if (attrType != null)
      {
        xsdElementDeclaration.setTypeDefinition(attrType);
      }

      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setContent(xsdElementDeclaration);
      setAttributeElementMultiplicity(attribute, particle);
      modelGroup.getContents().add(particle);
      map(particle, attribute);
    }

    protected void setAttributeElementMultiplicity(EAttribute attribute, XSDParticle particle)
    {
      if (attribute.isMany())
      {
        particle.setMinOccurs(attribute.getLowerBound());
      }
      else
      {
        particle.setMinOccurs(0);
      }

      particle.setMaxOccurs(attribute.getUpperBound());
    }

    protected boolean processAttribute(EAttribute attribute)
    {
      if (attribute.isTransient())
      {
        return false;
      }

      EDataType type = attribute.getEAttributeType();

      if (type != null && !type.isSerializable())
      {
          return false;
      }

      return true;
    }

    protected boolean makeAttributeDeclaration(EAttribute attribute)
    {
      return !attribute.isMany();
    }

    protected boolean makeAttributeElementDeclaration()
    {
      return true;
    }

    /**
     * Returns the model group for the given complex type definition. If there
     * is no model group, a model group is created and added to the complex
     * type definition.
     */
    protected XSDModelGroup getModelGroup(XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      if (xsdComplexTypeDefinition.getContent() == null)
      {
        return createModelGroup(xsdComplexTypeDefinition);
      }
      else
      {
        XSDParticle particle = (XSDParticle) xsdComplexTypeDefinition.getContent();
        return (XSDModelGroup) particle.getContent();
      }
    }

    protected XSDModelGroup createModelGroup(XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDModelGroup modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.SEQUENCE_LITERAL);
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setContent(modelGroup);
      xsdComplexTypeDefinition.setContent(particle);
      return modelGroup;
    }

    protected void setAttributeType(EAttribute attribute, XSDAttributeDeclaration attrDecl)
    {
      XSDSimpleTypeDefinition attrType = getType(attribute.getEAttributeType());

      if (attrType != null)
      {
        attrDecl.setTypeDefinition(attrType);
      }
    }

    protected void setUseToRequired(EAttribute attribute, XSDAttributeUse attrUse)
    {
    }

    protected void setDefaultValue(EAttribute attribute, XSDAttributeDeclaration attrDecl)
    {
      if (attribute.getDefaultValueLiteral() != null)
      {
        attrDecl.setConstraint(XSDConstraint.DEFAULT_LITERAL);
        attrDecl.setLexicalValue(attribute.getDefaultValueLiteral());
      }
    }

    protected void processReference(EReference reference, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      if (!skipReference(reference))
      {
        if (makeReferenceAttribute(reference))
        {
          makeReferenceAttribute(reference, xsdComplexTypeDefinition);
        }

        if (makeReferenceElement(reference))
        {
          makeReferenceElement(reference, xsdComplexTypeDefinition);
        }
      }
    }

    protected boolean makeReferenceAttribute(EReference reference)
    {
       return !reference.isContainment();
    }

    protected boolean makeReferenceElement(EReference reference)
    {
       return reference.isContainment();
    }

    protected boolean skipReference(EReference reference)
    {
      return reference.isTransient();
    }

    protected void makeReferenceAttribute(EReference reference, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDAttributeDeclaration attrDecl = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      attrDecl.setName(getName(reference));
      setReferenceAttribType(attrDecl);
      XSDAttributeUse attrUse = XSDFactory.eINSTANCE.createXSDAttributeUse();
      attrUse.setContent(attrDecl);
      xsdComplexTypeDefinition.getAttributeContents().add(attrUse);

      map(attrUse, reference);
    }

    protected void setReferenceAttribType(XSDAttributeDeclaration xsdAttributeDeclaration)
    {
      addEMFSchema();
      xsdAttributeDeclaration.setTypeDefinition(xsdSchema.getSchemaForSchema().resolveSimpleTypeDefinition(REFERENCE_TYPE_NAME));
    }

    protected void makeReferenceElement(EReference reference, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDModelGroup modelGroup = getModelGroup(xsdComplexTypeDefinition);
      XSDElementDeclaration xsdElementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      xsdElementDeclaration.setName(getName(reference));

      setReferenceElementType(reference, xsdElementDeclaration);

      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setContent(xsdElementDeclaration);

      setReferenceElementMultiplicity(reference, particle);

      modelGroup.getContents().add(particle);
      map(particle, reference);
    }

    protected void setReferenceElementType(EReference reference, XSDElementDeclaration xsdElementDeclaration)
    {
      if (reference.getEType() != null)
      {
        XSDTypeDefinition type;

        if (reference.getEType().getEPackage() == ePackage)
        {
          type = xsdSchema.resolveTypeDefinition(getName(reference.getEType()));
        }
        else
        {
          type = typeInOtherSchema(reference.getEType());
        }

        if (type != null)
        {
          xsdElementDeclaration.setTypeDefinition(type);
        }
      }
    }

    protected void setReferenceElementMultiplicity(EReference reference, XSDParticle particle)
    {
      particle.setMinOccurs(reference.getLowerBound());
      particle.setMaxOccurs(reference.getUpperBound());
    }

    protected XSDSimpleTypeDefinition getType(EDataType dataType)
    {
      if (dataType instanceof EEnum)
      {
        EPackage typePkg = dataType.getEPackage();

        if (typePkg == ePackage)
        {
           return xsdSchema.resolveSimpleTypeDefinition(getName(dataType));
        }
        else
        {
           return (XSDSimpleTypeDefinition) typeInOtherSchema(dataType);
        }
      }

      String name = null;

      if (dataType != null)
      {
        name = getName(dataType);
      }

      return xsdSchema.getSchemaForSchema().resolveSimpleTypeDefinition(getSchemaName(name));
    }

    private String getSchemaName(String name)
    {
      if (ecoreToSchemaName == null)
      {
        ecoreToSchemaName = new HashMap();
        ecoreToSchemaName.put("EBoolean", "boolean");
        ecoreToSchemaName.put("EBooleanObject", "boolean");
        ecoreToSchemaName.put("EInt", "int");
        ecoreToSchemaName.put("EIntegerObject", "int");
        ecoreToSchemaName.put("ELong", "long");
        ecoreToSchemaName.put("ELongObject", "long");
        ecoreToSchemaName.put("EFloat", "float");
        ecoreToSchemaName.put("EFloatObject", "float");
        ecoreToSchemaName.put("EDouble", "double");
        ecoreToSchemaName.put("EDoubleObject", "double");
        ecoreToSchemaName.put("EString", "string");
      }

      String schemaName = (String) ecoreToSchemaName.get(name);

      if (schemaName != null)
      {
        return schemaName;
      }
      else
      {
        return "string";
      }
    }
  }

  /**
   * This class produces an XSDSchema given an Ecore EPackage.
   * The XSDSchema is an XMI schema.
   */
  public static class EcoreXMISchemaBuilder extends EcoreXMLSchemaBuilder
  {
    protected static final String XMI_PREFIX = "xmi";
    protected static final String XMI_URI    = "http://www.omg.org/XMI";
    protected static final String XMI_SCHEMA_LOCATION = "XMI.xsd";

    public EcoreXMISchemaBuilder()
    {
      super();
    }

    public Collection generate(EPackage ePackage)
    {
       return generate(ePackage, null);
    }

    public Collection generate(EPackage ePackage, XMLResource.XMLMap map)
    {
      List result = (List) super.generate(ePackage, map);
      result.add(1, createXMISchema());
      return result;
    }

    protected void createSchema()
    {
      super.createSchema();

      Map namespaces = this.xsdSchema.getQNamePrefixToNamespaceMap();
      namespaces.put(XMI_PREFIX, XMI_URI);

      XSDImport xmiImport = XSDFactory.eINSTANCE.createXSDImport();
      xmiImport.setNamespace(XMI_URI);
      xmiImport.setSchemaLocation(XMI_SCHEMA_LOCATION);
      this.xsdSchema.getContents().add(xmiImport);
    }

    protected XSDSchema createXMISchema()
    {
      XSDSchema xmiSchema = XSDFactory.eINSTANCE.createXSDSchema();
      xmiSchema.setTargetNamespace(XMI_URI);
      xmiSchema.setSchemaForSchemaQNamePrefix("xsd");

      Map namespaces = xmiSchema.getQNamePrefixToNamespaceMap();
      namespaces.put(XMI_PREFIX, XMI_URI);
      namespaces.put("xsd", XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);

      // <xsd:attribute name="id" type="xsd:ID"/>
      XSDAttributeDeclaration xmiIdAttribute = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      xmiIdAttribute.setName("id");
      xmiIdAttribute.setTypeDefinition(xmiSchema.getSchemaForSchema().resolveSimpleTypeDefinition("ID"));
      xmiSchema.getContents().add(xmiIdAttribute);

      // <xsd:attributeGroup name="IdentityAttribs">
      //   <xsd:attribute name="label" type="xsd:string" use="optional" form="qualified"/>
      //   <xsd:attribute name="uuid" type="xsd:string" use="optional" form="qualified"/>
      // </xsd:attributeGroup>
      XSDAttributeGroupDefinition xmiIdentityAttribs = XSDFactory.eINSTANCE.createXSDAttributeGroupDefinition();
      xmiIdentityAttribs.setName("IdentityAttribs");
      List contents = xmiIdentityAttribs.getContents();
      contents.add(createAttributeUse(xmiSchema, "label", "string", "optional", "qualified", null));
      contents.add(createAttributeUse(xmiSchema, "uuid", "string", "optional", "qualified", null));
      xmiSchema.getContents().add(xmiIdentityAttribs);

      // <xsd:attributeGroup name="LinkAttribs">
      //   <xsd:attribute name="href" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="idref" type="xsd:IDREF" use="optional" form="qualified"/>
      // </xsd:attributeGroup>
      XSDAttributeGroupDefinition xmiLinkAttribs = XSDFactory.eINSTANCE.createXSDAttributeGroupDefinition();
      xmiLinkAttribs.setName("LinkAttribs");
      contents = xmiLinkAttribs.getContents();
      contents.add(createAttributeUse(xmiSchema, "href", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "idref", "IDREF", "optional", "qualified", null));
      xmiSchema.getContents().add(xmiLinkAttribs);

      // <xsd:attributeGroup name="ObjectAttribs">
      //   <xsd:attributeGroup ref="IdentityAttribs"/>
      //   <xsd:attributeGroup ref="LinkAttribs"/>
      //   <xsd:attribute name="version" type="xsd:string" use="optional" fixed="2.0" form="qualified"/>
      //   <xsd:attribute name="type" type="xsd:QName" use="optional" form="qualified"/>
      // </xsd:attributeGroup>
      XSDAttributeGroupDefinition xmiObjectAttribs = XSDFactory.eINSTANCE.createXSDAttributeGroupDefinition();
      xmiObjectAttribs.setName("ObjectAttribs");
      contents = xmiObjectAttribs.getContents();
      contents.add(createAttributeGroupReference(xmiSchema, "IdentityAttribs"));
      contents.add(createAttributeGroupReference(xmiSchema, "LinkAttribs"));
      contents.add(createAttributeUse(xmiSchema, "version", "string", "optional", "qualified", "2.0"));
      contents.add(createAttributeUse(xmiSchema, "type", "QName", "optional", "qualified", null));
      xmiSchema.getContents().add(xmiObjectAttribs);

      // <xsd:complexType name="XMI">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:any processContents="strict"/>
      //   </xsd:choice>
      //   <xsd:attribute ref="id"/>
      //   <xsd:attributeGroup ref="IdentityAttribs"/>
      //   <xsd:attributeGroup ref="LinkAttribs"/>
      //   <xsd:attribute name="type" type="xsd:QName" use="optional" form="qualified"/>
      //   <xsd:attribute name="version" type="xsd:string" use="required" fixed="2.0" form="qualified"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition xmiComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      xmiComplexTypeDefinition.setName("XMI");
      // choice model group
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      XSDModelGroup modelGroup = createAnyModelGroup("strict");
      particle.setContent(modelGroup);
      xmiComplexTypeDefinition.setContent(particle);
      contents = xmiComplexTypeDefinition.getAttributeContents();
      contents.add(createAttributeReference(xmiSchema, "id"));
      contents.add(createAttributeGroupReference(xmiSchema, "IdentityAttribs"));
      contents.add(createAttributeGroupReference(xmiSchema, "LinkAttribs"));
      contents.add(createAttributeUse(xmiSchema, "type", "QName", "optional", "qualified", null));
      contents.add(createAttributeUse(xmiSchema, "version", "string", "required", "qualified", "2.0"));
      xmiSchema.getContents().add(xmiComplexTypeDefinition);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "XMI", "XMI", false));

      // <xsd:complexType name="PackageReference">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:element name="name" type="xsd:string"/>
      //     <xsd:element name="version" type="xsd:string"/>
      //     <xsd:element ref="Extension"/>
      //   </xsd:choice>
      //   <xsd:attribute ref="id"/>
      //   <xsd:attributeGroup ref="ObjectAttribs"/>
      //   <xsd:attribute name="name" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="version" type="xsd:string" use="optional"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition prComplexTypeDefinition = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      prComplexTypeDefinition.setName("PackageReference");
      // choice model group
      particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.CHOICE_LITERAL);
      addElementDeclaration(xmiSchema, modelGroup, "name", "string");
      addElementDeclaration(xmiSchema, modelGroup, "version", "string");
      addElementDeclarationReference(xmiSchema, modelGroup, "Extension");
      particle.setContent(modelGroup);
      prComplexTypeDefinition.setContent(particle);
      contents = prComplexTypeDefinition.getAttributeContents();
      contents.add(createAttributeReference(xmiSchema, "id"));
      contents.add(createAttributeGroupReference(xmiSchema, "ObjectAttribs"));
      contents.add(createAttributeUse(xmiSchema, "name", "string", "optional", null, null));
      // EATM contents.add(createAttributeUse(xmiSchema, "version", "string", "optional", null, null));
      xmiSchema.getContents().add(prComplexTypeDefinition);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "PackageReference", "PackageReference", false));

      // <xsd:complexType name="Model">
      //   <xsd:complexContent>
      //     <xsd:extension base="PackageReference"/>
      //   </xsd:complexContent>
      // </xsd:complexType>
      xmiSchema.getContents().add(createExtendedComplexTypeDefinition(xmiSchema, "Model", "PackageReference"));

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Model", "Model", false));

      // <xsd:complexType name="Import">
      //   <xsd:complexContent>
      //     <xsd:extension base="PackageReference"/>
      //   </xsd:complexContent>
      // </xsd:complexType>
      xmiSchema.getContents().add(createExtendedComplexTypeDefinition(xmiSchema, "Import", "PackageReference"));

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Import", "Import", false));

      // <xsd:complexType name="MetaModel">
      //   <xsd:complexContent>
      //     <xsd:extension base="PackageReference"/>
      //   </xsd:complexContent>
      // </xsd:complexType>
      xmiSchema.getContents().add(createExtendedComplexTypeDefinition(xmiSchema, "MetaModel", "PackageReference"));

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "MetaModel", "MetaModel", false));

      // <xsd:complexType name="Documentation">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:element name="contact" type="xsd:string"/>
      //     <xsd:element name="exporter" type="xsd:string"/>
      //     <xsd:element name="exporterVersion" type="xsd:string"/>
      //     <xsd:element name="longDescription" type="xsd:string"/>
      //     <xsd:element name="shortDescription" type="xsd:string"/>
      //     <xsd:element name="notice" type="xsd:string"/>
      //     <xsd:element name="owner" type="xsd:string"/>
      //     <xsd:element ref="Extension"/>
      //   </xsd:choice>
      //   <xsd:attribute ref="id"/>
      //   <xsd:attributeGroup ref="ObjectAttribs"/>
      //   <xsd:attribute name="contact" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="exporter" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="exporterVersion" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="longDescription" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="shortDescription" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="notice" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="owner" type="xsd:string" use="optional"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition documentationComplexType = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      documentationComplexType.setName("Documentation");
      // choice model group
      particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.CHOICE_LITERAL);
      addElementDeclaration(xmiSchema, modelGroup, "contact", "string");
      addElementDeclaration(xmiSchema, modelGroup, "exporter", "string");
      addElementDeclaration(xmiSchema, modelGroup, "exporterVersion", "string");
      addElementDeclaration(xmiSchema, modelGroup, "longDescription", "string");
      addElementDeclaration(xmiSchema, modelGroup, "shortDescription", "string");
      addElementDeclaration(xmiSchema, modelGroup, "notice", "string");
      addElementDeclaration(xmiSchema, modelGroup, "owner", "string");
      addElementDeclarationReference(xmiSchema, modelGroup, "Extension");
      particle.setContent(modelGroup);
      documentationComplexType.setContent(particle);
      contents = documentationComplexType.getAttributeContents();
      contents.add(createAttributeReference(xmiSchema, "id"));
      contents.add(createAttributeGroupReference(xmiSchema, "ObjectAttribs"));
      contents.add(createAttributeUse(xmiSchema, "contact", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "exporter", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "exporterVersion", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "longDescription", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "shortDescription", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "notice", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "owner", "string", "optional", null, null));
      xmiSchema.getContents().add(documentationComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Documentation", "Documentation", false));

      // <xsd:complexType name="Extension">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:any processContents="lax"/>
      //   </xsd:choice>
      //   <xsd:attribute ref="id"/>
      //   <xsd:attributeGroup ref="ObjectAttribs"/>
      //   <xsd:attribute name="extender" type="xsd:string" use="optional"/>
      //   <xsd:attribute name="extenderID" type="xsd:string" use="optional"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition extensionComplexType = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      extensionComplexType.setName("Extension");
      // choice model group
      particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      modelGroup = createAnyModelGroup("lax");
      particle.setContent(modelGroup);
      extensionComplexType.setContent(particle);
      contents = extensionComplexType.getAttributeContents();
      contents.add(createAttributeReference(xmiSchema, "id"));
      contents.add(createAttributeGroupReference(xmiSchema, "ObjectAttribs"));
      contents.add(createAttributeUse(xmiSchema, "extender", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "extenderID", "string", "optional", null, null));
      xmiSchema.getContents().add(extensionComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Extension", "Extension", false));

      // <xsd:complexType name="Difference">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:element name="target">
      //       <xsd:complexType>
      //         <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //           <xsd:any processContents="skip"/>
      //         </xsd:choice>
      //         <xsd:anyAttribute processContents="skip"/>
      //       </xsd:complexType>
      //     </xsd:element>
      //     <xsd:element name="difference" type="Difference"/>
      //     <xsd:element name="container" type="Difference"/>
      //     <xsd:element ref="Extension"/>
      //   </xsd:choice>
      //   <xsd:attribute ref="id"/>
      //   <xsd:attributeGroup ref="ObjectAttribs"/>
      //   <xsd:attribute name="target" type="xsd:IDREFS" use="optional"/>
      //   <xsd:attribute name="container" type="xsd:IDREFS" use="optional"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition differenceComplexType = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      differenceComplexType.setName("Difference");
      particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.CHOICE_LITERAL);
      XSDParticle particle2 = XSDFactory.eINSTANCE.createXSDParticle();
      XSDElementDeclaration target = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      target.setName("target");
      XSDComplexTypeDefinition anonymous = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      XSDParticle particle3 = XSDFactory.eINSTANCE.createXSDParticle();
      particle3.setMinOccurs(0);
      particle3.setMaxOccurs(-1);
      particle3.setContent(createAnyModelGroup("skip"));
      anonymous.setContent(particle3);
      anonymous.setAttributeWildcardContent(createAny("skip"));
      target.setAnonymousTypeDefinition(anonymous);
      particle2.setContent(target);
      modelGroup.getContents().add(particle2);
      addElementDeclaration(xmiSchema, modelGroup, "difference", "Difference", false);
      addElementDeclaration(xmiSchema, modelGroup, "container", "Difference", false);
      addElementDeclarationReference(xmiSchema, modelGroup, "Extension");
      particle.setContent(modelGroup);
      differenceComplexType.setContent(particle);
      contents = differenceComplexType.getAttributeContents();
      contents.add(createAttributeReference(xmiSchema, "id"));
      contents.add(createAttributeGroupReference(xmiSchema, "ObjectAttribs"));
      contents.add(createAttributeUse(xmiSchema, "target", "IDREFS", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "container", "IDREFS", "optional", null, null));
      xmiSchema.getContents().add(differenceComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Difference", "Difference", false));

      // <xsd:complexType name="Add">
      //   <xsd:complexContent>
      //     <xsd:extension base="Difference">
      //       <xsd:attribute name="position" type="xsd:string" use="optional"/>
      //       <xsd:attribute name="addition" type="xsd:IDREFS" use="optional"/>
      //    </xsd:extension>
      //   </xsd:complexContent>
      // </xsd:complexType>
      XSDComplexTypeDefinition addComplexType = createExtendedComplexTypeDefinition(xmiSchema, "Add", "Difference");
      contents = addComplexType.getAttributeContents();
      contents.add(createAttributeUse(xmiSchema, "position", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "addition", "IDREFS", "optional", null, null));
      xmiSchema.getContents().add(addComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Add", "Add", false));

      // <xsd:complexType name="Replace">
      //   <xsd:complexContent>
      //     <xsd:extension base="Difference">
      //       <xsd:attribute name="position" type="xsd:string" use="optional"/>
      //       <xsd:attribute name="replacement" type="xsd:IDREFS" use="optional"/>
      //     </xsd:extension>
      //   </xsd:complexContent>
      // </xsd:complexType>
      XSDComplexTypeDefinition replaceComplexType = createExtendedComplexTypeDefinition(xmiSchema, "Replace", "Difference");
      contents = replaceComplexType.getAttributeContents();
      contents.add(createAttributeUse(xmiSchema, "position", "string", "optional", null, null));
      contents.add(createAttributeUse(xmiSchema, "replacement", "IDREFS", "optional", null, null));
      xmiSchema.getContents().add(replaceComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Replace", "Replace", false));

      // <xsd:complexType name="Delete">
      //   <xsd:complexContent>
      //     <xsd:extension base="Difference"/>
      //   </xsd:complexContent>
      // </xsd:complexType>
      XSDComplexTypeDefinition deleteComplexType = createExtendedComplexTypeDefinition(xmiSchema, "Delete", "Difference");
      xmiSchema.getContents().add(deleteComplexType);

      xmiSchema.getContents().add(createElementDeclaration(xmiSchema, "Delete", "Delete", false));

      // <xsd:complexType name="Any">
      //   <xsd:choice minOccurs="0" maxOccurs="unbounded">
      //     <xsd:any processContents="skip"/>
      //   </xsd:choice>
      //   <xsd:anyAttribute processContents="skip"/>
      // </xsd:complexType>
      XSDComplexTypeDefinition xmiAny = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      xmiAny.setName("Any");
      particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      particle.setContent(createAnyModelGroup("skip"));
      xmiAny.setContent(particle);
      xmiAny.setAttributeWildcardContent(createAny("skip"));
      xmiSchema.getContents().add(xmiAny);

      return xmiSchema;
    }

    protected XSDModelGroup createAnyModelGroup(String processContents)
    {
      XSDModelGroup modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.CHOICE_LITERAL);
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setContent(createAny(processContents));
      modelGroup.getContents().add(particle);
      return modelGroup;
    }

    protected XSDWildcard createAny(String processContents)
    {
      XSDWildcard any = XSDFactory.eINSTANCE.createXSDWildcard();

      if ("strict".equals(processContents))
      {
        any.setProcessContents(XSDProcessContents.STRICT_LITERAL);
      }
      else if ("skip".equals(processContents))
      {
        any.setProcessContents(XSDProcessContents.SKIP_LITERAL);
      }
      else if ("lax".equals(processContents))
      {
        any.setProcessContents(XSDProcessContents.LAX_LITERAL);
      }

      return any;
    }

    protected XSDComplexTypeDefinition createExtendedComplexTypeDefinition(XSDSchema schema, String name, String extension)
    {
      XSDComplexTypeDefinition complexType = XSDFactory.eINSTANCE.createXSDComplexTypeDefinition();
      complexType.setName(name);
      complexType.setDerivationMethod(XSDDerivationMethod.EXTENSION_LITERAL);
      complexType.setBaseTypeDefinition(schema.resolveTypeDefinition(extension));
      return complexType;
    }

    protected XSDAttributeGroupDefinition createAttributeGroupReference(XSDSchema schema, String name)
    {
      XSDAttributeGroupDefinition reference = XSDFactory.eINSTANCE.createXSDAttributeGroupDefinition();
      reference.setResolvedAttributeGroupDefinition(schema.resolveAttributeGroupDefinition(name));
      return reference;
    }

    protected XSDElementDeclaration createElementDeclaration(XSDSchema schema, String name, String type, boolean schemaType)
    {
      XSDElementDeclaration elementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      elementDeclaration.setName(name);

      if (schemaType)
      {
         elementDeclaration.setTypeDefinition(schema.getSchemaForSchema().resolveSimpleTypeDefinition(type));
      }
      else
      {
         elementDeclaration.setTypeDefinition(schema.resolveSimpleTypeDefinition(type));
      }

      return elementDeclaration;
    }

    protected void addElementDeclaration(XSDSchema schema, XSDModelGroup modelGroup, String name, String type)
    {
      addElementDeclaration(schema, modelGroup, name, type, true);
    }

    protected void addElementDeclaration(XSDSchema schema, XSDModelGroup modelGroup, String name, String type, boolean schemaType)
    {
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setContent(createElementDeclaration(schema, name, type, schemaType));
      modelGroup.getContents().add(particle);
    }

    protected void addElementDeclarationReference(XSDSchema schema, XSDModelGroup modelGroup, String reference)
    {
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      XSDElementDeclaration elementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      elementDeclaration.setResolvedElementDeclaration(schema.resolveElementDeclaration(reference));
      particle.setContent(elementDeclaration);
      modelGroup.getContents().add(particle);
    }

    protected XSDAttributeUse createAttributeReference(XSDSchema schema, String name)
    {
      XSDAttributeDeclaration attributeDeclaration = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      attributeDeclaration.setResolvedAttributeDeclaration(schema.resolveAttributeDeclaration(name));
      XSDAttributeUse attributeUse = XSDFactory.eINSTANCE.createXSDAttributeUse();
      attributeUse.setContent(attributeDeclaration);
      return attributeUse;
    }

    protected XSDAttributeUse createAttributeUse(XSDSchema schema, String name, String type, String use, String form, String fixed)
    {
      XSDAttributeDeclaration attributeDeclaration = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      attributeDeclaration.setName(name);
      attributeDeclaration.setTypeDefinition(schema.getSchemaForSchema().resolveSimpleTypeDefinition(type));

      if ("qualified".equals(form))
      {
        attributeDeclaration.setForm(XSDForm.QUALIFIED_LITERAL);
      }

      XSDAttributeUse attributeUse = XSDFactory.eINSTANCE.createXSDAttributeUse();
      attributeUse.setContent(attributeDeclaration);

      if ("optional".equals(use))
      {
         attributeUse.setUse(XSDAttributeUseCategory.OPTIONAL_LITERAL);
      }

      if ("required".equals(use))
      {
         attributeUse.setUse(XSDAttributeUseCategory.REQUIRED_LITERAL);
      }

      if (fixed != null)
      {
        attributeUse.setConstraint(XSDConstraint.FIXED_LITERAL);
        attributeUse.setLexicalValue("2.0");
      }

      return attributeUse;
    }

    protected boolean makeClassElementDeclaration(EClass eClass)
    {
      return true;
    }

    protected XSDModelGroup createModelGroup(XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDModelGroup modelGroup = XSDFactory.eINSTANCE.createXSDModelGroup();
      modelGroup.setCompositor(XSDCompositor.CHOICE_LITERAL);
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      particle.setMinOccurs(0);
      particle.setMaxOccurs(-1);
      particle.setContent(modelGroup);
      xsdComplexTypeDefinition.setContent(particle);
      return modelGroup;
    }

    protected void setAttributeElementMultiplicity(EAttribute attribute, XSDParticle particle)
    {
    }

    protected void additionalProcessing(EClass eClass, XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      if (eClass.getESuperTypes().size() == 0)
      {
        addXMIExtension(getModelGroup(xsdComplexTypeDefinition));
        addXMIAttributes(xsdComplexTypeDefinition);
      }
    }

    protected void addXMIExtension(XSDModelGroup modelGroup)
    {
      XSDParticle particle = XSDFactory.eINSTANCE.createXSDParticle();
      XSDElementDeclaration xsdElementDeclaration = XSDFactory.eINSTANCE.createXSDElementDeclaration();
      xsdElementDeclaration.setResolvedElementDeclaration(xsdElementDeclaration.resolveElementDeclaration(XMI_URI, "Extension"));
      particle.setContent(xsdElementDeclaration);
      modelGroup.getContents().add(particle);
    }

    protected void addXMIAttributes(XSDComplexTypeDefinition xsdComplexTypeDefinition)
    {
      XSDAttributeGroupDefinition objAttribs = XSDFactory.eINSTANCE.createXSDAttributeGroupDefinition();
      objAttribs.setResolvedAttributeGroupDefinition(objAttribs.resolveAttributeGroupDefinition(XMI_URI, "ObjectAttribs"));
      xsdComplexTypeDefinition.getAttributeContents().add(0, objAttribs);

      XSDAttributeDeclaration attrDecl = XSDFactory.eINSTANCE.createXSDAttributeDeclaration();
      attrDecl.setResolvedAttributeDeclaration(attrDecl.resolveAttributeDeclaration(XMI_URI, "id"));
      XSDAttributeUse attrUse = XSDFactory.eINSTANCE.createXSDAttributeUse();
      attrUse.setContent(attrDecl);
      xsdComplexTypeDefinition.getAttributeContents().add(0, attrUse);
    }

    protected void setDefaultValue(EAttribute attribute, XSDAttributeDeclaration attrDecl)
    {
    }

    protected boolean makeReferenceElement(EReference reference)
    {
       return true;
    }

    protected void setReferenceElementType(EReference reference, XSDElementDeclaration xsdElementDeclaration)
    {
      if (reference.getEType() != null)
      {
        super.setReferenceElementType(reference, xsdElementDeclaration);
      }
      else
      {
        xsdElementDeclaration.setTypeDefinition(xsdElementDeclaration.resolveSimpleTypeDefinition(XMI_URI, "Any"));
      }
    }

    protected void setReferenceElementMultiplicity(EReference reference, XSDParticle particle)
    {
    }
  }

  public static class GenericXMLResourceFactoryImpl extends XMLResourceFactoryImpl
  {
    protected XMLResource.XMLMap xmlMap = new XMLMapImpl();

    public GenericXMLResourceFactoryImpl()
    {
      super();
    }

    public Resource createResource(URI uri)
    {
      XMLResource result = new GenericXMLResourceImpl(uri);
      result.getDefaultSaveOptions().put(XMLResource.OPTION_XML_MAP, xmlMap);
      result.getDefaultLoadOptions().put(XMLResource.OPTION_XML_MAP, xmlMap);
      return result;
    }
  }

  public static class GenericXMLResourceImpl extends XMLResourceImpl
  {
    protected XSDEcoreBuilder xsdEcoreBuilder;
    protected XMLHelper xmlHelper;

    public GenericXMLResourceImpl(URI uri)
    {
      super(uri);
    }

    protected XMLHelper createXMLHelper()
    {
      if (xmlHelper == null)
      {
        xmlHelper =
          new XMLHelperImpl(this)
          {
            protected String getQName(String uri, String name)
            {
              if (uri == null)
              {
                return name;
              }

              EPackage pkg = (EPackage)xsdEcoreBuilder.getTargetNamespaceToEPackageMap().get(uri);
              if (pkg == null || pkg.getNsPrefix().equals(""))
              {
                return name;
              }
              else
              {
                packages.put(pkg, null);
                return pkg.getNsPrefix() + ":" + name;
              }
            }
          };
      }
      return xmlHelper;
    }

    public void doLoad(InputStream inputStream, final Map op) throws IOException
    {
      XMLLoad xmlStart =
        new XMLLoadImpl(createXMLHelper())
        {
          protected DefaultHandler makeDefaultHandler()
          {
            SAXXMLHandler saxXMLHandler =
              new SAXXMLHandler(resource, helper, op)
              {
                protected MyStack elementDeclarations = new MyStack();
                protected MyStack dfaStates = new MyStack();

                protected void createTopObject(String prefix, String name)
                {
                  String namespaceURI = helper.getURI(prefix);
                  for (int i = 0, size = attribs.getLength(); i < size; ++i)
                  {
                    String attributeName = attribs.getQName(i);
                    int index = attributeName.indexOf(":");
                    String attributeNamespaceURI = null;
                    String attributeLocalName = attributeName;
                    if (index != -1)
                    {
                      attributeNamespaceURI = helper.getURI(attributeName.substring(0, index));
                      attributeLocalName = attributeName.substring(index + 1);
                    }

                    if (XSDConstants.SCHEMA_INSTANCE_URI_2001.equals(attributeNamespaceURI) &&
                         (namespaceURI == null ? "noSchemaLocation" : "schemaLocation").equals(attributeLocalName))
                    {
                      String schemaLocationHint = null;

                      if (namespaceURI == null)
                      {
                        schemaLocationHint = attribs.getValue(i);
                      }
                      else
                      {
                        for (StringTokenizer stringTokenizer = new StringTokenizer(attribs.getValue(i));
                             stringTokenizer.hasMoreTokens(); )
                        {
                          String namespaceURIHint = stringTokenizer.nextToken();
                          if (stringTokenizer.hasMoreTokens())
                          {
                            if (namespaceURIHint.equals(namespaceURI))
                            {
                              schemaLocationHint = stringTokenizer.nextToken();
                              break;
                            }
                            else
                            {
                              stringTokenizer.nextToken();
                            }
                          }
                          else
                          {
                            break;
                          }
                        }
                      }

                      if (schemaLocationHint != null)
                      {
                        URI uri = URI.createDeviceURI(schemaLocationHint);
                        if (resolve && uri.isRelative() && uri.hasRelativePath())
                        {
                          uri = uri.resolve(resourceURI);
                        }

                        xsdEcoreBuilder = new XSDEcoreBuilder(XSDEcoreBuilder.TRIVIAL);
                        Collection resources = xsdEcoreBuilder.generateResources(uri);
                        resource.getResourceSet().getResources().addAll(resources);
                      }
                    }
                  }

                  if (xsdEcoreBuilder == null)
                  {
                    error(new XMIException("Cannot resolve schema location",  getLocation(), getLineNumber(), getColumnNumber()));
                  }
                  else
                  {
                    XSDElementDeclaration xsdElementDeclaration =
                      xsdEcoreBuilder.getSchema().resolveElementDeclaration(namespaceURI, name);
                    EClass eClass = (EClass)xsdEcoreBuilder.getXSDComponentToEModelElementMap().get(xsdElementDeclaration);
                    if (eClass != null)
                    {
                      processTopObject(eClass.getEPackage().getEFactoryInstance().create(eClass));
                      elementDeclarations.push(xsdElementDeclaration);
                      XSDParticle xsdParticle = xsdElementDeclaration.getTypeDefinition().getComplexType();
                      if (xsdParticle != null)
                      {
                        dfaStates.push(xsdParticle.getDFA().getInitialState());
                      }
                      else
                      {
                        dfaStates.push(null);
                      }
                    }
                    else
                    {
                      error(new XMIException("Cannot resolve EClass ",  getLocation(), getLineNumber(), getColumnNumber()));
                    }
                  }
                }

                protected void processElement(String name, String prefix, String localName)
                {
                  if (isError())
                  {
                    types.push(ERROR_TYPE);
                  }
                  else
                  {
                    if (objects.isEmpty())
                    {
                      createTopObject(prefix, localName);

                    }
                    else
                    {
                      EObject peekObject = (EObject)objects.peek();
                      XSDParticle.DFA.State state = (XSDParticle.DFA.State)dfaStates.peek();
                      if (state == null)
                      {
                        error(new XMIException("Cannot contain content ",  getLocation(), getLineNumber(), getColumnNumber()));
                      }
                      else
                      {
                        XSDParticle.DFA.Transition transition = state.accept(helper.getURI(prefix), localName);
                        if (transition == null)
                        {
                          error(new XMIException("Not expecting this element ",  getLocation(), getLineNumber(), getColumnNumber()));
                        }
                        else
                        {
                          dfaStates.set(dfaStates.size() - 1, transition.getState());

                          XSDParticle transitionXSDParticle = transition.getParticle();
                          XSDTerm xsdTerm = transitionXSDParticle.getTerm();
                          XSDElementDeclaration xsdElementDeclaration = null;
                          if (xsdTerm instanceof XSDElementDeclaration)
                          {
                            xsdElementDeclaration =  (XSDElementDeclaration)xsdTerm;
                          }
                          else
                          {
                            xsdElementDeclaration = xsdEcoreBuilder.getSchema().resolveElementDeclaration(helper.getURI(prefix), name);
                          }

                          EClass eClass = (EClass)xsdEcoreBuilder.getXSDComponentToEModelElementMap().get(xsdElementDeclaration);
                          if (eClass != null)
                          {
                            EObject eObject = eClass.getEPackage().getEFactoryInstance().create(eClass);
                            ((EList)peekObject.eGet(peekObject.eClass().getEStructuralFeature("contents"))).add(eObject);

                            processObject(eObject);
                            elementDeclarations.push(xsdElementDeclaration);
                            XSDParticle xsdParticle = xsdElementDeclaration.getTypeDefinition().getComplexType();
                            if (xsdParticle != null)
                            {
                              dfaStates.push(xsdParticle.getDFA().getInitialState());
                            }
                            else
                            {
                              dfaStates.push(null);
                              XSDSimpleTypeDefinition xsdSimpleTypeDefinition = xsdElementDeclaration.getTypeDefinition().getSimpleType();
                              if (xsdSimpleTypeDefinition != null)
                              {
                                EStructuralFeature valueFeature = eClass.getEStructuralFeature("value");
                                if (valueFeature != null)
                                {
                                  text = new StringBuffer();
                                  types.set(types.size() - 1, valueFeature);
                                }
                              }
                            }
                          }
                          else
                          {
                            error(new XMIException("Cannot resolve EClass ",  getLocation(), getLineNumber(), getColumnNumber()));
                          }
                        }
                      }
                    }
                  }
                }

                public void endElement(String uri, String localName, String name)
                {
                  EObject topObject = (EObject)objects.pop();
                  elements.pop();
                  Object type = types.pop();

                  if (text != null)
                  {
                    EAttribute eAttribute = (EAttribute)type;
                    EDataType eDataType = eAttribute.getEAttributeType();
                    Object value = eDataType.getEPackage().getEFactoryInstance().createFromString(eDataType, text.toString());
                    topObject.eSet(eAttribute, value);
                    text = null;
                  }

                  XSDParticle.DFA.State state = (XSDParticle.DFA.State)dfaStates.pop();
                  if (state != null && !state.isAccepting())
                  {
                    error(new XMIException("Need more content ",  getLocation(), getLineNumber(), getColumnNumber()));
                  }
                  elementDeclarations.pop();
                }
              };

            return new SAXWrapper(saxXMLHandler);
          }
        };
      xmlStart.load(this, inputStream, op);
    }
  }

  public static class GenericXMLLoadAction extends org.eclipse.ui.actions.ActionDelegate implements org.eclipse.ui.IActionDelegate
  {
    protected IFile file;

    public GenericXMLLoadAction()
    {
    }

    public void run(IAction action)
    {
       ResourceSet resourceSet = new ResourceSetImpl();
       resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().put("xml", new GenericXMLResourceFactoryImpl());

       Resource resource = resourceSet.getResource(URI.createPlatformResourceURI(file.getFullPath().toString()), true);
       resource.setURI(URI.createPlatformResourceURI(file.getFullPath().toString() + ".save.xml"));
       try
       {
         resource.save(Collections.EMPTY_MAP);
       }
       catch (IOException exception)
       {
         XSDEditorPlugin.INSTANCE.log(exception);
       }
    }

    public void selectionChanged(IAction action, ISelection selection)
    {
      if (selection instanceof IStructuredSelection)
      {
        Object object = ((IStructuredSelection)selection).getFirstElement();
        if (object instanceof IFile)
        {
          file = (IFile)object;
          action.setEnabled(true);
          return;
        }
      }
      file = null;
      action.setEnabled(false);
    }
  }
}
