/**
 * <copyright>
 *
 * Copyright (c) 2002 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * plugins/org.eclipse.xsd/src/org/eclipse/xsd/util/XSDResourceImpl.java, xsd, org.eclipse.dev, 20030620_1105VL
 * @version 1.21 6/20/03
 */
package org.eclipse.xsd.util;


import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.xml.serialize.OutputFormat;
import org.apache.xml.serialize.XMLSerializer;

import org.eclipse.core.runtime.IProgressMonitor;

import org.eclipse.emf.common.notify.Notification;

import org.eclipse.emf.common.util.URI;

import org.eclipse.emf.ecore.EObject;

import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;

import org.eclipse.emf.ecore.resource.impl.ResourceFactoryImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;

import org.eclipse.xsd.*;
import org.eclipse.xsd.XSDDiagnostic;
import org.eclipse.xsd.XSDFactory;
import org.eclipse.xsd.XSDPackage;
import org.eclipse.xsd.XSDPlugin;
import org.eclipse.xsd.XSDSchema;

import org.eclipse.xsd.impl.XSDSchemaImpl;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

import org.xml.sax.helpers.DefaultHandler;


/**
 * The <b>Resource</b> implementation for the model.
 * This specialized resource implementation supports it's own way of making keys and hrefs, and it's own serialization.
 * This class is not intended for subclassing outside of the model implementation;
 * it is intended to be used as is with the Resource framework.
 */
public class XSDResourceImpl extends ResourceImpl
{
  public static String XSD_TRACK_LOCATION = "XSD_TRACK_LOCATION";

  public static String XSD_PROGRESS_MONITOR = "XSD_PROGRESS_MONITOR";

  public static String XSD_ENCODING = "XSD_ENCODING";

  public static void serialize(OutputStream outputStream, Element element)
  {
    serialize(outputStream, element, null);
  }

  public static void serialize(OutputStream outputStream, Element element, String encoding)
  {
    try
    {
      doSerialize(outputStream, element, encoding);
    }
    catch (Exception exception)
    {
      exception.printStackTrace();
    }
  }

  protected static void doSerialize(OutputStream outputStream, Element element) throws IOException
  {
    doSerialize(outputStream, element, null);
  }

  protected static void doSerialize(OutputStream outputStream, Element element, String encoding) throws IOException
  {
    OutputFormat outputFormat = new OutputFormat(element.getOwnerDocument());
    outputFormat.setLineWidth(80);
    outputFormat.setIndenting(true);
    outputFormat.setIndent(4);
    outputFormat.setPreserveSpace(false);
    outputFormat.setEncoding(encoding);
    XMLSerializer serializer = new XMLSerializer(outputStream, outputFormat);
    serializer.serialize(element);
  }

  public static void serialize(OutputStream outputStream, Document document)
  {
    serialize(outputStream, document, null);
  }

  public static void serialize(OutputStream outputStream, Document document, String encoding)
  {
    try
    {
      doSerialize(outputStream, document, encoding);
    }
    catch (Exception exception)
    {
      exception.printStackTrace();
    }
  }

  protected static void doSerialize(OutputStream outputStream, Document document) throws IOException
  {
    doSerialize(outputStream, document, null);
  }

  protected static void doSerialize(OutputStream outputStream, Document document, String encoding) throws IOException
  {
    OutputFormat outputFormat = new OutputFormat(document);
    outputFormat.setLineWidth(80);
    outputFormat.setIndenting(true);
    outputFormat.setIndent(4);
    outputFormat.setPreserveSpace(false);
    outputFormat.setEncoding(encoding);
    XMLSerializer serializer = new XMLSerializer(outputStream, outputFormat);
    serializer.serialize(document);
  }

  /**
   * Builds a document using Xerces.
   * @param inputStream the contents to parse.
   * @param errorHandler the handled used by the parser.
   * @return a document.
   */
  protected static Document getDocument(InputStream inputStream, ErrorHandler errorHandler)  throws IOException
  {
    ClassLoader previousClassLoader = Thread.currentThread().getContextClassLoader();
    try
    {
      Thread.currentThread().setContextClassLoader(XSDResourceFactoryImpl.class.getClassLoader());

      DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
      documentBuilderFactory.setNamespaceAware(true);
      documentBuilderFactory.setValidating(false);

      DocumentBuilder documentBuilder = documentBuilderFactory.newDocumentBuilder();
  
      // Create a catalog with an entity mapping URL for the cached DTD for XML Schemas and set to into the build.
      //
      EntityResolver entityResolver = createEntityResolver();
      documentBuilder.setEntityResolver(entityResolver);

      documentBuilder.setErrorHandler(errorHandler);

      Document document = documentBuilder.parse(inputStream);
      return document;
    }
    catch (ParserConfigurationException exception)
    {
      throw new IOWrappedException(exception);
    }
    catch (SAXException exception)
    {
      throw new IOWrappedException(exception);
    }
    finally
    {
      Thread.currentThread().setContextClassLoader(previousClassLoader);
    }
  }

  /**
   * Creates an EntityResovler that can be used to help with parsing schema documents.
   * @return an EntityResolver.
   */
  protected static EntityResolver createEntityResolver()
  {
    final String baseURL = XSDPlugin.INSTANCE.getBaseURL().toString();

    // Create a catalog with an entity mapping URL for the cached DTD for XML Schemas and set to into the build.
    //
    EntityResolver entityResolver = 
      new DefaultHandler()
      {
        public InputSource resolveEntity(String publicId, String systemId) 
          throws SAXException
        {
          InputSource inputSource;
          if ("-//W3C//DTD XMLSCHEMA 200102//EN".equals(publicId) || "http://www.w3.org/2001/XMLSchema.dtd".equals(systemId))
          {
            inputSource = new InputSource(baseURL + "cache/www.w3.org/2001/XMLSchema.dtd");
            inputSource.setPublicId(publicId);
          }
          else
          {
            inputSource = super.resolveEntity(publicId, systemId);
          }

          return inputSource;
        }
      };

    return entityResolver;
  }

  /**
   * Assigns diagnostics to a schema.
   * @param xsdSchema the schema.
   * @param xsdDiagnostics the {@link XSDDiagnostic}s.
   */
  protected static void assignDiagnostics(XSDSchema xsdSchema, Collection xsdDiagnostics)
  {
    if (!xsdDiagnostics.isEmpty())
    {
      xsdSchema.getDiagnostics().addAll(xsdDiagnostics);
      for (Iterator i = xsdDiagnostics.iterator(); i.hasNext(); )
      {
        XSDDiagnostic xsdDiagnostic = (XSDDiagnostic)i.next();
        xsdDiagnostic.getComponents().add(xsdSchema);
        if (xsdSchema.getElement() != null)
        {
          xsdDiagnostic.setNode(xsdSchema.getElement());
        }
      }
    }
  }

  public XSDResourceImpl()
  {
    super();
  }

  public XSDResourceImpl(URI uri)
  {
    super(uri);
  }

  protected void doSave(OutputStream os, Map options) throws IOException
  {
    XSDSchema xsdSchema = getSchema();
    if (xsdSchema != null)
    {
      Document document = xsdSchema.getDocument();
      if (document == null)
      {
        xsdSchema.updateDocument();
        document = xsdSchema.getDocument();
      }

      if (xsdSchema.getElement() == null)
      {
        xsdSchema.updateElement();
      }

      doSerialize(os, document, options == null ? null : (String)options.get(XSD_ENCODING));
    }
  }

  /**
   * This gets the resource's schema.
   */
  public XSDSchema getSchema()
  {
    return 
      getContents().size() == 1 && getContents().get(0) instanceof XSDSchema ?
        (XSDSchema)getContents().get(0) :
        null;
  }

  /**
   * Loads a new {@link XSDResourceImpl} into the resource set.
   * @param resourceSet the resource set to hold the new resource.
   * @param uri the URI of the new resource.
   * @param inputStream the contents of the new resource.
   * @param options any options to influence loading behavior.
   * @return a new XSDResourceImpl.
   */
  protected void doLoad(InputStream inputStream, Map options) throws IOException
  {
    // This pattern avoids loading the IProgressMonitor class when there is no progress monitor.
    // This is important for stand-alone execution to work correctly.
    //
    IProgressMonitor progressMonitor = null;
    Object monitor = options == null ? null : options.get("XSD_PROGRESS_MONITOR");
    if (monitor != null)
    {
      progressMonitor = (IProgressMonitor)monitor;
      progressMonitor.setTaskName(XSDPlugin.INSTANCE.getString("_UI_ResourceLoad_progress"));
      progressMonitor.subTask(getURI().toString());
    }

    XSDParser xsdParser = new XSDParser();
    XSDSchema xsdSchema = null;
    try
    {
      Document document;
      if (options != null && Boolean.TRUE.equals(options.get("XSD_TRACK_LOCATION")))
      {
        xsdParser.parse(inputStream);
        document = xsdParser.getDocument();
      }
      else
      {
        document = getDocument(inputStream, xsdParser);
      }

      if (xsdParser.getEncoding() != null)
      {
        getDefaultSaveOptions().put(XSD_ENCODING, xsdParser.getEncoding());
      }

      if (document != null && document.getDocumentElement() != null)
      {
        ResourceSet globalResourceSet = XSDSchemaImpl.getGlobalResourceSet();
        Object oldMonitor = globalResourceSet.getLoadOptions().get("XSD_PROGRESS_MONITOR");
        try
        {
          XSDSchemaImpl.getGlobalResourceSet().getLoadOptions().put("XSD_PROGRESS_MONITOR", progressMonitor);
          xsdSchema =
            options != null && (options.containsKey("XSD_MAGIC_XML_SCHEMA") || options.containsKey("XSD_XML_SCHEMA")) ?
              XSDSchemaImpl.createMetaSchema(document.getDocumentElement()) :
              XSDSchemaImpl.createSchema(document.getDocumentElement());
        }
        finally
        {
          XSDSchemaImpl.getGlobalResourceSet().getLoadOptions().put("XSD_PROGRESS_MONITOR", oldMonitor);
        }
      }
      else
      {
        xsdSchema = XSDFactory.eINSTANCE.createXSDSchema();
        xsdSchema.getQNamePrefixToNamespaceMap().put(null, XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);
      }
    }
    catch (Exception exception)
    {
      XSDPlugin.INSTANCE.log(exception);

      xsdSchema = XSDFactory.eINSTANCE.createXSDSchema();
      xsdSchema.getQNamePrefixToNamespaceMap().put(null, XSDConstants.SCHEMA_FOR_SCHEMA_URI_2001);
    }

    if (xsdSchema != null)
    {
      getContents().add(xsdSchema);
      assignDiagnostics(xsdSchema, xsdParser.getDiagnostics());

      for (Iterator diagnostics = xsdParser.getDiagnostics().iterator(); diagnostics.hasNext(); )
      {
        XSDDiagnostic xsdDiagnostic = (XSDDiagnostic)diagnostics.next();
        switch (xsdDiagnostic.getSeverity().getValue())
        {
          case XSDDiagnosticSeverity.FATAL:
          case XSDDiagnosticSeverity.ERROR:
          {
            getErrors().add(xsdDiagnostic);
            break;
          }
          case XSDDiagnosticSeverity.WARNING:
          case XSDDiagnosticSeverity.INFORMATION:
          {
            getWarnings().add(xsdDiagnostic);
            break;
          }
        }
      }
    }

    if (progressMonitor != null)
    {
      progressMonitor.worked(1);
    }
  }

  public void attached(EObject eObject)
  {
    super.attached(eObject);

    if (eObject instanceof XSDSchema)
    {
      ((XSDSchema)eObject).setSchemaLocation(getURI().toString());
    }
  }

  public String getURIFragment(EObject eObject)
  {
    if (eObject instanceof XSDConcreteComponent)
    {
      XSDConcreteComponent xsdConcreteComponent = (XSDConcreteComponent)eObject;
      Element theElement = xsdConcreteComponent.getElement();
      if (theElement != null &&
            theElement.hasAttributeNS(null, XSDConstants.ID_ATTRIBUTE) &&
            theElement.getOwnerDocument().getElementById(theElement.getAttributeNS(null, XSDConstants.ID_ATTRIBUTE)) == theElement)
      {
        return theElement.getAttributeNS(null, XSDConstants.ID_ATTRIBUTE);
      }
    }

    return super.getURIFragment(eObject);
  }

  public EObject getEObject(String uriFragment)
  {
    // Do ID-based lookup.
    //
    if (!uriFragment.startsWith("/"))
    {
      Element theElement = getSchema().getElement();
      if (theElement != null)
      {
        // Navigate out through the elements.
        //
        Element resultElement =  theElement.getOwnerDocument().getElementById(uriFragment);
        List parents = new ArrayList();
        for (Node parent = resultElement; parent != null; parent = parent.getParentNode())
        {
          if (parent.getNodeType() == Node.ELEMENT_NODE)
          {
            parents.add(parent);
          }
        }
        return ((org.eclipse.xsd.impl.XSDSchemaImpl)getSchema()).getBestConcreteComponent(parents);
      } 
    }

    return super.getEObject(uriFragment); 
  }

  public Map getDefaultSaveOptions()
  {
    if (defaultSaveOptions == null)
    {
      defaultSaveOptions = new HashMap();
    }

    return defaultSaveOptions;
  }

  public Map getLoadSaveOptions()
  {
    if (defaultLoadOptions == null)
    {
      defaultLoadOptions = new HashMap();
    }

    return defaultLoadOptions;
  }
}
