/**
 * <copyright>
 *
 * Copyright (c) 2003-2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: SDOUtil.java,v 1.6 2004/06/09 18:22:24 emerks Exp $
 */
package org.eclipse.emf.ecore.sdo.util;


import java.io.IOException;
import java.io.InputStream;
import java.io.InvalidObjectException;
import java.io.ObjectStreamException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.Iterator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.sdo.EDataGraph;
import org.eclipse.emf.ecore.sdo.EDataObject;
import org.eclipse.emf.ecore.sdo.EProperty;
import org.eclipse.emf.ecore.sdo.EType;
import org.eclipse.emf.ecore.sdo.SDOFactory;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.util.FeatureMap;
import org.eclipse.emf.ecore.util.FeatureMapUtil;
import org.eclipse.emf.ecore.xmi.impl.EMOFResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.impl.EcoreResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLResourceFactoryImpl;

import commonj.sdo.DataGraph;
import commonj.sdo.DataObject;
import commonj.sdo.Property;
import commonj.sdo.Sequence;
import commonj.sdo.Type;


public final class SDOUtil
{
  public static List getInstanceProperties(EObject eObject)
  {
    List result = new UniqueEList();
    for (Iterator i = eObject.eClass().getEAllStructuralFeatures().iterator(); i.hasNext(); )
    {
      EStructuralFeature eStructuralFeature = (EStructuralFeature)i.next();
      if (FeatureMapUtil.isFeatureMap(eStructuralFeature))
      {
        for (Iterator j = ((List)eObject.eGet(eStructuralFeature)).iterator(); j.hasNext(); )
        {
          FeatureMap.Entry entry = (FeatureMap.Entry)j.next();
          EStructuralFeature entryFeature = entry.getEStructuralFeature();
          result.add(entryFeature);
        }
      }
      else
      {
        result.add(eStructuralFeature);
      }
    }
    return result;
  }

  protected static EStructuralFeature getEStructuralFeature(EObject eObject, String propertyName)
  {
    EProperty property = getType(eObject).getEProperty(propertyName);
    if (property == null)
    {
      List instanceProperties = getInstanceProperties(eObject);
      for (Iterator i = instanceProperties.iterator(); i.hasNext(); )
      {
        EStructuralFeature eStructuralFeature = (EStructuralFeature)i.next();
        if (eStructuralFeature.getName().equals(propertyName))
        {
          return eStructuralFeature;
        }
      }

      for (Iterator i = instanceProperties.iterator(); i.hasNext(); )
      {
        EStructuralFeature eStructuralFeature = (EStructuralFeature)i.next();
        if (ExtendedMetaData.INSTANCE.getName(eStructuralFeature).equals(propertyName))
        {
          return eStructuralFeature;
        }
      }

      throw new IllegalArgumentException("Class '" + eObject.eClass().getName() + "' does not have a feature named '" + propertyName + "'");
    }
    return property.getEStructuralFeature();
  }

  protected static EReference getContainmentEReference(EObject eObject, String propertyName)
  {
    EProperty property = getType(eObject).getEProperty(propertyName);
    if (property == null)
    {
      throw new IllegalArgumentException("Class '" + eObject.eClass().getName() + "' does not have a feature named '" + propertyName + "'");
    }

    return getContainmentEReference(property);
  }

  protected static EReference getContainmentEReference(EProperty property)
  {
    EStructuralFeature feature = property.getEStructuralFeature();
    if (feature instanceof EReference)
    {
      EReference reference = (EReference)feature;
      if (reference.isContainment())
      {
        return reference;
      }
    }
    throw new IllegalArgumentException("The feature '" + property.getName() + "' of '" + property.getContainingType().getName()
      + "' isn't a containment");
  }

  public static Object get(EObject eObject, String path)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      return get(eObject, property.getEStructuralFeature());
    }
    else
    {
      return Accessor.create(eObject, path).getAndRecyle();
    }
  }

  public static void set(EObject eObject, String path, Object value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      set(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor.create(eObject, path).setAndRecyle(value);
    }
  }

  public static boolean isSet(EObject eObject, String path)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      return isSet(eObject, property.getEStructuralFeature());
    }
    else
    {
      return Accessor.create(eObject, path).isSetAndRecyle();
    }
  }

  public static void unset(EObject eObject, String path)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      unset(eObject, property.getEStructuralFeature());
    }
    else
    {
      Accessor.create(eObject, path).unsetAndRecyle();
    }
  }

  public static DataObject getDataObject(EObject eObject, String path)
  {
    return (DataObject)get(eObject, path);
  }

  public static List getList(EObject eObject, String path)
  {
    return (List)get(eObject, path);
  }

  public static Sequence getSequence(EObject eObject, String path)
  {
    return (Sequence)get(eObject, path);
  }

  public static void setDataObject(EObject eObject, String path, DataObject value)
  {
    set(eObject, path, value);
  }

  public static void setList(EObject eObject, String path, List value)
  {
    set(eObject, path, value);
  }

  protected static EStructuralFeature getFeature(EObject eObject, int propertyIndex)
  {
    return (EStructuralFeature)eObject.eClass().getEAllStructuralFeatures().get(propertyIndex);
  }

  public static Object get(EObject eObject, int propertyIndex)
  {
    return get(eObject, getFeature(eObject, propertyIndex));
  }

  public static void set(EObject eObject, int propertyIndex, Object value)
  {
    set(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static boolean isSet(EObject eObject, int propertyIndex)
  {
    return isSet(eObject, getFeature(eObject, propertyIndex));
  }

  public static void unset(EObject eObject, int propertyIndex)
  {
    unset(eObject, getFeature(eObject, propertyIndex));
  }

  public static DataObject getDataObject(EObject eObject, int propertyIndex)
  {
    return (DataObject)get(eObject, getFeature(eObject, propertyIndex));
  }

  public static List getList(EObject eObject, int propertyIndex)
  {
    return (List)get(eObject, getFeature(eObject, propertyIndex));
  }

  public static Sequence getSequence(EObject eObject, int propertyIndex)
  {
    return (Sequence)get(eObject, getFeature(eObject, propertyIndex));
  }

  public static void setDataObject(EObject eObject, int propertyIndex, DataObject value)
  {
    set(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setList(EObject eObject, int propertyIndex, List value)
  {
    set(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static Object get(EObject eObject, Property property)
  {
    return get(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static void set(EObject eObject, Property property, Object value)
  {
    set(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static boolean isSet(EObject eObject, Property property)
  {
    return isSet(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static void unset(EObject eObject, Property property)
  {
    unset(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static DataObject getDataObject(EObject eObject, Property property)
  {
    return (DataObject)get(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static List getList(EObject eObject, Property property)
  {
    return (List)get(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static Sequence getSequence(EObject eObject, Property property)
  {
    return (Sequence)get(eObject, ((EProperty)property).getEStructuralFeature());
  }

  public static void setDataObject(EObject eObject, Property property, DataObject value)
  {
    set(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setList(EObject eObject, Property property, List value)
  {
    set(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static DataObject createDataObject(EObject eObject, String propertyName)
  {
    EReference eReference = getContainmentEReference(eObject, propertyName);
    EDataObject result = create(eReference.getEReferenceType());
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static DataObject createDataObject(EObject eObject, int propertyIndex)
  {
    EReference eReference = getContainmentEReference((EProperty)getType(eObject).getProperties().get(propertyIndex));
    EDataObject result = create(eReference.getEReferenceType());
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static DataObject createDataObject(EObject eObject, Property property)
  {
    EReference eReference = getContainmentEReference((EProperty)property);
    EDataObject result = create(eReference.getEReferenceType());
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static DataObject createDataObject(EObject eObject, String propertyName, String namespaceURI, String typeName)
  {
    return createDataObject(eObject, propertyName, getDataGraph(eObject).getType(namespaceURI, typeName));
  }

  public static DataObject createDataObject(EObject eObject, String propertyName, Type type)
  {
    EReference eReference = getContainmentEReference(eObject, propertyName);
    EDataObject result = create(type);
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static DataObject createDataObject(EObject eObject, int propertyIndex, String namespaceURI, String typeName)
  {
    return createDataObject(eObject, propertyIndex, getDataGraph(eObject).getType(namespaceURI, typeName));
  }

  public static DataObject createDataObject(EObject eObject, int propertyIndex, Type type)
  {
    EReference eReference = getContainmentEReference((EProperty)getType(eObject).getProperties().get(propertyIndex));
    EDataObject result = create(type);
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static DataObject createDataObject(EObject eObject, Property property, Type type)
  {
    EReference eReference = getContainmentEReference((EProperty)property);
    EDataObject result = create(type);
    if (FeatureMapUtil.isMany(eObject, eReference))
    {
      ((List)eObject.eGet(eReference)).add(result);
    }
    else
    {
      eObject.eSet(eReference, result);
    }
    return result;
  }

  public static void delete(EObject eObject)
  {
    EcoreUtil.remove(eObject);
    List eAllStructuralFeatures = eObject.eClass().getEAllStructuralFeatures();
    for (int i = 0, size = eAllStructuralFeatures.size(); i < size; ++i)
    {
      eObject.eUnset((EStructuralFeature)eAllStructuralFeatures.get(i));
    }
  }

  public static DataObject getContainer(EObject eObject)
  {
    return (DataObject)eObject.eContainer();
  }

  public static EProperty getContainmentProperty(EObject eObject)
  {
    return adaptProperty(eObject.eContainmentFeature());
  }

  public static EDataGraph getEDataGraph(EObject eObject)
  {
    Resource resource = eObject.eResource();
    if (resource != null)
    {
      ResourceSet resourceSet = resource.getResourceSet();
      if (resourceSet != null)
      {
        return (EDataGraph)EcoreUtil.getAdapter(resourceSet.eAdapters(), EDataGraph.class);
      }
    }

    return null;
  }

  public static DataGraph getDataGraph(EObject eObject)
  {
    return getEDataGraph(eObject);
  }

  public static EType getType(EObject eObject)
  {
    return adaptType(eObject.eClass());
  }

  public static EType adaptType(EClassifier eClassifier)
  {
    List adapters = eClassifier.eAdapters();
    Object result = EcoreUtil.getAdapter(adapters, EType.class);
    if (result != null)
    {
      return (EType)result;
    }
    else
    {
      EType eType = SDOFactory.eINSTANCE.createEType();
      adapters.add(0, eType);
      return eType;
    }
  }

  public static EProperty adaptProperty(EStructuralFeature eStructuralFeature)
  {
    List adapters = eStructuralFeature.eAdapters();
    Object result = EcoreUtil.getAdapter(adapters, EProperty.class);
    if (result != null)
    {
      return (EProperty)result;
    }
    else
    {
      EProperty eProperty = SDOFactory.eINSTANCE.createEProperty();
      adapters.add(eProperty);
      return eProperty;
    }
  }

  public static EDataObject create(Type type)
  {
    return create((EType)type);
  }

  public static EDataObject create(EType eType)
  {
    return create((EClass)eType.getEClassifier());
  }

  public static EDataObject create(EClass eClass)
  {
    return (EDataObject)EcoreUtil.create(eClass);
  }

  public static Object writeReplace(EObject eObject) throws ObjectStreamException
  {
    EDataGraph eDataGraph = getEDataGraph(eObject);
    if (eDataGraph != null)
    {
      return ((EDataGraph.Internal)eDataGraph).getWriteReplacement(eObject);
    }
    else
    {
      throw new InvalidObjectException("The object must be in a datagraph to be serialized " + eObject);
    }
  }

  public static Object get(EObject eObject, EProperty property)
  {
    return get(eObject, property.getEStructuralFeature());
  }

  public static void set(EObject eObject, EProperty property, Object value)
  {
    set(eObject, property.getEStructuralFeature(), value);
  }

  public static boolean isSet(EObject eObject, EProperty property)
  {
    return isSet(eObject, property.getEStructuralFeature());
  }

  public static void unset(EObject eObject, EProperty property)
  {
    unset(eObject, property.getEStructuralFeature());
  }

  protected static Object get(EObject eObject, EStructuralFeature feature)
  {
    Object result = eObject.eGet(feature);
    if (FeatureMapUtil.isFeatureMap(feature))
    {
      result = new BasicESequence((FeatureMap.Internal)result);
    }
    return result;
  }

  protected static void set(EObject eObject, EStructuralFeature feature, Object value)
  {
    eObject.eSet(feature, value);
  }

  protected static boolean isSet(EObject eObject, EStructuralFeature feature)
  {
    return eObject.eIsSet(feature);
  }

  protected static void unset(EObject eObject, EStructuralFeature feature)
  {
    eObject.eUnset(feature);
  }

  /**
   * Process the default EMF path and minimal XPath syntax.
   * This design is still under review and construction.
   *
   * Syntax:
   * 
   *<pre>
   * path = /? (step '/')* step
   * step = feature
   *      | feature '.' index_from_0 
   *      | feature '[' index_from_1 ']'
   *      | reference '[' attribute '=' value ']'
   *      | ..
   *      | '@' step
   *</pre>
   * 
   * feature = the name of an attribute or reference
   * attribute = the name of an attribute
   * reference = the name of a reference
   * index = positive integer
   * value = the string value of an attribute
   * leading / begins at the root
   * .. is containing object
   * 
   * features must be multi-valued to use '.' and '[' operations.
   * Only the last step may have an attribute as the feature.
   */
  protected static final class Accessor
  {
    /**
     * Creates an accessor for the path of the object.
     */
    public static Accessor create(EObject eObject, String path)
    {
      Accessor result = pool.get();
      result.init(eObject, path);
      return result;
    }

    /**
     * Only the get and recyle methods should be call; they are the only synchronized methods.
     */
    protected static class Pool extends BasicEList
    {
      protected Accessor[] accessors;

      public Pool()
      {
        super(10);
      }

      protected Object[] newData(int capacity)
      {
        return accessors = new Accessor [capacity];
      }

      /**
       *  Returns a recyled instance or a new instance.
       */
      public synchronized Accessor get()
      {
        if (size > 0)
        {
          return accessors[--size];
        }
        else
        {
          return new Accessor();
        }
      }

      /** Safely gives the accessor back for recycled use.
       */
      public synchronized void recycle(Accessor accessor)
      {
        int minimumCapacity = size + 1;
        if (minimumCapacity > data.length)
        {
          grow(minimumCapacity);
        }
        accessors[size++] = accessor;
      }
    }

    /**
     * A static thread safe pool of Accessors.
     */
    static final Pool pool = new Pool();

    protected static final int NO_INDEX = -1;

    protected EObject eObject;

    protected EStructuralFeature feature;

    protected int index;

    protected Accessor()
    {
    }

    protected Accessor(EObject eObject, String path)
    {
      init(eObject, path);
    }

    protected void init(EObject eObject, String path)
    {
      this.eObject = eObject;

      // This should only be called with a path right now.
      //
      //feature = getType(eObject).getEProperty(path).getEStructuralFeature(); 
      //if (feature == null)
      {
        process(path);
      }
      //else
      {
        //index = NO_INDEX;
      }
    }

    public Object get()
    {
      if (feature == null)
      {
        return eObject;
      }
      else
      {
        Object value = eObject.eGet(feature, true);
        if (index >= 0)
        {
          value = ((List)value).get(index);
          if (value instanceof FeatureMap.Entry)
          {
            value = ((FeatureMap.Entry)value).getValue();
          }
        }
        else if (FeatureMapUtil.isFeatureMap(feature))
        {
          value = new BasicESequence((FeatureMap.Internal)value);
        }
        return value;
      }
    }

    public Object getAndRecyle()
    {
      pool.recycle(this);
      return get();
    }

    public void set(Object newValue)
    {
      if (index >= 0)
      {
        List list = (List)eObject.eGet(feature, true);
        list.set(index, newValue);
      }
      else
      {
        // EATM newValue = string2Enum(feature, newValue);
        eObject.eSet(feature, newValue);
      }
    }

    public void setAndRecyle(Object newValue)
    {
      pool.recycle(this);
      set(newValue);
    }

    public boolean isSet()
    {
      return eObject.eIsSet(feature);
    }

    public boolean isSetAndRecyle()
    {
      pool.recycle(this);
      return isSet();
    }

    public void unset()
    {
      eObject.eUnset(feature);
    }

    public void unsetAndRecyle()
    {
      pool.recycle(this);
      unset();
    }

    public void recycle()
    {
      pool.recycle(this);
    }

    public EObject getEObject()
    {
      return eObject;
    }

    protected void setEObject(EObject eObject)
    {
      this.eObject = eObject;
      feature = null;
      index = NO_INDEX;
    }

    public EStructuralFeature getEStructuralFeature()
    {
      return feature;
    }

    protected void setFeatureName(String name)
    {
      if (name != null)
      {
        feature = SDOUtil.getEStructuralFeature(eObject, name);
      }
      else
      {
        feature = null;
      }
      index = NO_INDEX;
    }

    protected int getIndex()
    {
      return index;
    }

    protected void setIndex(int index)
    {
      this.index = index;
      if (index > 0 && !FeatureMapUtil.isMany(eObject, feature))
      {
        throw new IndexOutOfBoundsException("Index applies only to mutli-valued features.");
      }
    }

    protected void process(String pathString)
    {
      TokenList tokens = new TokenList(pathString.toCharArray());
      String token;
      int size = tokens.size();
      int x = 0;

      if ("/".equals(tokens.peek(0)))
      {
        setEObject(EcoreUtil.getRootContainer(eObject));
        x++;
      }

      for (; x < size; x++)
      {
        token = tokens.peek(x);
        char c = token.charAt(0);
        if ('/' == c)
        {
          setEObject((EObject)get());
        }
        else if ("..".equals(token))
        {
          EObject container = eObject.eContainer();
          if (container == null)
          {
            throw new IllegalArgumentException("No containing object for " + eObject);
          }
          setEObject(container);
        }
        else if ('.' == c)
        {
          x++; // skip .
          token = tokens.peek(x);
          int index = Integer.parseInt(token);
          setIndex(index);
        }
        else if ('[' == c)
        {
          x++; // skip [
          token = tokens.peek(x); // name or index
          char following = tokens.peek(x + 1).charAt(0);
          if ('=' != following)
          {
            int index = Integer.parseInt(token) - 1;
            setIndex(index);
            x++; // skip ]
          }
          else
          {
            x++; // done name
            x++; // skip =
            String attributeValue = tokens.peek(x); // value
            if ("\"".equals(attributeValue))
            {
              x++; // skip "
              attributeValue = tokens.peek(++x);
            }
            x++; // skip ]
            int index = matchingIndex((List)get(), token, attributeValue);
            if (index < 0)
            {
              throw new IndexOutOfBoundsException("No index with " + token + "=" + attributeValue);
            }
            setIndex(index);
          }
        }
        else if ('@' == c)
        {
          x++; // skip @
        }
        else
        {
          setFeatureName(token);
        }
      }
    }

    protected static int matchingIndex(List eObjects, String attributeName, String attributeValue)
    {
      for (int i = 0, size = eObjects.size(); i < size; i++)
      {
        EObject eObject = (EObject)eObjects.get(i);
        EStructuralFeature feature = getType(eObject).getEProperty(attributeName).getEStructuralFeature();
        if (feature != null)
        {
          Object test = eObject.eGet(feature, true);
          if (test != null)
          {
            String testString = EcoreUtil.convertToString((EDataType)feature.getEType(), test);
            if (attributeValue.equals(testString))
            {
              return i;
            }
          }
        }
      }
      return -1;
    }

    protected static class TokenList extends BasicEList
    {
      public TokenList(char[] path)
      {
        super(4);

        int pathLength = path.length;
        StringBuffer token = new StringBuffer();
        char cPrev;
        char c = 0;
        char cNext;
        char stringConstant = 0;
        for (int pos = 0; pos < pathLength; pos++)
        {
          cPrev = c;
          c = path[pos];
          cNext = pos < pathLength - 1 ? path[pos + 1] : 0;

          if (stringConstant != 0)
          {
            if (c == stringConstant)
            {
              endToken(token, true);
              stringConstant = 0;
            }
            else
            {
              token.append(c);
            }
          }
          else
          {
            switch (c)
            {
              case ' ':
              case 0xA:
              case 0xD:
              case 0x9:
                if (cPrev != ' ')
                {
                  endToken(token, false);
                }
                c = ' ';
                break;

              case '"':
              case '\'':
                endToken(token, false);
                stringConstant = c;
                break;

              // double or single tokens
              case '/':
              case ':':
              case '.':
                if (cPrev != c)
                {
                  endToken(token, false);
                }
                token.append(c);
                if (cNext != c)
                {
                  endToken(token, false);
                }
                break;

              // single tokens
              case '*':
              case '@':
              case '[':
              case ']':
              case '(':
              case ')':
              case '|':
                endToken(token, false);
                add(String.valueOf(c));
                break;

              // TODO: < > <= >= + - !=
              case '!':
                endToken(token, false);
                token.append(c);
                break;

              case '=':
                endToken(token, false);
                add(String.valueOf(c));
                break;

              default:
                token.append(c);
            }
          }
        }
        endToken(token, false);
      }

      public String peek()
      {
        return size > 0 ? (String)data[0] : " ";
      }

      public String peek(int index)
      {
        return index < size ? (String)data[index] : " ";
      }

      public TokenList pop()
      {
        remove(0);
        return this;
      }

      public TokenList pop(int count)
      {
        while (count-- > 0)
        {
          remove(count);
        }
        return this;
      }

      protected void endToken(StringBuffer token, boolean includeEmpty)
      {
        if (includeEmpty || token.length() > 0)
        {
          add(token.toString());
        }
        token.setLength(0);
      }

      protected boolean canContainNull()
      {
        return false;
      }

      protected Object[] newData(int capacity)
      {
        return new String [capacity];
      }
    }

    public String toString()
    {
      StringBuffer result = new StringBuffer("Accessor (object:");
      result.append(eObject == null ? "null" : eObject.toString());
      result.append(", feature:");
      result.append(feature == null ? "null" : feature.getName());
      result.append(", index:");
      result.append(index);
      result.append(")");
      return result.toString();
    }
  }

  public static ResourceSet createResourceSet()
  {
    ResourceSet result = new ResourceSetImpl();
    configureResourceSet(result);
    return result;
  }

  protected static Map registrations;

  protected static Map getRegistrations()
  {
    if (registrations == null)
    {
      Map result = new HashMap();
      ResourceSet resourceSet = new ResourceSetImpl();

      if (!(resourceSet.getResourceFactoryRegistry().getFactory(URI.createURI("*.datagraph")) instanceof DataGraphResourceFactoryImpl))
      {
        result.put("datagraph", new DataGraphResourceFactoryImpl());
      }
      if (!(resourceSet.getResourceFactoryRegistry().getFactory(URI.createURI("*.ecore")) instanceof EcoreResourceFactoryImpl))
      {
        result.put("ecore", new EcoreResourceFactoryImpl());
      }

      if (!(resourceSet.getResourceFactoryRegistry().getFactory(URI.createURI("*.emof")) instanceof EMOFResourceFactoryImpl))
      {
        result.put("emof", new EMOFResourceFactoryImpl());
      }

      if (resourceSet.getResourceFactoryRegistry().getFactory(URI.createURI("*.xmi")) == null)
      {
        result.put("xmi", new XMIResourceFactoryImpl());
      }

      if (resourceSet.getResourceFactoryRegistry().getFactory(URI.createURI("*.xml")) == null)
      {
        result.put("*", new XMLResourceFactoryImpl());
      }

      registrations = result;
    }

    return registrations;
  }

  public static void configureResourceSet(ResourceSet resourceSet)
  {
    resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap().putAll(getRegistrations());
  }

  public static EDataGraph loadDataGraph(InputStream inputStream, Map options) throws IOException
  {
    ResourceSet resourceSet = createResourceSet();
    Resource resource = resourceSet.createResource(URI.createURI("all.datagraph"));
    resource.load(inputStream, options);
    return (EDataGraph)resource.getContents().get(0);
  }

  public static BigDecimal getBigDecimal(EObject eObject, int propertyIndex)
  {
    return getBigDecimal(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static BigDecimal getBigDecimal(EObject eObject, Property property)
  {
    return getBigDecimal(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static BigDecimal getBigDecimal(EObject eObject, String path)
  {
    return getBigDecimal(get(eObject, path));
  }

  protected static BigDecimal getBigDecimal(Object value)
  {
    if (value instanceof BigDecimal)
    {
      return (BigDecimal)value;
    }

    if (value instanceof BigInteger)
    {
      return new BigDecimal((BigInteger)value);
    }

    if (value instanceof Number)
    {
      return new BigDecimal(((Number)value).doubleValue());
    }

    if (value instanceof String)
    {
      return new BigDecimal((String)value);
    }

    if (value == null)
    {
      return null;
    }

    throw new IllegalArgumentException();
  }

  public static void setBigDecimal(EObject eObject, int propertyIndex, BigDecimal value)
  {
    setBigDecimal(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setBigDecimal(EObject eObject, Property property, BigDecimal value)
  {
    setBigDecimal(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setBigDecimal(EObject eObject, String path, BigDecimal value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setBigDecimal(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setBigDecimal(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setBigDecimal(EObject eObject, EStructuralFeature eStructuralFeature, BigDecimal value)
  {
    EClassifier eType = eStructuralFeature.getEType();
    String name = eType.getInstanceClassName();
    
    if (value == null)
    {
      set(eObject, eStructuralFeature, eType.getDefaultValue());
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, value);
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, value.toBigInteger());
    }
    else if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte(value.byteValue()));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value.doubleValue()));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value.floatValue()));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer(value.intValue()));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value.longValue()));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short(value.shortValue()));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, value);
    }
  }

  public static BigInteger getBigInteger(EObject eObject, int propertyIndex)
  {
    return getBigInteger(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static BigInteger getBigInteger(EObject eObject, Property property)
  {
    return getBigInteger(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static BigInteger getBigInteger(EObject eObject, String path)
  {
    return getBigInteger(get(eObject, path));
  }

  protected static BigInteger getBigInteger(Object value)
  {
    if (value instanceof BigInteger)
    {
      return (BigInteger)value;
    }

    if (value instanceof BigDecimal)
    {
      return ((BigDecimal)value).toBigInteger();
    }

    if (value instanceof Number)
    {
      return BigInteger.valueOf(((Number)value).longValue());
    }

    if (value instanceof String)
    {
      return new BigInteger((String)value);
    }

    if (value instanceof byte[])
    {
      return new BigInteger((byte[])value);
    }

    if (value == null)
    {
      return null;
    }

    throw new IllegalArgumentException();
  }

  public static void setBigInteger(EObject eObject, int propertyIndex, BigInteger value)
  {
    setBigInteger(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setBigInteger(EObject eObject, Property property, BigInteger value)
  {
    setBigInteger(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setBigInteger(EObject eObject, String path, BigInteger value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setBigInteger(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setBigInteger(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }  
  
  protected static void setBigInteger(EObject eObject, EStructuralFeature eStructuralFeature, BigInteger value)
  {
    EClassifier eType = eStructuralFeature.getEType();
    String name = eType.getInstanceClassName();

    if (value == null)
    {
      set(eObject, eStructuralFeature, eType.getDefaultValue());
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, value);
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, new BigDecimal(value));
    }
    else if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte(value.byteValue()));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value.doubleValue()));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value.floatValue()));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer(value.intValue()));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value.longValue()));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short(value.shortValue()));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else if (name == "byte[]")
    {
      set(eObject, eStructuralFeature, value.toByteArray());
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, value);
    }
  }

  public static boolean getBoolean(EObject eObject, int propertyIndex)
  {
    return getBoolean(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static boolean getBoolean(EObject eObject, Property property)
  {
    return getBoolean(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static boolean getBoolean(EObject eObject, String path)
  {
    return getBoolean(get(eObject, path));
  }

  protected static boolean getBoolean(Object value)
  {
    if (value instanceof Boolean)
    {
      return ((Boolean)value).booleanValue();
    }

    if (value instanceof String)
    {
      return Boolean.valueOf((String)value).booleanValue();
    }

    if (value == null)
    {
      return false;
    }

    throw new IllegalArgumentException();
  }

  public static void setBoolean(EObject eObject, int propertyIndex, boolean value)
  {
    setBoolean(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setBoolean(EObject eObject, Property property, boolean value)
  {
    setBoolean(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setBoolean(EObject eObject, String path, boolean value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setBoolean(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setBoolean(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setBoolean(EObject eObject, EStructuralFeature eStructuralFeature, boolean value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Boolean" || name == "boolean")
    {
      set(eObject, eStructuralFeature, value ? Boolean.TRUE : Boolean.FALSE);
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Boolean(value));
    }
  }

  public static byte getByte(EObject eObject, int propertyIndex)
  {
    return getByte(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static byte getByte(EObject eObject, Property property)
  {
    return getByte(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static byte getByte(EObject eObject, String path)
  {
    return getByte(get(eObject, path));
  }

  protected static byte getByte(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).byteValue();
    }

    if (value instanceof String)
    {
      return Byte.parseByte((String)value);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setByte(EObject eObject, int propertyIndex, byte value)
  {
    setByte(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setByte(EObject eObject, Property property, byte value)
  {
    setByte(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setByte(EObject eObject, String path, byte value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setByte(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setByte(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setByte(EObject eObject, EStructuralFeature eStructuralFeature, byte value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte(value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer(value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short(value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Byte(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Byte(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Byte(value));
    }
  }

  public static byte[] getBytes(EObject eObject, int propertyIndex)
  {
    return getBytes(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static byte[] getBytes(EObject eObject, Property property)
  {
    return getBytes(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static byte[] getBytes(EObject eObject, String path)
  {
    return getBytes(get(eObject, path));
  }

  protected static byte[] getBytes(Object value)
  {
    if (value instanceof byte[])
    {
      return (byte[])value;
    }

    if (value instanceof BigInteger)
    {
      return ((BigInteger)value).toByteArray();
    }

    if (value == null)
    {
      return null;
    }

    throw new IllegalArgumentException();
  }

  public static void setBytes(EObject eObject, int propertyIndex, byte[] value)
  {
    setBytes(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setBytes(EObject eObject, Property property, byte[] value)
  {
    setBytes(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setBytes(EObject eObject, String path, byte[] value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setBytes(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setBytes(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setBytes(EObject eObject, EStructuralFeature eStructuralFeature, byte[] value)
  {
    EClassifier eType = eStructuralFeature.getEType();
    String name = eType.getInstanceClassName();

    if (value == null)
    {
      set(eObject, eStructuralFeature, eType.getDefaultValue());
    }
    else if (name == "byte[]")
    {
      set(eObject, eStructuralFeature, value);
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, new BigInteger(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, value);
    }
  }

  public static char getChar(EObject eObject, int propertyIndex)
  {
    return getChar(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static char getChar(EObject eObject, Property property)
  {
    return getChar(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static char getChar(EObject eObject, String path)
  {
    return getChar(get(eObject, path));
  }

  protected static char getChar(Object value)
  {
    if (value instanceof Character)
    {
      return ((Character)value).charValue();
    }

    if (value instanceof String)
    {
      return ((String)value).charAt(0);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setChar(EObject eObject, int propertyIndex, char value)
  {
    setChar(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setChar(EObject eObject, Property property, char value)
  {
    setChar(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setChar(EObject eObject, String path, char value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setChar(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setChar(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setChar(EObject eObject, EStructuralFeature eStructuralFeature, char value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Character" || name == "char")
    {
      set(eObject, eStructuralFeature, new Character(value));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Character(value));
    }
  }

  public static Date getDate(EObject eObject, int propertyIndex)
  {
    return getDate(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static Date getDate(EObject eObject, Property property)
  {
    return getDate(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static Date getDate(EObject eObject, String path)
  {
    return getDate(get(eObject, path));
  }

  protected static Date getDate(Object value)
  {
    if (value instanceof Date)
    {
      return (Date)value;
    }

    if (value instanceof Long)
    {
      return new Date(((Long)value).longValue());
    }

    if (value == null)
    {
      return null;
    }

    throw new IllegalArgumentException();
  }

  public static void setDate(EObject eObject, int propertyIndex, Date value)
  {
    setDate(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setDate(EObject eObject, Property property, Date value)
  {
    setDate(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setDate(EObject eObject, String path, Date value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setDate(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setDate(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setDate(EObject eObject, EStructuralFeature eStructuralFeature, Date value)
  {
    EClassifier eType = eStructuralFeature.getEType();
    String name = eType.getInstanceClassName();

    if (value == null)
    {
      set(eObject, eStructuralFeature, eType.getDefaultValue());
    }
    else if (name == "java.util.Date")
    {
      set(eObject, eStructuralFeature, value);
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value.getTime()));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, value);
    }
  }

  public static double getDouble(EObject eObject, int propertyIndex)
  {
    return getDouble(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static double getDouble(EObject eObject, Property property)
  {
    return getDouble(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static double getDouble(EObject eObject, String path)
  {
    return getDouble(get(eObject, path));
  }

  protected static double getDouble(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).doubleValue();
    }

    if (value instanceof String)
    {
      return Double.parseDouble((String)value);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setDouble(EObject eObject, int propertyIndex, double value)
  {
    setDouble(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setDouble(EObject eObject, Property property, double value)
  {
    setDouble(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setDouble(EObject eObject, String path, double value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setDouble(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setDouble(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setDouble(EObject eObject, EStructuralFeature eStructuralFeature, double value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte((byte)value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer((int)value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long((long)value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short((short)value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Double(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Double(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      //Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Double(value));
    }
  }

  public static float getFloat(EObject eObject, int propertyIndex)
  {
    return getFloat(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static float getFloat(EObject eObject, Property property)
  {
    return getFloat(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static float getFloat(EObject eObject, String path)
  {
    return getFloat(get(eObject, path));
  }

  protected static float getFloat(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).floatValue();
    }

    if (value instanceof String)
    {
      return Float.parseFloat((String)value);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setFloat(EObject eObject, int propertyIndex, float value)
  {
    setFloat(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setFloat(EObject eObject, Property property, float value)
  {
    setFloat(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setFloat(EObject eObject, String path, float value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setFloat(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setFloat(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setFloat(EObject eObject, EStructuralFeature eStructuralFeature, float value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte((byte)value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer((int)value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long((long)value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short((short)value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Float(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Float(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      // Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Float(value));
    }
  }

  public static int getInt(EObject eObject, int propertyIndex)
  {
    return getInt(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static int getInt(EObject eObject, Property property)
  {
    return getInt(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static int getInt(EObject eObject, String path)
  {
    return getInt(get(eObject, path));
  }

  protected static int getInt(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).intValue();
    }

    if (value instanceof String)
    {
      return Integer.parseInt((String)value);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setInt(EObject eObject, int propertyIndex, int value)
  {
    setInt(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setInt(EObject eObject, Property property, int value)
  {
    setInt(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setInt(EObject eObject, String path, int value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setInt(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setInt(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setInt(EObject eObject, EStructuralFeature eStructuralFeature, int value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte((byte)value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer(value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short((short)value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Integer(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Integer(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      // Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Integer(value));
    }
  }

  public static long getLong(EObject eObject, int propertyIndex)
  {
    return getLong(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static long getLong(EObject eObject, Property property)
  {
    return getLong(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static long getLong(EObject eObject, String path)
  {
    return getLong(get(eObject, path));
  }

  protected static long getLong(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).longValue();
    }

    if (value instanceof String)
    {
      return Long.parseLong((String)value);
    }

    if (value instanceof Date)
    {
      return ((Date)value).getTime();
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setLong(EObject eObject, int propertyIndex, long value)
  {
    setLong(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setLong(EObject eObject, Property property, long value)
  {
    setLong(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setLong(EObject eObject, String path, long value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setLong(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setLong(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setLong(EObject eObject, EStructuralFeature eStructuralFeature, long value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte((byte)value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer((int)value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short((short)value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Long(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Long(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else if (name == "java.util.Date")
    {
      set(eObject, eStructuralFeature, new Date(value));
    }
    else
    {
      // Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Long(value));
    }
  }

  public static short getShort(EObject eObject, int propertyIndex)
  {
    return getShort(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static short getShort(EObject eObject, Property property)
  {
    return getShort(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static short getShort(EObject eObject, String path)
  {
    return getShort(get(eObject, path));
  }

  protected static short getShort(Object value)
  {
    if (value instanceof Number)
    {
      return ((Number)value).shortValue();
    }

    if (value instanceof String)
    {
      return Short.parseShort((String)value);
    }

    if (value == null)
    {
      return 0;
    }

    throw new IllegalArgumentException();
  }

  public static void setShort(EObject eObject, int propertyIndex, short value)
  {
    setShort(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setShort(EObject eObject, Property property, short value)
  {
    setShort(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setShort(EObject eObject, String path, short value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setShort(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setShort(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setShort(EObject eObject, EStructuralFeature eStructuralFeature, short value)
  {
    String name = eStructuralFeature.getEType().getInstanceClassName();
    if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, new Byte((byte)value));
    }
    else if (name == "java.lang.Double" || name == "double")
    {
      set(eObject, eStructuralFeature, new Double(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, new Integer(value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, new Long(value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, new Short(value));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(new Short(value)));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(new Short(value)));
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, String.valueOf(value));
    }
    else
    {
      // Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, new Short(value));
    }
  }

  public static String getString(EObject eObject, int propertyIndex)
  {
    return getString(get(eObject, getFeature(eObject, propertyIndex)));
  }

  public static String getString(EObject eObject, Property property)
  {
    return getString(get(eObject, ((EProperty)property).getEStructuralFeature()));
  }

  public static String getString(EObject eObject, String path)
  {
    return getString(get(eObject, path));
  }

  protected static String getString(Object value)
  {
    if (value instanceof String)
    {
      return (String)value;
    }

    if (value instanceof Number || value instanceof Boolean || value instanceof Character)
    {
      return String.valueOf(value);
    }

    if (value == null)
    {
      return null;
    }

    throw new IllegalArgumentException();
  }

  public static void setString(EObject eObject, int propertyIndex, String value)
  {
    setString(eObject, getFeature(eObject, propertyIndex), value);
  }

  public static void setString(EObject eObject, Property property, String value)
  {
    setString(eObject, ((EProperty)property).getEStructuralFeature(), value);
  }

  public static void setString(EObject eObject, String path, String value)
  {
    EProperty property = getType(eObject).getEProperty(path);
    if (property != null)
    {
      setString(eObject, property.getEStructuralFeature(), value);
    }
    else
    {
      Accessor accessor = Accessor.create(eObject, path);
      setString(accessor.getEObject(), accessor.getEStructuralFeature(), value);
      accessor.recycle();
    }
  }

  protected static void setString(EObject eObject, EStructuralFeature eStructuralFeature, String value)
  {
    EClassifier eType = eStructuralFeature.getEType();
    String name = eType.getInstanceClassName();

    if (value == null)
    {
      set(eObject, eStructuralFeature, eType.getDefaultValue());
    }
    else if (name == "java.lang.String")
    {
      set(eObject, eStructuralFeature, value);
    }
    else if (name == "java.lang.Byte" || name == "byte")
    {
      set(eObject, eStructuralFeature, Byte.valueOf(value));
    }
    else if (name == "java.lang.Double" || name == "double" || name == "java.lang.Number")
    {
      set(eObject, eStructuralFeature, Double.valueOf(value));
    }
    else if (name == "java.lang.Float" || name == "float")
    {
      set(eObject, eStructuralFeature, new Float(value));
    }
    else if (name == "java.lang.Integer" || name == "int")
    {
      set(eObject, eStructuralFeature, Integer.valueOf(value));
    }
    else if (name == "java.lang.Long" || name == "long")
    {
      set(eObject, eStructuralFeature, Long.valueOf(value));
    }
    else if (name == "java.lang.Short" || name == "short")
    {
      set(eObject, eStructuralFeature, Short.valueOf(value));
    }
    else if (name == "java.lang.Character" || name == "char")
    {
      set(eObject, eStructuralFeature, new Character(value.charAt(0)));
    }
    else if (name == "java.math.BigDecimal")
    {
      set(eObject, eStructuralFeature, getBigDecimal(value));
    }
    else if (name == "java.math.BigInteger")
    {
      set(eObject, eStructuralFeature, getBigInteger(value));
    }
    else if (name == "java.lang.Boolean" || name == "boolean")
    {
      set(eObject, eStructuralFeature, Boolean.valueOf(value));
    }
    else
    {
      // Instead of throwing an IllegalArgumentException we are passing the value to the feature
      set(eObject, eStructuralFeature, value);
    }
  }
}
