/**
 * <copyright>
 *
 * Copyright (c) 2003-2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: EvaluatePathAction.java,v 1.4 2007/05/28 13:09:04 emerks Exp $
 */
package org.eclipse.emf.ecore.sdo.action;


import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Date;
import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IActionDelegate;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionDelegate;

import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.sdo.EDataObject;
import org.eclipse.emf.ecore.sdo.SDOPackage;
import org.eclipse.emf.ecore.sdo.presentation.SDOEditorPlugin;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;

import commonj.sdo.DataObject;
import commonj.sdo.Sequence;


/**
 * 
 */
public class EvaluatePathAction 
  extends ActionDelegate
  implements IActionDelegate
{
  protected EDataObject eDataObject;

  public EvaluatePathAction()
  {
    super();
  }

  @Override
  public void run(IAction action)
  {
    EvaluationDialog dialog = new EvaluationDialog(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), eDataObject);
    dialog.open();
  }

  @Override
  public void selectionChanged(IAction action, ISelection selection) 
  {
    if (selection instanceof IStructuredSelection)
    {
      Object object = ((IStructuredSelection)selection).getFirstElement();
      object = AdapterFactoryEditingDomain.unwrap(object);
      if (object instanceof EDataObject)
      {
        eDataObject = (EDataObject)object;

        action.setEnabled(true);
        return;
      }
    }
    eDataObject = null;
    action.setEnabled(false);
  }

  public static class EvaluationDialog extends Dialog
  {
    protected static final Object [][] TYPE_DATA = 
      {
        new Object [] { "BigDecimal", Boolean.FALSE, BigDecimal.class, EcorePackage.eINSTANCE.getEBigDecimal() },
        new Object [] { "BigInteger", Boolean.FALSE, BigInteger.class, EcorePackage.eINSTANCE.getEBigInteger() },
        new Object [] { "Boolean", Boolean.FALSE, Boolean.TYPE, EcorePackage.eINSTANCE.getEBoolean() },
        new Object [] { "Byte", Boolean.FALSE, Byte.TYPE, EcorePackage.eINSTANCE.getEByte() },
        new Object [] { "Bytes", Boolean.FALSE, byte[].class, EcorePackage.eINSTANCE.getEByteArray() },
        new Object [] { "Char", Boolean.FALSE, Character.TYPE, EcorePackage.eINSTANCE.getEChar() },
        new Object [] { "DataObject", Boolean.FALSE, DataObject.class, SDOPackage.eINSTANCE.getDataObject() },
        new Object [] { "Date", Boolean.FALSE, Date.class, EcorePackage.eINSTANCE.getEDate() },
        new Object [] { "Double", Boolean.FALSE, Double.TYPE, EcorePackage.eINSTANCE.getEDouble() },
        new Object [] { "Float", Boolean.FALSE, Float.TYPE, EcorePackage.eINSTANCE.getEFloat() },
        new Object [] { "Int", Boolean.FALSE, Integer.TYPE, EcorePackage.eINSTANCE.getEInt() },
        new Object [] { "List", Boolean.FALSE, List.class, EcorePackage.eINSTANCE.getEJavaObject() },
        new Object [] { "Long", Boolean.FALSE, Long.TYPE, EcorePackage.eINSTANCE.getELong() },
        new Object [] { "Object", Boolean.TRUE, Object.class, EcorePackage.eINSTANCE.getEJavaObject() },
        new Object [] { "Sequence", Boolean.FALSE, Sequence.class, SDOPackage.eINSTANCE.getSequence() },
        new Object [] { "Short", Boolean.FALSE, Short.TYPE, EcorePackage.eINSTANCE.getEShort() },
        new Object [] { "String", Boolean.FALSE, String.class, EcorePackage.eINSTANCE.getEString() }
      };

    protected Method [][] methods = null;
    protected EDataObject eDataObject;
    protected Text pathText;
    protected Text resultText;
    protected Combo valueText;
    protected Combo operationChoice;
    protected Combo typeChoice;

    public EvaluationDialog(Shell shell, EDataObject eDataObject)
    {
      super(shell);
      setShellStyle(SWT.RESIZE | SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
      this.eDataObject = eDataObject;

      try
      {
        methods = new Method [TYPE_DATA.length][];
        for (int i = 0; i < TYPE_DATA.length; ++i)
        {
          methods[i] = 
            new Method []
            {
              EDataObject.class.getMethod
                ("get" + (TYPE_DATA[i][1] == Boolean.TRUE ? "" : TYPE_DATA[i][0]), new Class [] { String.class }),
              "Sequence".equals(TYPE_DATA[i][0]) ?
                null :
                EDataObject.class.getMethod
                  ("set" + (TYPE_DATA[i][1] == Boolean.TRUE ? "" : TYPE_DATA[i][0]), new Class [] { String.class, (Class<?>)TYPE_DATA[i][2] }),
            };
         }
       }
       catch (NoSuchMethodException exception)
       {
         exception.printStackTrace();
       }
    }

    @Override
    protected void configureShell(Shell newShell)
    {
      super.configureShell(newShell);
      newShell.setText
        (SDOEditorPlugin.INSTANCE.getString
          ("_UI_PathEvaluation_title", new Object [] { EcoreUtil.getURI(eDataObject).toString()}));
    }

    @Override
    protected Control createDialogArea(Composite parent)
    {
      Composite composite = new Composite(parent, SWT.NONE);
      // Composite composite = (Composite)super.createDialogArea(parent);

      {
        GridLayout layout = new GridLayout();
        layout.numColumns = 1;
        layout.verticalSpacing = 12;
        composite.setLayout(layout);

        GridData data = new GridData();
        data.verticalAlignment = GridData.FILL;
        data.grabExcessVerticalSpace = true;
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        composite.setLayoutData(data);
      }

      Label pathLabel = new Label(composite, SWT.LEFT);
      {
        pathLabel.setText(SDOEditorPlugin.INSTANCE.getString("_UI_Path_label"));
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        pathLabel.setLayoutData(data);
      }

      pathText = new Text(composite, SWT.SINGLE | SWT.BORDER);
      {
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        pathText.setLayoutData(data);
      }

      pathText.addModifyListener
        (new ModifyListener()
         {
           public void modifyText(ModifyEvent event)
           {
             resultText.setText("");
           }
         });

      Label operationLabel = new Label(composite, SWT.LEFT);
      {
        operationLabel.setText(SDOEditorPlugin.INSTANCE.getString("_UI_Operation_label"));
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        operationLabel.setLayoutData(data);
      }

      operationChoice = new Combo(composite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
      {
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        operationChoice.setLayoutData(data);
      }
      operationChoice.add("get");
      operationChoice.add("set");
      operationChoice.add("isSet");
      operationChoice.add("unset");
      operationChoice.select(0);
      operationChoice.addModifyListener
        (new ModifyListener()
         {
           public void modifyText(ModifyEvent event)
           {
             resultText.setText("");
             typeChoice.setEnabled(operationChoice.getSelectionIndex() == 0 || operationChoice.getSelectionIndex() == 1);
             valueText.setEnabled(operationChoice.getSelectionIndex() == 1);
           }
         });

      Label typeLabel = new Label(composite, SWT.LEFT);
      {
        typeLabel.setText(SDOEditorPlugin.INSTANCE.getString("_UI_Type_label"));
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        typeLabel.setLayoutData(data);
      }

      typeChoice = new Combo(composite, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
      {
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        typeChoice.setLayoutData(data);
      }
      for (int i = 0; i < TYPE_DATA.length; ++i)
      {
        typeChoice.add((String)TYPE_DATA[i][0]);
        if (TYPE_DATA[i][1] == Boolean.TRUE)
        {
          typeChoice.select(i);
        }
      }
      typeChoice.select(13);

      Label valueLabel = new Label(composite, SWT.LEFT);
      {
        valueLabel.setText(SDOEditorPlugin.INSTANCE.getString("_UI_Value_label"));
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        valueLabel.setLayoutData(data);
      }

      valueText = new Combo(composite, SWT.SINGLE | SWT.BORDER);
      valueText.setEnabled(false);
      {
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        valueText.setLayoutData(data);
      }

      Label resultLabel = new Label(composite, SWT.LEFT);
      {
        resultLabel.setText(SDOEditorPlugin.INSTANCE.getString("_UI_Result_label"));
        GridData data = new GridData();
        data.horizontalAlignment = GridData.FILL;
        resultLabel.setLayoutData(data);
      }

      resultText = new Text(composite, SWT.MULTI | SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL);
      {
        GridData data = new GridData();
        data.heightHint = 200;
        data.horizontalAlignment = GridData.FILL;
        data.grabExcessHorizontalSpace = true;
        data.widthHint = 500;
        data.verticalAlignment = GridData.FILL;
        data.grabExcessVerticalSpace = true;
        resultText.setLayoutData(data);
      }
      return composite;
    }

    @Override
    protected void createButtonsForButtonBar(Composite parent) 
    {
      createButton(parent, IDialogConstants.CLIENT_ID + 1, "Evaluate", true);
      createButton(parent, IDialogConstants.OK_ID, IDialogConstants.OK_LABEL, false);
      createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CANCEL_LABEL, false);
    }

    @Override
    protected void buttonPressed(int buttonID) 
    {
      super.buttonPressed(buttonID);
      if (buttonID == IDialogConstants.CLIENT_ID + 1)
      {
        pathText.setFocus();
        evaluate();
      }
    }

    protected void evaluate()
    {
      try
      {
        Object result = null;
        switch (operationChoice.getSelectionIndex())
        {
          case 0:
          {
            result = 
              methods[typeChoice.getSelectionIndex()][0].invoke
                (eDataObject, new Object [] { pathText.getText() });
            break;
          }
          case 1:
          {
            if (TYPE_DATA[typeChoice.getSelectionIndex()][3] instanceof EDataType)
            {
              EDataType eDataType = (EDataType)TYPE_DATA[typeChoice.getSelectionIndex()][3];
              if (eDataType.isSerializable())
              {
                result = EcoreUtil.createFromString(eDataType, valueText.getText());
              }
              else
              {
                result = valueText.getText();
              }
            }

            methods[typeChoice.getSelectionIndex()][1].invoke
              (eDataObject, new Object [] { pathText.getText(), result });
            break;
          }
          case 2:
          {
            result = eDataObject.isSet(pathText.getText()) ? Boolean. TRUE : Boolean.FALSE;
            break;
          }
          case 3:
          {
            eDataObject.unset(pathText.getText());
            break;
          }
        }

        if (result == null)
        {
          resultText.setText("null");
        }
        else if (TYPE_DATA[typeChoice.getSelectionIndex()][3] instanceof EDataType)
        {
          EDataType eDataType = (EDataType)TYPE_DATA[typeChoice.getSelectionIndex()][3];
          if (eDataType.isSerializable())
          {
            resultText.setText(EcoreUtil.convertToString(eDataType, result));
          }
          else if (result instanceof List)
          {
            StringBuffer text = new StringBuffer();
            for (Object object : ((List<?>)result))
            {
              text.append(object);
              text.append("\n");
            }
            resultText.setText(text.toString());
          }
          else
          {
            resultText.setText(result.toString());
          }
        }
        else
        {
          resultText.setText(result.toString());
        }
      }
      catch (InvocationTargetException exception)
      {
        StringWriter writer = new StringWriter();
        exception.getTargetException().printStackTrace(new PrintWriter(writer));
        resultText.setText(writer.getBuffer().toString());
      }
      catch (Exception exception)
      {
        StringWriter writer = new StringWriter();
        exception.printStackTrace(new PrintWriter(writer));
        resultText.setText(writer.getBuffer().toString());
      }
    }
  }
}
