/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.ui.editor.contentassist;

import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.preferences.IPreferencesService;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.dltk.core.IType;
import org.eclipse.dltk.ui.PreferenceConstants;
import org.eclipse.dltk.ui.text.ScriptTextTools;
import org.eclipse.dltk.ui.text.completion.ScriptCompletionProposal;
import org.eclipse.jface.internal.text.html.BrowserInformationControl;
import org.eclipse.jface.text.DefaultInformationControl;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IInformationControl;
import org.eclipse.jface.text.IInformationControlCreator;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.viewers.StyledString;
import org.eclipse.php.core.compiler.ast.nodes.NamespaceReference;
import org.eclipse.php.internal.core.PHPCoreConstants;
import org.eclipse.php.internal.core.PHPCorePlugin;
import org.eclipse.php.internal.core.codeassist.ProposalExtraInfo;
import org.eclipse.php.internal.core.codeassist.strategies.IncludeStatementStrategy;
import org.eclipse.php.internal.core.typeinference.PHPModelUtils;
import org.eclipse.php.internal.ui.PHPUiPlugin;
import org.eclipse.php.internal.ui.text.hover.CompletionHoverControlCreator;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Shell;

public class PHPCompletionProposal extends ScriptCompletionProposal implements IPHPCompletionProposalExtension {

	/**
	 * The control creator.
	 */
	private IInformationControlCreator fCreator;

	public PHPCompletionProposal(String replacementString, int replacementOffset, int replacementLength, Image image,
			String displayString, int relevance) {
		super(replacementString, replacementOffset, replacementLength, image, displayString, relevance);
	}

	public PHPCompletionProposal(String replacementString, int replacementOffset, int replacementLength, Image image,
			StyledString displayString, int relevance) {
		super(replacementString, replacementOffset, replacementLength, image, displayString, relevance, false);
	}

	public PHPCompletionProposal(String replacementString, int replacementOffset, int replacementLength, Image image,
			StyledString displayString, int relevance, boolean indoc) {
		super(replacementString, replacementOffset, replacementLength, image, displayString, relevance, indoc);
	}

	@Override
	protected int getPatternMatchRule(String pattern, String string) {
		// XXX: avoid unnecessary StringIndexOutOfBoundsExceptions in
		// super.getPatternMatchRule(pattern, string) because
		// "pattern" and "string" may have nothing in common
		// when generated by class PHPCompletionProposal
		if (string.length() < pattern.length()) {
			return -1;
		}
		return super.getPatternMatchRule(pattern, string);
	}

	@Override
	protected boolean isValidPrefix(String prefix) {
		String word = getDisplayString();
		if (word.startsWith("$") && !prefix.startsWith("$")) { //$NON-NLS-1$ //$NON-NLS-2$
			word = word.substring(1);
		}
		boolean result = isPrefix(prefix, word);
		if (!result && prefix.length() > 0 && prefix.charAt(0) == NamespaceReference.NAMESPACE_SEPARATOR) {
			result = isPrefix(prefix.substring(1), word);
		}

		if (!result && ProposalExtraInfo.isMemberInNamespace(getExtraInfo()) && (getModelElement() instanceof IType)) {
			IType type = (IType) getModelElement();
			if (ProposalExtraInfo.isAbsoluteName(getExtraInfo())) {
				result = isPrefix(prefix.substring(1), PHPModelUtils.getFullName(type));
			} else {
				result = isPrefix(prefix, PHPModelUtils.getFullName(type));
			}
		}
		if (!result && ProposalExtraInfo.isFullName(getExtraInfo()) && getModelElement() != null) {
			if (ProposalExtraInfo.isAbsoluteName(getExtraInfo())) {
				result = isPrefix(prefix.substring(1), PHPModelUtils.getFullName(getModelElement()));
			} else {
				result = isPrefix(prefix, PHPModelUtils.getFullName(getModelElement()));
			}
		}
		// int index = word.indexOf(" - ");
		// if (!result && index >= 0) {
		// StringBuffer sb = new StringBuffer();
		// sb.append(word.substring(index + " - ".length()));
		// sb.append('\\');
		// sb.append(word.substring(0, index));
		// result = isPrefix(prefix, sb.toString());
		// }
		return result;
	}

	@Override
	protected boolean isSmartTrigger(char trigger) {
		return trigger == '$';
	}

	@Override
	public void apply(IDocument document, char trigger, int offset) {
		IModelElement modelElement = getModelElement();

		boolean activateCodeAssist = false;
		String replacementString = getReplacementString();
		if (modelElement instanceof IScriptProject
				&& replacementString.endsWith(IncludeStatementStrategy.FOLDER_SEPARATOR)) {
			// workaround for:
			// https://bugs.eclipse.org/bugs/show_bug.cgi?id=269634
			activateCodeAssist = true;
		} else {
			IPreferencesService preferencesService = Platform.getPreferencesService();
			boolean enableAutoactivation = preferencesService.getBoolean(PHPCorePlugin.ID,
					PHPCoreConstants.CODEASSIST_AUTOACTIVATION, false, null);
			if (enableAutoactivation) {
				char lastChar = replacementString.charAt(replacementString.length() - 1);
				for (char autoActivationChar : PHPCompletionProcessor.completionAutoActivationChars) {
					if (autoActivationChar == lastChar) {
						activateCodeAssist = true;
						break;
					}
				}
			}
		}
		if (activateCodeAssist) {
			AutoActivationTrigger.register(document);
		}

		UseStatementInjector injector = new UseStatementInjector(this);
		offset = injector.inject(document, getTextViewer(), offset);

		super.apply(document, trigger, offset);

		setCursorPosition(calcCursorPosition());
	}

	private int calcCursorPosition() {
		String replacementString = getReplacementString();
		int i = replacementString.lastIndexOf('(');
		if (i != -1) {
			return i + 1;
		}
		i = replacementString.lastIndexOf('\'');
		if (i != -1) {
			return i;
		}
		i = replacementString.lastIndexOf('\"');
		if (i != -1) {
			return i;
		}
		return replacementString.length();
	}

	@Override
	public IContextInformation getContextInformation() {
		String displayString = getDisplayString();
		if (displayString.indexOf('(') == -1) {
			return null;
		}
		return super.getContextInformation();
	}

	@Override
	protected boolean isCamelCaseMatching() {
		return true;
	}

	@Override
	protected boolean insertCompletion() {
		return Platform.getPreferencesService().getBoolean(PHPCorePlugin.ID,
				PHPCoreConstants.CODEASSIST_INSERT_COMPLETION, true, null);
	}

	@Override
	protected ScriptTextTools getTextTools() {
		return PHPUiPlugin.getDefault().getTextTools();
	}

	@Override
	public IInformationControlCreator getInformationControlCreator() {
		if (fCreator == null) {
			fCreator = new CompletionHoverControlCreator(new IInformationControlCreator() {
				@Override
				public IInformationControl createInformationControl(Shell parent) {
					if (BrowserInformationControl.isAvailable(parent)) {
						return new BrowserInformationControl(parent, PreferenceConstants.APPEARANCE_DOCUMENTATION_FONT,
								true);
					} else {
						return new DefaultInformationControl(parent, true);
					}
				}
			}, true);
		}
		return fCreator;
	}

	@Override
	public Object getExtraInfo() {
		return ProposalExtraInfo.DEFAULT;
	}
}