/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.server.ui.launching;

import java.net.MalformedURLException;
import java.net.URL;

import org.eclipse.core.internal.content.ContentTypeManager;
import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.content.IContentType;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.internal.ui.SWTFactory;
import org.eclipse.dltk.core.IScriptProject;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.TitleAreaDialog;
import org.eclipse.php.internal.core.documentModel.provisional.contenttype.ContentTypeIdForPHP;
import org.eclipse.php.internal.core.project.PHPNature;
import org.eclipse.php.internal.debug.ui.Logger;
import org.eclipse.php.internal.debug.ui.PHPDebugUIPlugin;
import org.eclipse.php.internal.debug.ui.launching.LaunchUtilities;
import org.eclipse.php.internal.server.PHPServerUIMessages;
import org.eclipse.php.internal.server.core.Server;
import org.eclipse.php.internal.server.core.manager.ServersManager;
import org.eclipse.php.internal.server.ui.ServersPluginImages;
import org.eclipse.php.internal.ui.IPHPHelpContextIds;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.PlatformUI;

/**
 *
 * Dialog for instant launch of PHP Web Page projects  
 */
public class PHPWebPageLaunchDialog extends TitleAreaDialog {

	private final Server server;
	private final IScriptProject project;
	private Text fFile;
	private Button fileButton;
	private final String mode;
	private Button breakOnFirstLine;
	private Button autoGeneratedURL;
	private Label fURLLabel;
	private Text fURLHost;
	private Text fURLPath;
	private WidgetListener fListener = new WidgetListener();
	private boolean isBreakAtFirstLine;
	private String pathString;
	private String fileName;

	public PHPWebPageLaunchDialog(String mode, IScriptProject obj) {
		super(PHPDebugUIPlugin.getActiveWorkbenchShell());
		// , "Launch Web Page",  null, "", INFORMATION, new String[] { IDialogConstants.OK_LABEL, IDialogConstants.CANCEL_LABEL }, 0);
		this.mode = mode;
		this.project = obj;
		this.server = ServersManager.getLocalServer(obj.getProject());
	}

	/**
	 * Create a drop-down list for the URL.
	 */
	protected Control createDialogArea(Composite parent) {
		parent = (Composite) super.createDialogArea(parent);

		setTitle(getActionTitle(this.mode));
		setMessage(getActionMessage(this.mode));
		setTitleImage(ServersPluginImages.DESC_WIZ_SERVER.createImage());
		
		Composite content = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout(1, false);
		GridData gd = new GridData(GridData.FILL_BOTH);
		content.setLayout(layout);
		content.setLayoutData(gd);
		
		createFileComponent(content);
		createBreakControl(content);
		createURLControl(content);
		
		initializeURLControl();
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, IPHPHelpContextIds.DEBUGGING);
		return parent;
	}
	
	@Override
	protected Control createContents(Composite parent) {
		final Control createContents = super.createContents(parent);
		disableOkButton();		
		return createContents;
	}

	private void disableOkButton() {
		getButton(IDialogConstants.OK_ID).setEnabled(false);
	}

	private void enableOkButton() {
		getButton(IDialogConstants.OK_ID).setEnabled(true);
	}

	
	private String getActionMessage(String mode) {
		if (ILaunchManager.DEBUG_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.0"); //$NON-NLS-1$
		} else if (ILaunchManager.RUN_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.1"); //$NON-NLS-1$
		} else if (ILaunchManager.PROFILE_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.2"); //$NON-NLS-1$
		}
		throw new IllegalArgumentException();
	}

	private String getActionTitle(String mode) {
		if (ILaunchManager.DEBUG_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.3"); //$NON-NLS-1$
		} else if (ILaunchManager.RUN_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.4"); //$NON-NLS-1$
		} else if (ILaunchManager.PROFILE_MODE.equals(mode)) {
			return PHPServerUIMessages.getString("PHPWebPageLaunchDialog.5"); //$NON-NLS-1$
		}
		throw new IllegalArgumentException();
	}

	// In case this is a debug mode, display 'Break on first line' attribute checkbox.
	protected void createBreakControl(Composite parent) {
		if (!ILaunchManager.DEBUG_MODE.equals(this.mode)) {
			return;
		}
		
		Group group = new Group(parent, SWT.NONE);
		group.setText(PHPServerUIMessages.getString("PHPWebPageLaunchDialog.6")); //$NON-NLS-1$
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(layout);
		group.setLayoutData(gridData);

		breakOnFirstLine = createCheckButton(group, PHPServerUIMessages.getString("PHPWebPageLaunchDialog.7")); //$NON-NLS-1$
		breakOnFirstLine.addSelectionListener(new SelectionAdapter() {

			public void widgetSelected(SelectionEvent e) {
				PHPWebPageLaunchDialog.this.isBreakAtFirstLine = breakOnFirstLine.getSelection();
				// updateLaunchConfigurationDialog();
			}
		});
		
	}

	private Button createCheckButton(Group group, String string) {
		return SWTFactory.createCheckButton(group, string, null, false, 1);
	}

	protected void createFileComponent(Composite parent) {
		Group group = new Group(parent, SWT.NONE);
		String projectLabel = PHPServerUIMessages.getString("ServerTab.file_project"); //$NON-NLS-1$
		group.setText(projectLabel);
		GridLayout layout = new GridLayout(3, false);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(layout);
		group.setLayoutData(gd);

		fFile = new Text(group, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		//gridData = new GridData();
		//gridData.widthHint = IDialogConstants.ENTRY_FIELD_WIDTH;
		fFile.setLayoutData(gd);
		fFile.addModifyListener(fListener);
		fileButton = createPushButton(group, PHPServerUIMessages.getString("ServerTab.browse")); //$NON-NLS-1$
		gd = (GridData) fileButton.getLayoutData();
		gd.horizontalSpan = 1;
		fileButton.addSelectionListener(fListener);
	}
	
	public void createURLControl(Composite composite) {
		Group group = new Group(composite, SWT.NONE);
		String projectLabel = PHPServerUIMessages.getString("ServerTab.url"); //$NON-NLS-1$
		group.setText(projectLabel);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(layout);
		group.setLayoutData(gridData);

		autoGeneratedURL = new Button(group, SWT.CHECK);
		autoGeneratedURL.setText(PHPServerUIMessages.getString("ServerTab.autoGenerate")); //$NON-NLS-1$
		autoGeneratedURL.setSelection(true);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		autoGeneratedURL.setLayoutData(gridData);
		autoGeneratedURL.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				initializeURLControl();
				// updateLaunchConfigurationDialog();
			}
		});
		fURLLabel = new Label(group, SWT.NONE);
		fURLLabel.setText(PHPServerUIMessages.getString("ServerTab.urlLabel")); //$NON-NLS-1$
		gridData = new GridData();
		gridData.horizontalIndent = 20;
		gridData.horizontalSpan = 1;
		fURLLabel.setLayoutData(gridData);

		Composite urlComposite = new Composite(group, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		urlComposite.setLayout(gridLayout);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		urlComposite.setLayoutData(gridData);

		fURLHost = new Text(urlComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		fURLHost.setLayoutData(gridData);
		fURLHost.setEnabled(false);

		fURLPath = new Text(urlComposite, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		fURLPath.setLayoutData(gridData);
		fURLPath.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				// updateLaunchConfigurationDialog();
			}
		});
	}
	
	
	private Button createPushButton(Group group, String string) {
		return SWTFactory.createPushButton(group, string, null);	
	}

	/**
	 * Override the okPressed to save the URL to the URLs history for this PHP IDE session.
	 * Also, add the URL to the launch configuration attributes.
	 */
	protected void buttonPressed(int buttonId) {
		if (buttonId == OK) {

		
		}
		super.buttonPressed(buttonId);
	}

	public Server getServer() {
		return this.server;
	}

	public String getPhpPathString() {
		return this.pathString;
	}
	
	protected class WidgetListener extends SelectionAdapter implements ModifyListener {
		public void modifyText(ModifyEvent e) {
			initializeURLControl();
			validate();
		}

		public void widgetSelected(SelectionEvent e) {
			Object source = e.getSource();
			if (source == fileButton) {
				handleFileButtonSelected();
			}
			validate();			
		}
	}
	
	protected void handleFileButtonSelected() {

		IResource file = null;
		file = getFileFromDialog(this.project.getProject());

		if (file == null)
			return;

		String fName = file.getFullPath().toString();
		fFile.setText(fName);

		if (autoGeneratedURL.getSelection()) {
			updateURLComponents(computeURL(formatFileName(fName)));
		}
	}
	
	private IResource getFileFromDialog(IProject project) {
		final String contenttypeidPhp = ContentTypeIdForPHP.ContentTypeID_PHP;
		final IContentType contentType = ContentTypeManager.getInstance().getContentType(contenttypeidPhp);
		final String[] fileSpecs = contentType.getFileSpecs(IContentType.FILE_EXTENSION_SPEC);
		return LaunchUtilities.getFileFromDialog(project, getShell(), fileSpecs , new String[] { PHPNature.ID } , false);
	}

	protected void initializeURLControl() {
		if (fFile == null || fURLPath == null || fURLHost == null) {
			return;
		}
		String file;
		file = autoGeneratedURL.getSelection() ? formatFileName(fFile.getText()) : fURLPath.getText();
		fURLPath.setEnabled(!autoGeneratedURL.getSelection());
		updateURLComponents(computeURL(file));
	}
	
	/**
	 * Constructs the URL string according to the given context root and the file name.
	 * @param fileName
	 * @return
	 */
	protected String computeURL(String fileName) {
		if (server == null) {
			return ""; //$NON-NLS-1$
		}
		String urlString = server.getBaseURL();

		if (urlString.equals("")) { //$NON-NLS-1$
			urlString = "http://localhost"; //$NON-NLS-1$
		}
		StringBuffer url = new StringBuffer(urlString);
		if (!fileName.equals("")) { //$NON-NLS-1$
			url.append(fileName);
		}
		return url.toString();
	}
	
	private String formatFileName(String fileName) {
		String formatFile = null;

		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource resource = root.findMember(fileName);
		if (resource == null) {
			return fileName;
		}
		int type = resource.getType();
		if (type == IResource.FILE || type == IResource.FOLDER) {
			formatFile = resource.getFullPath().toString();
		} else {
			formatFile = "/"; //$NON-NLS-1$
		}
		if (!formatFile.startsWith("/")) { //$NON-NLS-1$
			formatFile = "/" + formatFile; //$NON-NLS-1$
		}
		return formatFile;
	}

	protected void updateURLComponents(String urlStr) {
		try {
			URL url = new URL(urlStr);
			String port = url.getPort() == -1 ? "" : ":" + url.getPort();
			fURLHost.setText(url.getProtocol() + "://" + url.getHost() + port + "/"); //$NON-NLS-1$ //$NON-NLS-2$
			if (url.getQuery() != null) {
				fURLPath.setText(url.getPath() + "?" + url.getQuery());
			} else {
				fURLPath.setText(url.getPath());
			}
		} catch (MalformedURLException e) {
			Logger.logException(e);
		}
	}

	public boolean isBreakAtFirstLine() {
		return isBreakAtFirstLine;
	}
	
	private boolean validate() {
		String fileName = this.fFile.getText();
		if (fileName == null || fileName.length() == 0) {
			return false;
		}
		IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(fileName);
		if (!(resource instanceof IFile)) {
			setErrorMessage(PHPServerUIMessages.getString("ServerTab.file_project_doesNotExist")); //$NON-NLS-1$
			return false;
		}
		setErrorMessage(null);
		return true;
	}
	
	@Override
	public void setErrorMessage(String newErrorMessage) {
		if (newErrorMessage != null) {
			disableOkButton();	
		} else {
			enableOkButton();
		}
		super.setErrorMessage(newErrorMessage);
	}
	
	@Override
	protected void okPressed() {
		String urlPath = fURLPath.getText().replace('\\', '/');
		if (urlPath.startsWith("/")) {
			urlPath = urlPath.substring(1);
		}
		this.pathString =  this.fURLHost.getText() + urlPath;
		this.fileName = this.fFile.getText();

		super.okPressed();
		
	}

	public String getFilename() {
		return this.fileName;
	}
	
}
