/*******************************************************************************
 * Copyright (c) 2006 Zend Corporation and IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.php.internal.core.documentModel.parser;

import java.io.IOException;
import java.io.Reader;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.text.Segment;

import org.eclipse.core.resources.IProject;
import org.eclipse.php.internal.core.documentModel.parser.regions.PHPRegionTypes;
import org.eclipse.php.internal.core.documentModel.partitioner.PHPPartitionTypes;
import org.eclipse.php.internal.core.preferences.TaskPatternsProvider;
import org.eclipse.php.internal.core.util.collections.IntHashtable;
import org.eclipse.wst.sse.core.internal.parser.ContextRegion;
import org.eclipse.wst.sse.core.internal.provisional.text.ITextRegion;

public abstract class PhpLexer implements Scanner, PHPRegionTypes {

	// should be auto-generated by jflex
	protected abstract int getYy_endRead();

	protected abstract int getYy_lexical_state();

	protected abstract int getYy_markedPos();

	protected abstract int getYy_pushBackPosition();

	protected abstract int getYy_startRead();

	protected abstract void pushBack(int i);

	public abstract char[] getYy_buffer();

	public abstract void yybegin(int newState);

	public abstract int yylength();

	public abstract String yytext();

	public abstract void reset(Reader reader, char[] buffer, int[] parameters);

	public abstract int yystate();
	
	protected abstract boolean isHeredocState(int state);

	public abstract int[] getParamenters();

	// A pool of states. To avoid creation of a new state on each createMemento.
	private static final IntHashtable lexerStates = new IntHashtable(100);

	/**
	 * This character denotes the end of file
	 */
	final public static int YYEOF = -1;

	protected static final boolean isLowerCase(final String text) {
		if (text == null)
			return false;
		for (int i = 0; i < text.length(); i++)
			if (!Character.isLowerCase(text.charAt(i)))
				return false;
		return true;
	}

	protected boolean asp_tags = true;

	protected int defaultReturnValue = -1;

	protected int firstPos = -1; // the first position in the array

	protected String heredoc = null;
	protected int heredoc_len = 0;
	protected StateStack phpStack;

	/**
	 * build a key that represents the current state of the lexer.
	 */
	private int buildStateKey() {
		int rv = getYy_lexical_state();

		for (int i = 0; i < phpStack.size(); i++)
			rv = 31 * rv + phpStack.get(i);
		if (heredoc != null)
			for (int i = 0; i < heredoc.length(); i++)
				rv = 31 * rv + heredoc.charAt(i);
		return rv;
	}

	public Object createLexicalStateMemento() {
		// buffered token state
		if (bufferedTokens != null && !bufferedTokens.isEmpty()) {
			return bufferedState;
		}

		//System.out.println("lexerStates size:" + lexerStates.size());
		final int key = buildStateKey();
		Object state = lexerStates.get(key);
		if (state == null) {
			state = new BasicLexerState(this);
			if (isHeredocState(getYy_lexical_state()))
				state = new HeredocState((BasicLexerState) state, this);
			lexerStates.put(key, state);
		}
		return state;
	}

	public boolean getAspTags() {
		return asp_tags;
	}

	// lex to the EOF. and return the ending state.
	public Object getEndingState() throws IOException {
		lexToEnd();
		return createLexicalStateMemento();
	}

	/**
	 * return the index where start we started to lex.
	 */
	public int getFirstIndex() {
		return firstPos;
	}

	public int getMarkedPos() {
		return getYy_markedPos();
	}

	public void getText(final int start, final int length, final Segment s) {
		if (start + length > getYy_endRead())
			throw new RuntimeException("bad segment !!"); //$NON-NLS-1$
		s.array = getYy_buffer();
		s.offset = start;
		s.count = length;
	}

	public int getTokenStart() {
		return getYy_startRead() - getYy_pushBackPosition();
	}

	/**
	 * reset to a new segment. this do not change the state of the lexer.
	 * This method is used to scan nore than one segment as if the are one segment.
	 */
	public void reset(final Segment s) {
		reset(s.array, s.offset, s.count);
	}

	public void initialize(final int state) {
		phpStack = new StateStack();
		yybegin(state);
	}

	/**
	 * reset to a new segment. this do not change the state of the lexer.
	 * This method is used to scan nore than one segment as if the are one segment.
	 */

	// lex to the end of the stream.
	public String lexToEnd() throws IOException {
		String curr = yylex();
		String last = curr;
		while (curr != null) {
			last = curr;
			curr = yylex();
		}
		return last;
	}

	public String lexToTokenAt(final int offset) throws IOException {
		if (firstPos + offset < getYy_markedPos())
			throw new RuntimeException("Bad offset"); //$NON-NLS-1$
		String t = yylex();
		while (getYy_markedPos() < firstPos + offset && t != null)
			t = yylex();
		return t;
	}

	protected void popState() {
		yybegin(phpStack.popStack());
	}

	protected void pushState(final int state) {
		phpStack.pushStack(getYy_lexical_state());
		yybegin(state);
	}

	public void setAspTags(final boolean b) {
		asp_tags = b;
	}

	public void setState(final Object state) {
		((LexerState) state).restoreState(this);
	}

	public int yystart() {
		return getYy_startRead();
	}

	public LinkedList bufferedTokens = null;
	public int bufferedLength;
	public Object bufferedState;

	/**
	 * @return the next token from the php lexer
	 * @throws IOException
	 */
	public String getNextToken() throws IOException {
		if (bufferedTokens != null) {
			if (bufferedTokens.isEmpty()) {
				bufferedTokens = null;
			} else {
				return removeFromBuffer();
			}
		}

		bufferedState = createLexicalStateMemento();
		String yylex = yylex();
		if (PHPPartitionTypes.isPHPDocCommentState(yylex)) {
			final StringBuffer buffer = new StringBuffer();
			int length = 0;
			while (PHPPartitionTypes.isPHPDocCommentState(yylex)) {
				buffer.append(yytext());
				yylex = yylex();
				length++;
			}
			bufferedTokens = new LinkedList();
			checkForTodo(bufferedTokens, PHPRegionTypes.PHPDOC_COMMENT, 0, length, buffer.toString());
			bufferedTokens.add(new ContextRegion(yylex, 0, yylength(), yylength()));
			yylex = removeFromBuffer();
		} else if (PHPPartitionTypes.isPHPCommentState(yylex)) {
			bufferedTokens = new LinkedList();
			checkForTodo(bufferedTokens, yylex, 0, yylength(), yytext());
			yylex = removeFromBuffer();
		}

		if (yylex == PHP_CLOSETAG) {
			pushBack(getLength());
		}

		return yylex;
	}

	/**
	 * @return the last token from buffer
	 */
	private String removeFromBuffer() {
		ITextRegion region = (ITextRegion) bufferedTokens.removeFirst();
		bufferedLength = region.getLength();
		return region.getType();
	}

	public int getLength() {
		return bufferedTokens == null ? yylength() : bufferedLength;
	}

	private Pattern[] todos;

	public void setPatterns(IProject project) {
		if (project != null) {
			todos = TaskPatternsProvider.getInstance().getPatternsForProject(project);
		} else {
			todos = TaskPatternsProvider.getInstance().getPetternsForWorkspace();
		}
	}

	/**
	 * @param bufferedTokens2 
	 * @param token
	 * @param commentStart
	 * @param commentLength
	 * @param comment
	 * @return a list of todo ITextRegion
	 */
	private void checkForTodo(List result, String token, int commentStart, int commentLength, String comment) {
		ArrayList matchers = createMatcherList(comment);
		int startPosition = 0;

		Matcher matcher = getMinimalMatcher(matchers, startPosition);
		ITextRegion tRegion = null;
		while (matcher != null) {
			int startIndex = matcher.start();
			int endIndex = matcher.end();
			if (startIndex != startPosition) {
				tRegion = new ContextRegion(token, commentStart + startPosition, startIndex - startPosition, startIndex - startPosition);
				result.add(tRegion);
			}
			tRegion = new ContextRegion(PHPRegionTypes.TASK, commentStart + startIndex, endIndex - startIndex, endIndex - startIndex);
			result.add(tRegion);
			startPosition = endIndex;
			matcher = getMinimalMatcher(matchers, startPosition);
		}
		final int length = commentLength - startPosition;
		result.add(new ContextRegion(token, commentStart + startPosition, length, length));
	}

	private ArrayList createMatcherList(String content) {
		ArrayList list = new ArrayList(todos.length);
		for (int i = 0; i < todos.length; i++) {
			list.add(i, todos[i].matcher(content));
		}
		return list;
	}

	private Matcher getMinimalMatcher(ArrayList matchers, int startPosition) {
		Matcher minimal = null;
		int size = matchers.size();
		for (int i = 0; i < size;) {
			Matcher tmp = (Matcher) matchers.get(i);
			if (tmp.find(startPosition)) {
				if (minimal == null || tmp.start() < minimal.start()) {
					minimal = tmp;
				}
				i++;
			} else {
				matchers.remove(i);
				size--;
			}
		}
		return minimal;
	}

	private static class BasicLexerState implements LexerState {

		private final byte lexicalState;
		private StateStack phpStack;

		public BasicLexerState(PhpLexer lexer) {
			if (!lexer.phpStack.isEmpty()) {
				phpStack = lexer.phpStack.createClone();
			}
			lexicalState = (byte) lexer.getYy_lexical_state();
		}

		@Override
		public boolean equals(final Object o) {
			if (o == this)
				return true;
			if (o == null)
				return false;
			if (!(o instanceof BasicLexerState))
				return false;
			final BasicLexerState tmp = (BasicLexerState) o;
			if (tmp.lexicalState != lexicalState)
				return false;
			if (phpStack != null && !phpStack.equals(tmp.phpStack))
				return false;
			return phpStack == tmp.phpStack;
		}

		public boolean equalsCurrentStack(final LexerState obj) {
			if (obj == this)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof BasicLexerState))
				return false;
			final BasicLexerState tmp = (BasicLexerState) obj;
			if (tmp.lexicalState != lexicalState)
				return false;
			final StateStack activeStack = getActiveStack();
			final StateStack otherActiveStack = tmp.getActiveStack();
			if (!(activeStack == otherActiveStack || activeStack != null && activeStack.equals(otherActiveStack)))
				return false;
			return true;
		}

		public boolean equalsTop(final LexerState obj) {
			return obj != null && obj.getTopState() == lexicalState;
		}

		protected StateStack getActiveStack() {
			return phpStack;
		}

		public int getTopState() {
			return lexicalState;
		}

		public boolean isSubstateOf(final int state) {
			if (lexicalState == state)
				return true;
			final StateStack activeStack = getActiveStack();
			if (activeStack == null)
				return false;
			return activeStack.contains(state);
		}

		public void restoreState(final Scanner scanner) {
			final PhpLexer lexer = (PhpLexer) scanner;

			if (phpStack == null)
				lexer.phpStack.clear();
			else
				lexer.phpStack.copyFrom(phpStack);

			lexer.yybegin(lexicalState);
		}

		@Override
		public String toString() {
			final StateStack stack = getActiveStack();
			final String stackStr = stack == null ? "null" : stack.toString(); //$NON-NLS-1$
			return "Stack: " + stackStr + ", currState: " + lexicalState; //$NON-NLS-1$ //$NON-NLS-2$
		}

	}

	private static class HeredocState implements LexerState {
		private final String myHeredoc;
		private final BasicLexerState theState;

		public HeredocState(final BasicLexerState state, PhpLexer lexer) {
			theState = state;
			myHeredoc = lexer.heredoc;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (getClass() != obj.getClass())
				return false;
			final HeredocState other = (HeredocState) obj;
			if (myHeredoc == null) {
				if (other.myHeredoc != null)
					return false;
			} else if (!myHeredoc.equals(other.myHeredoc))
				return false;
			if (theState == null) {
				if (other.theState != null)
					return false;
			} else if (!theState.equals(other.theState))
				return false;
			return true;
		}

		public boolean equalsCurrentStack(final LexerState obj) {
			if (obj == this)
				return true;
			if (obj == null)
				return false;
			if (!(obj instanceof HeredocState))
				return false;
			return theState.equals(((HeredocState) obj).theState);
		}

		public boolean equalsTop(final LexerState obj) {
			return theState.equalsTop(obj);
		}

		public int getTopState() {
			return theState.getTopState();
		}

		public boolean isSubstateOf(final int state) {
			return theState.isSubstateOf(state);
		}

		public void restoreState(final Scanner scanner) {
			final PhpLexer lexer = (PhpLexer) scanner;
			theState.restoreState(lexer);
			lexer.heredoc = myHeredoc;
			lexer.heredoc_len = myHeredoc.length();
		}

	}

}
