/*******************************************************************************
 * Copyright (c) 2006 Zend Corporation and IBM Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   Zend and IBM - Initial implementation
 *******************************************************************************/
package org.eclipse.php.server.ui;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.*;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.window.Window;
import org.eclipse.php.core.util.FileUtils;
import org.eclipse.php.server.PHPServerUIMessages;
import org.eclipse.php.server.core.Server;
import org.eclipse.php.server.core.manager.ServersManager;
import org.eclipse.php.server.internal.ui.ClosableWizardDialog;
import org.eclipse.php.server.internal.ui.ServerWizard;
import org.eclipse.php.server.ui.wizard.WizardModel;
import org.eclipse.php.ui.preferences.ui.ScrolledCompositeImpl;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.PlatformUI;

public class ServerTab extends AbstractLaunchConfigurationTab {

	protected IFile selectedFile = null;

	protected Text fFile;
	protected Label fPublishToLabel;
	protected Text fPublishToRoot;
	protected Text fPublishTo;
	
//	protected Label fContextRootLabel;
	protected Label fURLLabel;
	protected Text fURL;
//	protected Text fContextRoot;
	
	protected Button projectButton;
	protected Button fileButton;
	protected Button publish;
	protected Button createNewServer;
	protected Button configureServers;
	protected Button autoGeneratedURL;
	protected String[] serverTypeIds;
	protected Combo serverCombo;
	protected Server server;
	protected boolean serverCanPublish = false;
	protected String contextRoot; //  A context root that will be calculated on every URL edit.

	// list of servers that are in combo
	protected List servers;

	protected boolean init = true;
	// flag to be used to decide whether to enable combo in launch config dialog
	// after the user requests a launch, they cannot change it
	private static final String READ_ONLY = "read-only"; //$NON-NLS-1$

	/**Indicates that the URL field is auto-generated according to the user Server and resource selections.*/
	public static final String AUTO_GENERATED_URL = "auto_generated_url"; //$NON-NLS-1$

	protected WidgetListener fListener = new WidgetListener();
	private boolean saveWorkingCopy;

	private Composite innerPublishContextComposite;

	protected class WidgetListener extends SelectionAdapter implements ModifyListener {
		public void modifyText(ModifyEvent e) {
			updateLaunchConfigurationDialog();
			if (autoGeneratedURL.getSelection()) {
				initializeURLControl();
			}
		}

		public void widgetSelected(SelectionEvent e) {
			setDirty(true);
			Object source = e.getSource();

			if (source == fileButton) {
				handleFileButtonSelected();
			} else if (source == createNewServer) {
				handleServerButtonSelected();
			} else if (source == configureServers) {
				handleConfigureButtonSelected();
			}
		}
	}

	/**
	 * Create a new server launch configuration tab.
	 */
	public ServerTab() {
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#createControl(Composite)
	 */
	public void createControl(Composite parent) {

		ScrolledCompositeImpl scrolledComposite = new ScrolledCompositeImpl(parent, SWT.V_SCROLL);
		Composite composite = new Composite(scrolledComposite, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 1;
		composite.setLayout(layout);

		createServerControl(composite);
		createFileComponent(composite);
		createURLControl(composite);
		createExtensionControls(composite);

		Dialog.applyDialogFont(composite);
		scrolledComposite.setContent(composite);
		Point size = composite.computeSize(SWT.DEFAULT, SWT.DEFAULT);
		scrolledComposite.setMinSize(size.x, size.y);

		setControl(scrolledComposite);
	}

	public void createExtensionControls(Composite composite) {
		return;
	}

	public void createURLControl(Composite composite) {
		Group group = new Group(composite, SWT.NONE);
		String projectLabel = PHPServerUIMessages.getString("ServerTab.url"); //$NON-NLS-1$
		group.setText(projectLabel);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(layout);
		group.setLayoutData(gridData);

		autoGeneratedURL = new Button(group, SWT.CHECK);
		autoGeneratedURL.setText(PHPServerUIMessages.getString("ServerTab.autoGenerate")); //$NON-NLS-1$
		autoGeneratedURL.setSelection(true);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 2;
		autoGeneratedURL.setLayoutData(gridData);
		autoGeneratedURL.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (autoGeneratedURL.getSelection()) {
					initializeURLControl();
				}
				updateLaunchConfigurationDialog();
			}
		});

		fURLLabel = new Label(group, SWT.NONE);
		fURLLabel.setText(PHPServerUIMessages.getString("ServerTab.urlLabel")); //$NON-NLS-1$
		gridData = new GridData();
		gridData.horizontalIndent = 20;
		gridData.horizontalSpan = 1;
		fURLLabel.setLayoutData(gridData);
		
		fURL = new Text(group, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 1;
		fURL.setLayoutData(gridData);
		fURL.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
			}
		});
		/*		
		fContextRootLabel = new Label(group, SWT.NONE);
		fContextRootLabel.setText("Context Root: ");
		gridData = new GridData();
		gridData.horizontalIndent = 20;
		gridData.horizontalSpan = 1;
		fContextRootLabel.setLayoutData(gridData);
		
		fContextRoot = new Text(group, SWT.SINGLE | SWT.BORDER);
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalSpan = 1;
		fContextRoot.setLayoutData(gridData);
		fContextRoot.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
			}
		});
		*/
	}

	protected void createServerSelectionControl(Composite parent) {
		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(data);

		Group group = new Group(composite, SWT.NONE);
		group.setText(PHPServerUIMessages.getString("ServerTab.server")); //$NON-NLS-1$
		GridLayout ly = new GridLayout();
		ly.numColumns = 3;
		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(ly);
		group.setLayoutData(gridData);

		serverCombo = new Combo(group, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
		serverCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		serverCombo.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleServerSelection();
			}
		});

		createNewServer = createPushButton(group, PHPServerUIMessages.getString("ServerTab.new"), null); //$NON-NLS-1$
		createNewServer.addSelectionListener(fListener);

		configureServers = createPushButton(group, PHPServerUIMessages.getString("ServerTab.configure"), null); //$NON-NLS-1$
		configureServers.addSelectionListener(fListener);

		servers = new ArrayList();
		populateServerList((ArrayList) servers);
		// initialize

		if (!servers.isEmpty()) {
			for (int i = 0; i < servers.size(); i++) {
				Server svr = (Server) servers.get(i);
				serverCombo.add(svr.getName());
			}
		}

		// select first item in list
		if (serverCombo.getItemCount() > 0)
			serverCombo.select(0);

		serverCombo.forceFocus();
	}

	protected void populateServerList(ArrayList serverList) {
		Server[] servers = ServersManager.getServers();

		if (serverList == null)
			serverList = new ArrayList();

		if (servers != null) {
			int size = servers.length;
			for (int i = 0; i < size; i++) {
				serverList.add(servers[i]);
			}
		}
	}

	public void createServerControl(Composite parent) {
		createServerSelectionControl(parent);
		GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL);
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(data);
		handleServerSelection();
	}

	protected void createFileComponent(Composite parent) {
		Group group = new Group(parent, SWT.NONE);
		String projectLabel = PHPServerUIMessages.getString("ServerTab.file_project"); //$NON-NLS-1$
		group.setText(projectLabel);
		GridLayout layout = new GridLayout(3, false);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		group.setLayout(layout);
		group.setLayoutData(gd);

		fFile = new Text(group, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		//gridData = new GridData();
		//gridData.widthHint = IDialogConstants.ENTRY_FIELD_WIDTH;
		fFile.setLayoutData(gd);
		fFile.addModifyListener(fListener);

		fileButton = createPushButton(group, PHPServerUIMessages.getString("ServerTab.browse"), null); //$NON-NLS-1$
		gd = (GridData) fileButton.getLayoutData();
		gd.horizontalSpan = 1;
		fileButton.addSelectionListener(fListener);

		publish = createCheckButton(group, PHPServerUIMessages.getString("ServerTab.publish")); //$NON-NLS-1$
		gd = (GridData) publish.getLayoutData();
		gd.horizontalSpan = 3;

		fPublishToLabel = new Label(group, SWT.NONE);
		fPublishToLabel.setText(PHPServerUIMessages.getString("ServerTab.publishLabel")); //$NON-NLS-1$
		gd = new GridData();
		gd.horizontalSpan = 1;
		gd.horizontalIndent = 20;
		fPublishToLabel.setLayoutData(gd);

		
		innerPublishContextComposite = new Composite(group, SWT.NONE);
		layout = new GridLayout(3, false);
		layout.horizontalSpacing = 0;
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		innerPublishContextComposite.setLayout(layout);
		innerPublishContextComposite.setLayoutData(gd);
		fPublishToRoot = new Text(innerPublishContextComposite, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		gd.grabExcessHorizontalSpace = false;
		fPublishToRoot.setLayoutData(gd);
		fPublishToRoot.setEditable(false);
		
		fPublishTo = new Text(innerPublishContextComposite, SWT.SINGLE | SWT.BORDER);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 1;
		fPublishTo.setLayoutData(gd);
		
		fPublishTo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateLaunchConfigurationDialog();
				if (autoGeneratedURL.getSelection()) {
					initializeURLControl();
				}
			}
		});

		publish.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent se) {
				updateLaunchConfigurationDialog();
			}
		});

		handleServerSelection();
	}

	
	public String[] getRequiredNatures() {
		return null;
	}

	protected void handleServerButtonSelected() {
		final Server newServer = getServerFromWizard();
		if (newServer != null) {
			Display.getDefault().asyncExec(new Runnable() {
				public void run() {
					servers.add(newServer);
					serverCombo.add(newServer.getName());
					serverCombo.select(serverCombo.indexOf(newServer.getName()));
					handleServerSelection();
				}
			});
		}
	}

	protected void handleConfigureButtonSelected() {
		int selectionIndex = serverCombo.getSelectionIndex();
		Server server = (Server) servers.get(selectionIndex);
		String serverName = server.getName();
		Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
		NullProgressMonitor monitor = new NullProgressMonitor();
		ServerEditDialog dialog = new ServerEditDialog(shell, server);
		if (dialog.open() == Window.CANCEL) {
			monitor.setCanceled(true);
			return;
		}
		ServersManager.save();
		String newName = server.getName();
		if (!newName.equals(serverName)) {
			serverCombo.remove(selectionIndex);
			serverCombo.add(newName, selectionIndex);
			serverCombo.select(selectionIndex);
		}
		saveWorkingCopy = true;
		handleServerSelection();
	}

	public String[] getFileExtensions() {
		return null;
	}

	private IResource getFileFromDialog(IProject project) {
		return ServerUtilities.getFileFromDialog(project, getShell(), getFileExtensions(), getRequiredNatures());
	}

	protected void handleFileButtonSelected() {

		IResource file = null;
		file = getFileFromDialog(null);

		if (file == null)
			return;

		String fName = file.getFullPath().toString();
		fFile.setText(fName);

		if (autoGeneratedURL.getSelection()) {
			fURL.setText(computeURL(fPublishTo.getText(), fName));
		}
	}

	protected IProject getProject(String name) {
		IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
		IProject[] projects = workspaceRoot.getProjects();

		if (projects == null || projects.length == 0)
			return null;

		for (int i = 0; i < projects.length; i++) {
			IProject project = projects[i];
			if (project.getName().equals(name))
				return project;
		}

		return null;
	}

	/**
	 * Called when a server is selected.
	 * This method should not be called directly.
	 */
	protected void handleServerSelection() {
		int numEntries = serverCombo.getItemCount();
		int index = serverCombo.getSelectionIndex();

		if (!servers.isEmpty()) {
			Object obj = servers.get(serverCombo.getSelectionIndex());
			if (obj != null && obj instanceof Server) {
				server = (Server) servers.get(serverCombo.getSelectionIndex());
				boolean canPublish = false;
				if (server != null) {
					canPublish = server.canPublish();
				}
				serverCanPublish = canPublish;
				if (publish != null) {
					publish.setSelection(canPublish);
					publish.setEnabled(canPublish);
				}
				if (fURL != null) {
					if (autoGeneratedURL.getSelection()) {
						initializeURLControl();
					}
				}
				updatePublishContextRoot();
			}
		}

		if (server == null && (index + 1) != numEntries)
			setErrorMessage(PHPServerUIMessages.getString("ServerTab.noSelectedServerError")); //$NON-NLS-1$
		else {
			setErrorMessage(null);
		}
		updateLaunchConfigurationDialog();
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#setDefaults(ILaunchConfigurationWorkingCopy)
	 */
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		setErrorMessage(null);
		if (serverCombo != null) {
			serverCombo.setEnabled(true);
			if (serverCombo.getItemCount() > 0)
				serverCombo.select(0);
		}
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#initializeFrom(ILaunchConfiguration)
	 */
	public void initializeFrom(ILaunchConfiguration configuration) {
		serverCombo.setEnabled(true);
		//remove error message that other instances may have set
		setErrorMessage(null);

		try {
			String fileName = configuration.getAttribute(Server.FILE_NAME, ""); //$NON-NLS-1$
			String contextRoot = configuration.getAttribute(Server.CONTEXT_ROOT, ""); //$NON-NLS-1$
			String publishTo = configuration.getAttribute(Server.PUBLISH_TO, ""); //$NON-NLS-1$
			boolean deployable = configuration.getAttribute(Server.PUBLISH, false);
			String url = configuration.getAttribute(Server.BASE_URL, ""); //$NON-NLS-1$
			boolean isAutoGeneratedURL = configuration.getAttribute(AUTO_GENERATED_URL, true);

			publish.setSelection(deployable);

			initializeServerControl(configuration);

			fFile.setText(fileName);
			updatePublishContextRoot();
			fPublishTo.setText(publishTo);

			if (isAutoGeneratedURL) {
				autoGeneratedURL.setSelection(true);
				String computedURL = computeURL(contextRoot, fileName);
				fURLLabel.setEnabled(false);
				fURL.setText(computedURL);
				fURL.setEditable(false);
//				fContextRootLabel.setEnabled(false);
//				fContextRoot.setEditable(false);
//				fContextRoot.setText(publishTo);
			} else {
				autoGeneratedURL.setSelection(false);
				fURLLabel.setEnabled(true);
				fURL.setText(url);
				fURL.setEditable(true);
//				fContextRootLabel.setEnabled(true);
//				fContextRoot.setEditable(true);
//				fContextRoot.setText(contextRoot);
			}
		} catch (CoreException e) {
			// ignore
		}
		initializeExtensionControls(configuration);
		isValid(configuration);
	}

	private void updatePublishContextRoot() {
		if (fPublishToRoot == null) {
			return;
		}
		if (server == null || server.getDocumentRoot() == null) {
			fPublishToRoot.setText(""); //$NON-NLS-1$
			return;
		}
		String docRoot = server.getDocumentRoot().trim();
		if (!docRoot.endsWith(File.separator)) {
			docRoot += File.separator;
		}
		fPublishToRoot.setText(docRoot);
		innerPublishContextComposite.layout();
	}

	protected void initializeExtensionControls(ILaunchConfiguration configuration) {
		return;
	}

	/**
	 * Constructs the URL string according to the given context root and the file name.
	 * @param publishTo
	 * @param fileName
	 * @return
	 */
	protected String computeURL(String publishTo, String fileName) {
		if (server == null) {
			return ""; //$NON-NLS-1$
		}
		String urlString = server.getBaseURL();

		if (urlString.equals("")) { //$NON-NLS-1$
			urlString = "http://localhost"; //$NON-NLS-1$
		}
		StringBuffer url = new StringBuffer(urlString);

		if (!publishTo.equals("")) { //$NON-NLS-1$
			publishTo = publishTo.replaceAll("\\\\", "/"); //$NON-NLS-1$ //$NON-NLS-2$
			url.append("/"); //$NON-NLS-1$
			url.append(publishTo);
		}
		if (!fileName.equals("")) { //$NON-NLS-1$
			url.append(formatFileName(fileName));
		}
		return url.toString();
	}

	private String formatFileName(String fileName) {
		String formatFile = null;

		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		IResource resource = root.findMember(fileName);

		if (resource == null) {
			return ""; //$NON-NLS-1$
		}
		int type = resource.getType();
		if (type == IResource.FILE || type == IResource.FOLDER) {
			formatFile = resource.getProjectRelativePath().toString();
		} else if (resource.getType() == IResource.PROJECT) {
			formatFile = "/"; //$NON-NLS-1$
		}
		if (!formatFile.startsWith("/")) { //$NON-NLS-1$
			formatFile = "/" + formatFile; //$NON-NLS-1$
		}
		return formatFile;
	}

	protected void initializeURLControl() {
		if (fPublishTo == null || fFile == null) {
			return;
		}
		fURL.setText(computeURL(fPublishTo.getText(), fFile.getText()));
//		fContextRoot.setText(fPublishTo.getText().replaceAll("\\\\", "/"));
	}

	protected void initializeServerControl(ILaunchConfiguration configuration) {
		try {
			String serverName = configuration.getAttribute(Server.NAME, ""); //$NON-NLS-1$
			if (serverName != null && !serverName.equals("")) { //$NON-NLS-1$
				server = ServersManager.getServer(serverName);

				if (server == null) { //server no longer exists				
					setErrorMessage(PHPServerUIMessages.getString("ServerTab.invalidServerError")); //$NON-NLS-1$
					serverCombo.setEnabled(false);
					return;
				}

				serverCombo.setText(server.getName());
				publish.setEnabled(server.canPublish());
			} else {
				if (serverCombo.getItemCount() > 0) {
					// Select the default server
					String projectName = configuration.getAttribute("org.eclipse.php.debug.core.PHP_Project", (String) null);
					IProject project = null;
					if (projectName != null) {
						project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
					}
					Server defaultServer = ServersManager.getDefaultServer(project);
					int nameIndex = serverCombo.indexOf(defaultServer.getName());
					if (nameIndex > -1) {
						serverCombo.select(nameIndex);
					} else {
						serverCombo.select(0);
					}
					server = ServersManager.getServer(serverCombo.getText());
					publish.setEnabled(server.canPublish());
				}
			}
			//flag should only be set if launch has been attempted on the config
			if (configuration.getAttribute(READ_ONLY, false)) {
				serverCombo.setEnabled(false);
			}
		} catch (Exception e) {
		}
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#performApply(ILaunchConfigurationWorkingCopy)
	 */
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		if (server != null) {
			configuration.setAttribute(Server.NAME, server.getName());
		} else {
			configuration.setAttribute(Server.NAME, (String) null);
		}
		String fileName = fFile.getText();
		String publishTo = fPublishTo.getText();
		String url = fURL.getText();
		boolean deployable = publish.getSelection();
		// compute the context root
		if (autoGeneratedURL.getSelection()) {
			// The context root is the same as the 'publish to' (in case we publish)
			if (deployable) {
				contextRoot = publishTo.replaceAll("\\\\", "/"); //$NON-NLS-1$ //$NON-NLS-2$;
			} else {
				contextRoot = computeContextRoot(url, fileName, server);
			}
		} else {
			contextRoot = computeContextRoot(url, fileName, server);
		}
		configuration.setAttribute(Server.FILE_NAME, fileName);
		configuration.setAttribute(Server.CONTEXT_ROOT, contextRoot);
		configuration.setAttribute(Server.PUBLISH_TO, publishTo);
		configuration.setAttribute(Server.PUBLISH, deployable);
		configuration.setAttribute(Server.BASE_URL, url);
		configuration.setAttribute(AUTO_GENERATED_URL, autoGeneratedURL.getSelection());

		boolean autoGenerateURL = autoGeneratedURL.getSelection();
		fURLLabel.setEnabled(!autoGenerateURL);
		fURL.setEditable(!autoGenerateURL);
//		fContextRootLabel.setEnabled(!autoGenerateURL);
//		fContextRoot.setEditable(!autoGenerateURL);
		
		boolean publishEnabled = publish.getSelection();
		fPublishToLabel.setEnabled(publishEnabled);
		fPublishTo.setEditable(publishEnabled);

		applyExtension(configuration);

		if (saveWorkingCopy) {
			try {
				configuration.doSave();
			} catch (CoreException e) {
			}
			saveWorkingCopy = false;
		}
	}

	private static String computeContextRoot(String url, String fileName, Server server) {
		String serverBaseURL = server.getBaseURL();
		if (url.length() > serverBaseURL.length() + 1) {
			url = url.substring(serverBaseURL.length() + 1);
		} else if (url.length() == serverBaseURL.length() || url.length() == serverBaseURL.length() + 1) {
			return ""; //$NON-NLS-1$
		}
		// Remove the project name from the file name
		if (fileName.length() > 0) {
			fileName = fileName.substring(1);
			int pathIndex = fileName.indexOf('/');
			if (pathIndex < 0) {
				fileName = ""; //$NON-NLS-1$
			} else {
				fileName = fileName.substring(pathIndex);
			}
		}
		if (url.length() > fileName.length()) {
			url = url.substring(0, url.length() - fileName.length());
		} else {
			return ""; //$NON-NLS-1$
		}
		return url;
	}

	protected void applyExtension(ILaunchConfigurationWorkingCopy configuration) {
		return;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#isValid(ILaunchConfiguration) 
	 */
	public boolean isValid(ILaunchConfiguration launchConfig) {
		setMessage(null);
		setErrorMessage(null);
		// TODO - Note that this method of removing invalid launches is still buggy
		// In order to fix it completely, we might have to perform similar checks when removing or renaming a server from
		// the PHP Servers preferences page.
		try {
			String serverName = launchConfig.getAttribute(Server.NAME, ""); //$NON-NLS-1$
			if (!serverName.equals("")) { //$NON-NLS-1$
				Server server = ServersManager.getServer(serverName);
				if (server == null) {
					deleteLaunchConfiguration(launchConfig);
					ILaunchConfiguration[] allConfigurations = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations();
					for (int i = 0; i < allConfigurations.length; i++) {
						launchConfig = allConfigurations[i];
						serverName = launchConfig.getAttribute(Server.NAME, ""); //$NON-NLS-1$
						if (!serverName.equals("") && ServersManager.getServer(serverName) == null) { //$NON-NLS-1$
							// The server was removed, so remove this launch configuration!
							deleteLaunchConfiguration(launchConfig);
						}
					}
					return false;
				}
			}

			String fileName = launchConfig.getAttribute(Server.FILE_NAME, ""); //$NON-NLS-1$
			if (!FileUtils.fileExists(fileName)) {
				setErrorMessage(PHPServerUIMessages.getString("ServerTab.file_project_doesNotExist")); //$NON-NLS-1$
				return false;
			}

		} catch (CoreException e) {
			// ignore
		}

		return isValidExtension(launchConfig);
	}

	private void deleteLaunchConfiguration(final ILaunchConfiguration launchConfig) throws CoreException {
		Display.getDefault().asyncExec(new Runnable() {
			public void run() {
				ILaunchConfiguration config = launchConfig;
				try {
					if (config instanceof ILaunchConfigurationWorkingCopy) {
						config = ((ILaunchConfigurationWorkingCopy) config).getOriginal();
					}
					if (config != null) {
						config.delete();
					}
				} catch (CoreException ce) {
					// Ignore
				}
			}
		});

	}

	protected boolean isValidExtension(ILaunchConfiguration launchConfig) {
		return true;
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getImage()
	 */
	public Image getImage() {
		return null; // TODO - ImageResource.getImage(ImageResource.IMG_SERVER);
	}

	/**
	 * @see org.eclipse.debug.ui.ILaunchConfigurationTab#getName()
	 */
	public String getName() {
		return PHPServerUIMessages.getString("ServerTab.server"); //$NON-NLS-1$
	}

	/*
	 * Fix for Bug 60163 Accessibility: New Builder Dialog missing object info for textInput controls
	 */
	public void addControlAccessibleListener(Control control, String controlName) {
		//strip mnemonic (&)
		String[] strs = controlName.split("&"); //$NON-NLS-1$
		StringBuffer stripped = new StringBuffer();
		for (int i = 0; i < strs.length; i++) {
			stripped.append(strs[i]);
		}
		control.getAccessible().addAccessibleListener(new ControlAccessibleListener(stripped.toString()));
	}

	private class ControlAccessibleListener extends AccessibleAdapter {
		private String controlName;

		ControlAccessibleListener(String name) {
			controlName = name;
		}

		public void getName(AccessibleEvent e) {
			e.result = controlName;
		}

	}

	protected Server getServerFromWizard() {
		Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
		NullProgressMonitor monitor = new NullProgressMonitor();
		Server theServer = null;

		ServerWizard wizard = new ServerWizard();
		ClosableWizardDialog dialog = new ClosableWizardDialog(shell, wizard);
		if (dialog.open() == Window.CANCEL) {
			monitor.setCanceled(true);
			return null;
		}
		theServer = (Server) wizard.getRootFragment().getWizardModel().getObject(WizardModel.SERVER);
		if (theServer != null) {
			ServersManager.addServer(theServer);
			ServersManager.save();
		}
		return theServer;
	}

}
