/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.ui.projectoutlineview;

import java.util.*;

import org.eclipse.dltk.core.IField;
import org.eclipse.dltk.core.IModelElement;
import org.eclipse.dltk.core.IType;
import org.eclipse.dltk.core.ModelException;
import org.eclipse.dltk.core.search.IDLTKSearchScope;
import org.eclipse.dltk.core.search.SearchEngine;
import org.eclipse.php.internal.core.compiler.PHPFlags;
import org.eclipse.php.internal.core.typeinference.PHPModelUtils;
import org.eclipse.php.internal.ui.Logger;
import org.eclipse.php.internal.ui.PHPUIMessages;
import org.eclipse.php.internal.ui.explorer.NamespaceNode;
import org.eclipse.php.internal.ui.util.PHPPluginImages;
import org.eclipse.swt.graphics.Image;

/**
 * Enum class , which holds the "Project Outline" view, first level of nodes,
 * and provides methods for getting relevant children (using 'mixin' model)
 * 
 * @author nir.c
 */
public enum ProjectOutlineGroups {

	GROUP_CLASSES(PHPPluginImages.DESC_OBJ_PHP_CLASSES_GROUP.createImage(), PHPUIMessages.getString("PHPProjectOutline.nodes.classes")),
	GROUP_NAMESPACES(PHPPluginImages.DESC_OBJ_PHP_NAMESPACES_GROUP.createImage(), PHPUIMessages.getString("PHPProjectOutline.nodes.namespaces")),
	GROUP_CONSTANTS(PHPPluginImages.DESC_OBJ_PHP_CONSTANTS_GROUP.createImage(), PHPUIMessages.getString("PHPProjectOutline.nodes.constants")),
	GROUP_FUNCTIONS(PHPPluginImages.DESC_OBJ_PHP_FUNCTIONS_GROUP.createImage(), PHPUIMessages.getString("PHPProjectOutline.nodes.functions"));

	private final Image image;
	private final String text;

	protected static final Object[] NO_CHILDREN = new Object[0];

	ProjectOutlineGroups(Image image, String text) {
		this.image = image;
		this.text = text;
	}

	public Image getImage() {
		return image;
	}

	public String getText() {
		return text;
	}

	protected Object[] getChildren() {
		if (ProjectOutlineContentProvider.scripProject != null) {

			IDLTKSearchScope scope = SearchEngine.createSearchScope(ProjectOutlineContentProvider.scripProject, IDLTKSearchScope.SOURCES);

			TreeSet<IModelElement> childrenList = new TreeSet<IModelElement>(new Comparator<IModelElement>() {
				public int compare(IModelElement o1, IModelElement o2) {
					int res = o1.getElementName().compareTo(o2.getElementName());
					if (res == 0) {
						return (o1.getPath().toOSString() + o1.getElementName()).compareTo(o2.getPath().toOSString() + o2.getElementName());
					}
					return res;

				}
			});
			switch (this) {
				case GROUP_NAMESPACES:
					IType[] allNamespaces = PHPModelUtils.getAllNamespaces(scope);
					Map<String, List<IType>> nsByName = new HashMap<String, List<IType>>();
					for (IType namespace : allNamespaces) {
						String namespaceName = namespace.getElementName();
						List<IType> nsList = nsByName.get(namespaceName);
						if (nsList == null) {
							nsList = new LinkedList<IType>();
							nsByName.put(namespaceName, nsList);
						}
						nsList.add(namespace);
					}
					for (String namespaceName : nsByName.keySet()) {
						List<IType> nsList = nsByName.get(namespaceName);
						childrenList.add(new NamespaceNode(ProjectOutlineContentProvider.scripProject, 
							namespaceName, nsList.toArray(new IType[nsList.size()])));
					}
					break;

				case GROUP_CLASSES:
					childrenList.addAll(Arrays.asList(PHPModelUtils.getAllClassesAndInterfaces(scope)));
					break;

				case GROUP_FUNCTIONS:
					childrenList.addAll(Arrays.asList(PHPModelUtils.getAllFunctions(scope)));
					break;

				case GROUP_CONSTANTS:
					IField[] fields = PHPModelUtils.getAllFields(scope);
					for (IField iField : fields) {
						try {
							if (PHPFlags.isConstant(iField.getFlags())) {
								childrenList.add(iField);
							}
						} catch (ModelException e) {
							Logger.logException(e);
						}
					}
					break;
			}
			return childrenList.toArray();
		}

		return NO_CHILDREN;
	}

	public boolean hasChildren() {
		Object[] children = getChildren();
		if (null == children || children.length == 0) {
			return false;
		}
		return true;
	}

}