/*******************************************************************************
 * Copyright (c) 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Zend Technologies
 *******************************************************************************/
package org.eclipse.php.internal.debug.ui.preferences;

import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ProjectScope;
import org.eclipse.core.runtime.Preferences;
import org.eclipse.core.runtime.preferences.IEclipsePreferences;
import org.eclipse.core.runtime.preferences.IScopeContext;
import org.eclipse.jface.dialogs.IPageChangedListener;
import org.eclipse.jface.dialogs.PageChangedEvent;
import org.eclipse.jface.preference.PreferenceDialog;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.php.internal.debug.core.PHPDebugPlugin;
import org.eclipse.php.internal.debug.core.preferences.*;
import org.eclipse.php.internal.debug.core.zend.communication.DebuggerCommunicationDaemon;
import org.eclipse.php.internal.debug.ui.Logger;
import org.eclipse.php.internal.debug.ui.PHPDebugUIMessages;
import org.eclipse.php.internal.server.core.Server;
import org.eclipse.php.internal.server.core.manager.ServersManager;
import org.eclipse.php.internal.ui.preferences.AbstractPHPPreferencePageBlock;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.dialogs.PreferencesUtil;
import org.eclipse.ui.ide.IDEEncoding;
import org.osgi.service.prefs.BackingStoreException;

/**
 * PHP debug options preferences add-on. This add-on specifies the default
 * debugger, executable and server for the workspace or the project specific.
 * 
 * @author Shalom Gibly
 */
public class PHPDebugPreferencesBlock extends AbstractPHPPreferencePageBlock {

	private static final String UNRESOLVED_PHP_VERSION = "Unresolved PHP Version"; //$NON-NLS-1$
	private static final String DEBUGGERS_PAGE_ID = "org.eclipse.php.debug.ui.installedDebuggersPage"; //$NON-NLS-1$
	private static final String SERVERS_PAGE_ID = "org.eclipse.php.server.internal.ui.PHPServersPreferencePage"; //$NON-NLS-1$
	private static final String PHP_EXE_PAGE_ID = "org.eclipse.php.debug.ui.preferencesphps.PHPsPreferencePage"; //$NON-NLS-1$

	private Button fStopAtFirstLine;
	private Combo fDefaultDebugger;
	private Combo fDefaultServer;
	private Combo fDefaultPHPExe;
	private Collection<String> debuggersIds;
	private Combo fDebugEncodingSettings;
	private Combo fOutputEncodingSettings;
	private PreferencePage propertyPage;

	public void setCompositeAddon(Composite parent) {
		Composite composite = addPageContents(parent);
		addProjectPreferenceSubsection(composite);
	}

	public void initializeValues(PreferencePage propertyPage) {
		this.propertyPage = propertyPage;
		Preferences prefs = PHPProjectPreferences.getModelPreferences();
		IScopeContext[] preferenceScopes = createPreferenceScopes(propertyPage);

		boolean stopAtFirstLine = prefs
				.getBoolean(PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE);
		String debuggerName = PHPDebuggersRegistry.getDebuggerName(prefs
				.getString(PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID));
		String serverName = ServersManager.getDefaultServer(null).getName();
		PHPexes exes = PHPexes.getInstance();
		String phpExeName = PHPDebugUIMessages.PhpDebugPreferencePage_noExeDefined;
		if (exes.hasItems(PHPDebugPlugin.getCurrentDebuggerId())) {
			String name = exes.getDefaultItem(
					PHPDebugPlugin.getCurrentDebuggerId()).getName();
			if (null != name)
				phpExeName = name;
		}
		String transferEncoding = prefs
				.getString(PHPDebugCorePreferenceNames.TRANSFER_ENCODING);
		String outputEncoding = prefs
				.getString(PHPDebugCorePreferenceNames.OUTPUT_ENCODING);
		loadDebuggers(fDefaultDebugger);
		loadServers(fDefaultServer);
		boolean exeLoaded = false;
		// Update the values in case we have a project-specific settings.
		if (preferenceScopes[0] instanceof ProjectScope) {
			IEclipsePreferences node = preferenceScopes[0]
					.getNode(getPreferenceNodeQualifier());
			if (node != null && getProject(propertyPage) != null) {
				String projectServerName = ServersManager.getDefaultServer(
						getProject(propertyPage)).getName();
				if (!projectServerName.equals("")) { //$NON-NLS-1$
					String debuggerId = node.get(
							PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID,
							PHPDebugPlugin.getCurrentDebuggerId());
					debuggerName = PHPDebuggersRegistry
							.getDebuggerName(debuggerId);
					serverName = projectServerName;
					stopAtFirstLine = node.getBoolean(
							PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE,
							stopAtFirstLine);
					transferEncoding = node.get(
							PHPDebugCorePreferenceNames.TRANSFER_ENCODING,
							transferEncoding); //$NON-NLS-1$
					outputEncoding = node.get(
							PHPDebugCorePreferenceNames.OUTPUT_ENCODING,
							outputEncoding); //$NON-NLS-1$
					phpExeName = node
							.get(PHPDebugCorePreferenceNames.DEFAULT_PHP,
									phpExeName);
					// Check that if the project had a non-defined exe, and now
					// there is one that is valid. we set
					// it with the new valid default exe.
					if (PHPDebugUIMessages.PhpDebugPreferencePage_noExeDefined
							.equals(phpExeName)) {
						if (exes.hasItems(debuggerId)) {
							phpExeName = exes.getDefaultItem(debuggerId)
									.getName();
							node.put(PHPDebugCorePreferenceNames.DEFAULT_PHP,
									phpExeName);
							try {
								node.flush();
							} catch (BackingStoreException e) {
							}
						}
					}
					loadPHPExes(fDefaultPHPExe, exes.getItems(node.get(
							PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID,
							PHPDebugPlugin.getCurrentDebuggerId())));
					exeLoaded = true;
				}
			}
		}
		if (!exeLoaded) {
			loadPHPExes(fDefaultPHPExe, exes.getItems(PHPDebugPlugin
					.getCurrentDebuggerId()));
		}
		fStopAtFirstLine.setSelection(stopAtFirstLine);
		fDefaultDebugger.select(fDefaultDebugger.indexOf(debuggerName));
		fDefaultServer.select(fDefaultServer.indexOf(serverName));
		fDefaultPHPExe.select(fDefaultPHPExe.indexOf(phpExeName));
		fDebugEncodingSettings.setText(transferEncoding);
		fOutputEncodingSettings.setText(outputEncoding);
	}

	public boolean performOK(boolean isProjectSpecific) {
		savePreferences(isProjectSpecific);
		return true;
	}

	public void performApply(boolean isProjectSpecific) {
		performOK(isProjectSpecific);
	}

	public boolean performCancel() {
		return true;
	}

	public void performDefaults() {
		Preferences prefs = PHPProjectPreferences.getModelPreferences();
		fStopAtFirstLine
				.setSelection(prefs
						.getDefaultBoolean(PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE));
		loadDebuggers(fDefaultDebugger);
		loadServers(fDefaultServer);
		loadPHPExes(fDefaultPHPExe, PHPexes.getInstance().getItems(
				PHPDebugPlugin.getCurrentDebuggerId()));
		fDebugEncodingSettings
				.setText(prefs
						.getDefaultString(PHPDebugCorePreferenceNames.TRANSFER_ENCODING));
		fOutputEncodingSettings.setText(prefs
				.getDefaultString(PHPDebugCorePreferenceNames.OUTPUT_ENCODING));
	}

	protected String getPreferenceNodeQualifier() {
		return PHPProjectPreferences.getPreferenceNodeQualifier();
	}

	private void addProjectPreferenceSubsection(Composite composite) {

		Group defultPrefsGroup = new Group(composite, SWT.NONE);
		defultPrefsGroup.setText(PHPDebugUIMessages.PHPDebugPreferencesBlock_0);
		defultPrefsGroup.setFont(composite.getFont());
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		layout.verticalSpacing = 10;
		defultPrefsGroup.setLayout(layout);
		defultPrefsGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		addLabelControl(defultPrefsGroup,
				PHPDebugUIMessages.PhpDebugPreferencePage_phpDebugger,
				PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID);
		fDefaultDebugger = addCombo(defultPrefsGroup, 2);
		addLink(defultPrefsGroup,
				PHPDebugUIMessages.PHPDebugPreferencesBlock_1,
				DEBUGGERS_PAGE_ID);

		addLabelControl(defultPrefsGroup,
				PHPDebugUIMessages.PhpDebugPreferencePage_9,
				ServersManager.DEFAULT_SERVER_PREFERENCES_KEY);
		fDefaultServer = addCombo(defultPrefsGroup, 2);
		addLink(defultPrefsGroup,
				PHPDebugUIMessages.PhpDebugPreferencePage_serversLink,
				SERVERS_PAGE_ID);

		addLabelControl(defultPrefsGroup,
				PHPDebugUIMessages.PhpDebugPreferencePage_12,
				PHPDebugCorePreferenceNames.DEFAULT_PHP);
		fDefaultPHPExe = addCombo(defultPrefsGroup, 2);
		addLink(defultPrefsGroup,
				PHPDebugUIMessages.PhpDebugPreferencePage_installedPHPsLink,
				PHP_EXE_PAGE_ID);

		new Label(composite, SWT.NONE); // dummy label

		Group encodingGroup = new Group(composite, SWT.NONE);
		encodingGroup.setText(PHPDebugUIMessages.PHPDebugPreferencesBlock_2);
		encodingGroup.setLayout(new GridLayout(2, false));
		encodingGroup.setLayoutData(new GridData(GridData.FILL_BOTH));

		addLabelControl(
				encodingGroup,
				PHPDebugUIMessages.PHPDebugPreferencesAddon_debugTransferEncoding,
				PHPDebugCorePreferenceNames.TRANSFER_ENCODING);
		fDebugEncodingSettings = addEncodingSettings(encodingGroup);

		addLabelControl(
				encodingGroup,
				PHPDebugUIMessages.PHPDebugPreferencesAddon_debugOutputEncoding,
				PHPDebugCorePreferenceNames.OUTPUT_ENCODING);
		fOutputEncodingSettings = addEncodingSettings(encodingGroup);

		ModifyListener modifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				boolean isValid = isValidEncoding(((Combo) e.getSource())
						.getText());
				if (isValid) {
					propertyPage.setErrorMessage(null);
				} else {
					propertyPage
							.setErrorMessage(PHPDebugUIMessages.PHPDebugPreferencesAddon_unsupportedEncoding);
				}
				propertyPage.setValid(isValid);
			}
		};
		fDebugEncodingSettings.addModifyListener(modifyListener);
		fOutputEncodingSettings.addModifyListener(modifyListener);

		new Label(composite, SWT.NONE); // dummy label

		fStopAtFirstLine = addCheckBox(composite,
				PHPDebugUIMessages.PhpDebugPreferencePage_1,
				PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE, 0);

		// Add a default debugger listener that will update the possible
		// executables
		// and, maybe, servers that can work with this debugger.
		fDefaultDebugger.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String selectedDebugger = getSelectedDebuggerId();
				PHPexeItem[] items = PHPexes.getInstance().getItems(
						selectedDebugger);
				loadPHPExes(fDefaultPHPExe, items);
			}
		});
	}

	private void loadPHPExes(Combo combo, PHPexeItem[] items) {
		combo.removeAll();
		if (items == null || items.length == 0) {
			combo.add(PHPDebugUIMessages.PhpDebugPreferencePage_noExeDefined);
			combo.select(0);
			return;
		}
		for (PHPexeItem element : items) {
			String name = element.getName();
			if (null == name)
				name = UNRESOLVED_PHP_VERSION;
			combo.add(name);
		}
		// select the default item for the current selected debugger
		if (fDefaultDebugger.getItemCount() > 0) {
			PHPexeItem defaultItem = PHPexes.getInstance().getDefaultItem(
					getSelectedDebuggerId());
			String defaultItemName;
			if (defaultItem != null) {
				defaultItemName = defaultItem.getName();
				if (null == defaultItemName)
					defaultItemName = UNRESOLVED_PHP_VERSION;
			} else {
				defaultItemName = PHPDebugUIMessages.PHPDebuggersTable_notDefined;
				if (combo.indexOf(defaultItemName) == -1) {
					combo.add(defaultItemName, 0);
					combo.select(0);
				}
			}
			int index = combo.indexOf(defaultItemName);
			if (index > -1) {
				combo.select(index);
			} else if (combo.getItemCount() > 0) {
				// select first item in list
				combo.select(0);
			}
		}

	}

	private void loadServers(Combo combo) {
		combo.removeAll();
		Server[] servers = ServersManager.getServers();
		if (servers != null) {
			for (Server element : servers) {
				combo.add(element.getName());
			}
			// select first item in list
			if (combo.getItemCount() > 0) {
				combo.select(0);
			}
		}
	}

	private void loadDebuggers(Combo combo) {
		debuggersIds = PHPDebuggersRegistry.getDebuggersIds();
		String defaultDebuggerID = DebuggerCommunicationDaemon.ZEND_DEBUGGER_ID;
		combo.removeAll();
		Iterator<String> debuggers = debuggersIds.iterator();
		int defaultIndex = 0;
		int index = 0;
		while (debuggers.hasNext()) {
			String id = debuggers.next();
			if (defaultDebuggerID.equals(id)) {
				defaultIndex = index;
			} else {
				index++;
			}
			String debuggerName = PHPDebuggersRegistry.getDebuggerName(id);
			combo.add(debuggerName);
		}
		// select the default item in list
		if (combo.getItemCount() > 0) {
			combo.select(defaultIndex);
		}
	}

	private void addLink(Composite parent, String label,
			final String propertyPageID) {
		Link link = new Link(parent, SWT.NONE);
		link.setFont(parent.getFont());
		link.setLayoutData(new GridData(SWT.END, SWT.BEGINNING, true, false));
		link.setText(label);
		link.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				PreferenceDialog dialog = PreferencesUtil
						.createPreferenceDialogOn(
								PHPDebugPreferencesBlock.this.propertyPage
										.getShell(), propertyPageID, null, null);
				dialog.setBlockOnOpen(true);
				dialog.addPageChangedListener(new IPageChangedListener() {
					public void pageChanged(PageChangedEvent event) {
						Display.getDefault().asyncExec(new Runnable() {
							public void run() {
								String selectedDebugger = fDefaultDebugger
										.getText();
								String selectedServer = fDefaultServer
										.getText();
								String selectedPHP = fDefaultPHPExe.getText();
								loadDebuggers(fDefaultDebugger);
								loadServers(fDefaultServer);
								loadPHPExes(fDefaultPHPExe, PHPexes
										.getInstance().getItems(
												getSelectedDebuggerId()));
								selectComboItem(fDefaultDebugger,
										fDefaultDebugger
												.indexOf(selectedDebugger));
								selectComboItem(fDefaultServer, fDefaultServer
										.indexOf(selectedServer));
								selectComboItem(fDefaultPHPExe, fDefaultPHPExe
										.indexOf(selectedPHP));
							}
						});
					}
				});
				dialog.open();
			}
		});
	}

	private void selectComboItem(Combo combo, int itemIndex) {
		if (itemIndex < 0) {
			if (combo.getItemCount() > 0) {
				combo.select(0);
			}
		} else {
			combo.select(itemIndex);
		}
	}

	private Combo addEncodingSettings(Composite parent) {
		Combo encodingCombo = new Combo(parent, SWT.NONE);
		GridData data = new GridData();
		encodingCombo.setLayoutData(data);

		List encodings = IDEEncoding.getIDEEncodings();
		String[] encodingStrings = new String[encodings.size()];
		encodings.toArray(encodingStrings);
		encodingCombo.setItems(encodingStrings);

		return encodingCombo;
	}

	private Combo addCombo(Composite parent, int horizontalIndent) {
		Combo combo = new Combo(parent, SWT.SINGLE | SWT.BORDER | SWT.READ_ONLY);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalIndent = horizontalIndent;
		combo.setLayoutData(gd);
		return combo;
	}

	private void savePreferences(boolean isProjectSpecific) {
		String phpExe = fDefaultPHPExe.getText();
		// Check if it's still valid, since this method can be called after
		// removing a php executable from another preferences page (PHP
		// Executables)
		PHPexes exes = PHPexes.getInstance();
		if (exes.getItem(getSelectedDebuggerId(), phpExe) == null) {
			PHPexeItem item = exes.getDefaultItem(getSelectedDebuggerId());
			if (item != null) {
				phpExe = item.getName();
			} else {
				phpExe = ""; //$NON-NLS-1$
			}
		}
		// TODO - Might do the same for the default server
		Preferences prefs = PHPProjectPreferences.getModelPreferences();
		IScopeContext[] preferenceScopes = createPreferenceScopes(propertyPage);
		IEclipsePreferences debugUINode = preferenceScopes[0]
				.getNode(getPreferenceNodeQualifier());
		IProject project = getProject(propertyPage);
		if (isProjectSpecific && debugUINode != null
				&& preferenceScopes[0] instanceof ProjectScope
				&& project != null) {
			debugUINode.putBoolean(
					PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE,
					fStopAtFirstLine.getSelection());
			debugUINode.put(PHPDebugCorePreferenceNames.DEFAULT_PHP, phpExe);
			debugUINode.put(PHPDebugCorePreferenceNames.TRANSFER_ENCODING,
					fDebugEncodingSettings.getText());
			debugUINode.put(PHPDebugCorePreferenceNames.OUTPUT_ENCODING,
					fOutputEncodingSettings.getText());
			debugUINode.put(PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID,
					getSelectedDebuggerId());
			ServersManager.setDefaultServer(project, fDefaultServer.getText());
		} else {
			if (project == null) {
				// Workspace settings
				prefs.setValue(PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE,
						fStopAtFirstLine.getSelection());
				prefs.setValue(PHPDebugCorePreferenceNames.TRANSFER_ENCODING,
						fDebugEncodingSettings.getText());
				prefs.setValue(PHPDebugCorePreferenceNames.OUTPUT_ENCODING,
						fOutputEncodingSettings.getText());
				prefs.setValue(PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID,
						getSelectedDebuggerId());
				exes.setDefaultItem(getSelectedDebuggerId(), phpExe);
				ServersManager.setDefaultServer(null, fDefaultServer.getText());
			} else {
				if (debugUINode != null) {
					// Removed a project specific
					debugUINode
							.remove(PHPDebugCorePreferenceNames.STOP_AT_FIRST_LINE);
					// debugUINode.remove(PHPDebugCorePreferenceNames.ZEND_DEBUG_PORT);
					// // No need
					debugUINode.remove(PHPDebugCorePreferenceNames.DEFAULT_PHP);
					ServersManager.setDefaultServer(project, (Server) null);
					debugUINode
							.remove(PHPDebugCorePreferenceNames.TRANSFER_ENCODING);
					debugUINode
							.remove(PHPDebugCorePreferenceNames.OUTPUT_ENCODING);
					debugUINode
							.remove(PHPDebugCorePreferenceNames.PHP_DEBUGGER_ID);
				}
			}
		}
		try {
			debugUINode.flush();
			exes.save();
			PHPDebugPlugin.getDefault().savePluginPreferences();
		} catch (BackingStoreException e) {
			Logger.logException(e);
		}
	}

	private String getSelectedDebuggerId() {
		int selectedIndex = fDefaultDebugger.getSelectionIndex();
		String debuggerId = DebuggerCommunicationDaemon.ZEND_DEBUGGER_ID; // default
		if (selectedIndex > -1 && debuggersIds.size() > selectedIndex) {
			debuggerId = debuggersIds.toArray()[selectedIndex].toString();
		}
		return debuggerId;
	}

	/**
	 * Returns whether or not the given encoding is valid.
	 * 
	 * @param enc
	 *            the encoding to validate
	 * @return <code>true</code> if the encoding is valid, <code>false</code>
	 *         otherwise
	 * @see 
	 *      org.eclipse.ui.ide.dialogs.AbstractEncodingFieldEditor.isValidEncoding
	 *      (String)
	 */
	private boolean isValidEncoding(String enc) {
		try {
			return Charset.isSupported(enc);
		} catch (IllegalCharsetNameException e) {
			// This is a valid exception
			return false;
		}
	}
}
